/* ************************************************************************
*   File: structs.h                                     Part of CircleMUD *
*  Usage: header file for central structures and contstants               *
*                                                                         *
*  All rights reserved.  See license.doc for complete information.        *
*                                                                         *
*  Copyright (C) 1993, 94 by the Trustees of the Johns Hopkins University *
*  CircleMUD is based on DikuMUD, Copyright (C) 1990, 1991.               *
************************************************************************ */


/* preamble *************************************************************/


#include <sys/types.h>

#define BIT(name)  unsigned int name: 1
#define NOWHERE    -1    /* nil reference for room-database	*/
#define NOTHING    -1    /* nil reference for objects		*/
#define NOBODY     -1    /* nil reference for mobiles		*/

#define SPECIAL(name) \
   int (name)(struct char_data *ch, void *me, int cmd, char *argument)


/* room-related defines *************************************************/


/* The cardinal directions: used as index to room_data.dir_option[] */
enum direction {
  NORTH,
  EAST,
  SOUTH,
  WEST,
  UP,
  DOWN
};


/* Room flags: used in room_data.room_flags */
/* WARNING: In the world files, NEVER set the bits marked "R" ("Reserved") */
struct roomflags {
  BIT(ROOM_DARK);		/* Dark				*/
  BIT(ROOM_DEATH);		/* Death trap			*/
  BIT(ROOM_NOMOB);		/* MOBs not allowed		*/
  BIT(ROOM_INDOORS);		/* Indoors			*/
  BIT(ROOM_PEACEFUL);		/* Violence not allowed		*/
  BIT(ROOM_SOUNDPROOF);		/* Shouts, gossip blocked	*/
  BIT(ROOM_NOTRACK);		/* Track won't go through	*/
  BIT(ROOM_NOMAGIC);		/* Magic not allowed		*/
  BIT(ROOM_TUNNEL);		/* room for only 1 pers		*/
  BIT(ROOM_PRIVATE);		/* Can't teleport in		*/
  BIT(ROOM_GODROOM);		/* LVL_GOD+ only allowed	*/
  BIT(ROOM_HOUSE);		/* (R) Room is a house		*/
  BIT(ROOM_HOUSE_CRASH);	/* (R) House needs saving	*/
  BIT(ROOM_ATRIUM);		/* (R) The door to a house	*/
  BIT(ROOM_OLC);		/* (R) Modifyable/!compress	*/
  BIT(ROOM_BFS_MARK);		/* (R) breath-first srch mrk	*/
};


/* Exit info: used in room_data.dir_option.exit */
struct exit_info {
  BIT(EX_ISDOOR);		/* Exit is a door		*/
  BIT(EX_CLOSED);		/* The door is closed		*/
  BIT(EX_LOCKED);		/* The door is locked		*/
  BIT(EX_PICKPROOF);		/* Lock can't be picked		*/
};


/* Sector types: used in room_data.sector_type */
enum sector {
SECT_INSIDE,			/* Indoors			*/
SECT_CITY,			/* In a city			*/
SECT_FIELD,			/* In a field			*/
SECT_FOREST,			/* In a forest			*/
SECT_HILLS,			/* In the hills			*/
SECT_MOUNTAIN,			/* On a mountain		*/
SECT_WATER_SWIM,		/* Swimmable water		*/
SECT_WATER_NOSWIM,		/* Water - need a boat		*/
SECT_UNDERWATER,		/* Underwater			*/
SECT_FLYING			/* Wheee!			*/
};


/* char and mob-related defines *****************************************/

#define SPELLO_SPELL 0
#define SPELLO_SKILL 1

/* PC classes */
#define CLASS_UNDEFINED	  -1
#define NUM_CLASSES	  5  /* This must be the number of classes!! */
enum class {
  CLASS_MAGIC_USER,
  CLASS_CLERIC,
  CLASS_THIEF,
  CLASS_WARRIOR,
  CLASS_PSI,
  /* add new classes above this */
  Class_pc_end = NUM_CLASSES
};

/* NPC classes (currently unused - feel free to implement!) */
#define CLASS_OTHER       0
#define CLASS_UNDEAD      1
#define CLASS_HUMANOID    2
#define CLASS_ANIMAL      3
#define CLASS_DRAGON      4
#define CLASS_GIANT       5


/* Sex */
enum sex {
  SEX_NEUTRAL,
  SEX_MALE,
  SEX_FEMALE
};


/* Positions */
enum position {
  POS_DEAD,			/* dead				*/
  POS_MORTALLYW,		/* mortally wounded		*/
  POS_INCAP,			/* incapacitated		*/
  POS_STUNNED,			/* stunned			*/
  POS_SLEEPING,			/* sleeping			*/
  POS_RESTING,			/* resting			*/
  POS_SITTING,			/* sitting			*/
  POS_FIGHTING,			/* fighting			*/
  POS_STANDING			/* standing			*/
};


/* Player flags: used by char_data.char_specials.act */
struct playerflags {
  BIT(PLR_KILLER);		/* Player is a player-killer		*/
  BIT(PLR_THIEF);		/* Player is a player-thief		*/
  BIT(PLR_FROZEN);		/* Player is frozen			*/
  BIT(PLR_DONTSET);		/* Don't EVER set (ISNPC bit)		*/
  BIT(PLR_WRITING);		/* Player writing /olc)			*/
  BIT(PLR_MAILING);		/* Player is writing mail		*/
  BIT(PLR_CRASH);		/* Player needs to be crash-saved	*/
  BIT(PLR_SITEOK);		/* Player has been site-cleared		*/
  BIT(PLR_NOSHOUT);		/* Player not allowed to shout/goss	*/
  BIT(PLR_NOTITLE);		/* Player not allowed to set title	*/
  BIT(PLR_DELETED);		/* Player deleted - space reusable	*/
  BIT(PLR_LOADROOM);		/* Player uses nonstandard loadroom	*/
  BIT(PLR_NOWIZLIST);		/* Player shouldn't be on wizlist	*/
  BIT(PLR_NODELETE);		/* Player shouldn't be deleted		*/
  BIT(PLR_INVSTART);		/* Player should enter game wizinvis	*/
  BIT(PLR_CRYO);		/* Player is cryo-saved (purge prog)	*/
};


/* Mobile flags: used by char_data.char_specials.act */
struct mobflags {
  BIT(MOB_SPEC);		/* Mob has a callable spec-proc		*/
  BIT(MOB_SENTINEL);		/* Mob should not move			*/
  BIT(MOB_SCAVENGER);		/* Mob picks up stuff on the ground	*/
  BIT(MOB_ISNPC);		/* (R) Automatically set on all Mobs	*/
  BIT(MOB_AWARE);		/* Mob can't be backstabbed		*/
  BIT(MOB_AGGRESSIVE);		/* Mob hits players in the room		*/
  BIT(MOB_STAY_ZONE);		/* Mob shouldn't wander out of zone	*/
  BIT(MOB_WIMPY);		/* Mob flees if severely injured	*/
  BIT(MOB_AGGR_EVIL);		/* auto attack evil PC's		*/
  BIT(MOB_AGGR_GOOD);		/* auto attack good PC's		*/
  BIT(MOB_AGGR_NEUTRAL);	/* auto attack neutral PC's		*/
  BIT(MOB_MEMORY);		/* remember attackers if attacked	*/
  BIT(MOB_HELPER);		/* attack PCs fighting other NPCs	*/
  BIT(MOB_NOCHARM);		/* Mob can't be charmed			*/
  BIT(MOB_NOSUMMON);		/* Mob can't be summoned		*/
  BIT(MOB_NOSLEEP);		/* Mob can't be slept			*/
  BIT(MOB_NOBASH);		/* Mob can't be bashed (e.g. trees)	*/
  BIT(MOB_NOBLIND);		/* Mob can't be blinded			*/
};


/* Preference flags: used by char_data.player_specials.pref */
struct preferances {
  BIT(PRF_BRIEF);		/* Room descs won't normally be shown	*/
  BIT(PRF_COMPACT);		/* No extra CRLF pair before prompts	*/
  BIT(PRF_DEAF);		/* Can't hear shouts			*/
  BIT(PRF_NOTELL);		/* Can't receive tells			*/
  BIT(PRF_DISPHP);		/* Display hit points in prompt		*/
  BIT(PRF_DISPMANA);		/* Display mana points in prompt	*/
  BIT(PRF_DISPMOVE);		/* Display move points in prompt	*/
  BIT(PRF_AUTOEXIT);		/* Display exits in a room		*/
  BIT(PRF_NOHASSLE);		/* Aggr mobs won't attack		*/
  BIT(PRF_QUEST);		/* On quest				*/
  BIT(PRF_SUMMONABLE);		/* Can be summoned			*/
  BIT(PRF_NOREPEAT);		/* No repetition of comm commands	*/
  BIT(PRF_HOLYLIGHT);		/* Can see in dark			*/
  BIT(PRF_COLOR_1);		/* Color (low bit)			*/
  BIT(PRF_COLOR_2);		/* Color (high bit)			*/
  BIT(PRF_NOWIZ);		/* Can't hear wizline			*/
  BIT(PRF_LOG1);		/* On-line System Log (low bit)		*/
  BIT(PRF_LOG2);		/* On-line System Log (high bit)	*/
  BIT(PRF_NOAUCT);		/* Can't hear auction channel		*/
  BIT(PRF_NOGOSS);		/* Can't hear gossip channel		*/
  BIT(PRF_NOGRATZ);		/* Can't hear grats channel		*/
  BIT(PRF_ROOMFLAGS);		/* Can see room flags (ROOM_x)		*/
};


/* Affect bits: used in char_data.char_specials.saved.affected_by */
/* WARNING: In the world files, NEVER set the bits marked "R" ("Reserved") */
struct affections {
  BIT(AFF_BLIND);		/* (R) Char is blind		*/
  BIT(AFF_INVISIBLE);		/* Char is invisible		*/
  BIT(AFF_DETECT_ALIGN);	/* Char is sensitive to align	*/
  BIT(AFF_DETECT_INVIS);	/* Char can see invis chars  	*/
  BIT(AFF_DETECT_MAGIC);	/* Char is sensitive to magic	*/
  BIT(AFF_SENSE_LIFE);		/* Char can sense hidden life	*/
  BIT(AFF_WATERWALK);		/* Char can walk on water	*/
  BIT(AFF_SANCTUARY);		/* Char protected by sanct.	*/
  BIT(AFF_GROUP);		/* (R) Char is grouped		*/
  BIT(AFF_CURSE);		/* Char is cursed		*/
  BIT(AFF_INFRAVISION);		/* Char can see in dark		*/
  BIT(AFF_POISON);		/* (R) Char is poisoned		*/
  BIT(AFF_PROTECT_EVIL);	/* Char protected from evil  	*/
  BIT(AFF_PROTECT_GOOD);	/* Char protected from good  	*/
  BIT(AFF_SLEEP);		/* (R) Char magically asleep	*/
  BIT(AFF_NOTRACK);		/* Char can't be tracked	*/
  BIT(AFF_DETECT_EVIL);		/* Room for future expansion	*/
  BIT(AFF_DETECT_GOOD);		/* Room for future expansion	*/
  BIT(AFF_SNEAK);		/* Char can move quietly	*/
  BIT(AFF_HIDE);		/* Char is hidden		*/
  BIT(AFF_FLY);  		/* Room for future expansion	*/
  BIT(AFF_CHARM);		/* Char is charmed		*/
  BIT(AFF_ENFEEBLEMENT);
  BIT(AFF_FAERIE_FIRE);
  BIT(AFF_RESIST_ELECTRICITY);
  BIT(AFF_RESIST_FIRE);
  BIT(AFF_RESIST_COLD);
  BIT(AFF_DETECT_CURSE);
  BIT(AFF_HASTE);
  BIT(AFF_SLOW);
};


/* Modes of connectedness: used by descriptor_data.state */
enum connectedness {
  CON_PLAYING,	 		/* Playing - Nominal state	*/
  CON_CLOSE,	 		/* Disconnecting		*/
  CON_GET_NAME,	 		/* By what name ..?		*/
  CON_NAME_CNFRM,	 	/* Did I get that right, x?	*/
  CON_PASSWORD,	 		/* Password:			*/
  CON_NEWPASSWD, 		/* Give me a password for x	*/
  CON_CNFPASSWD, 		/* Please retype password:	*/
  CON_QSEX,	 		/* Sex?				*/
  CON_QCLASS,	 		/* Class?			*/
  CON_RMOTD,	 		/* PRESS RETURN after MOTD	*/
  CON_MENU,			/* Your choice: (main menu)	*/
  CON_EXDESC,			/* Enter a new description:	*/
  CON_CHPWD_GETOLD,		/* Changing passwd: get old	*/
  CON_CHPWD_GETNEW,		/* Changing passwd: get new	*/
  CON_CHPWD_VRFY, 		/* Verify new password		*/
  CON_DELCNF1,			/* Delete confirmation 1	*/
  CON_DELCNF2	 		/* Delete confirmation 2	*/
};


/* Character equipment positions: used as index for char_data.equipment[] */
/* NOTE: Don't confuse these constants with the ITEM_ bitvectors
   which control the valid places you can wear a piece of equipment */
enum eq_position {
  WEAR_LIGHT,
  WEAR_FINGER_R,
  WEAR_FINGER_L,
  WEAR_NECK_1,
  WEAR_NECK_2,
  WEAR_BODY,
  WEAR_HEAD,
  WEAR_LEGS,
  WEAR_FEET,
  WEAR_HANDS,
  WEAR_ARMS,
  WEAR_SHIELD,
  WEAR_ABOUT,
  WEAR_WAIST,
  WEAR_WRIST_R,
  WEAR_WRIST_L,
  WEAR_WIELD,
  WEAR_HOLD
};

#define NUM_WEARS      18	/* This must be the # of eq positions!! */


/* object-related defines ********************************************/


/* Item types: used by obj_data.obj_flags.type_flag */
  enum item_type {
  ITEM_LIGHT = 1,		/* Item is a light source	*/
  ITEM_SCROLL,	 		/* Item is a scroll		*/
  ITEM_WAND,			/* Item is a wand		*/
  ITEM_STAFF,			/* Item is a staff		*/
  ITEM_WEAPON,			/* Item is a weapon		*/
  ITEM_FIREWEAPON,		/* Unimplemented		*/
  ITEM_MISSILE,			/* Unimplemented		*/
  ITEM_TREASURE,		/* Item is a treasure, not gold	*/
  ITEM_ARMOR,			/* Item is armor		*/
  ITEM_POTION,	 		/* Item is a potion		*/
  ITEM_WORN,			/* Unimplemented		*/
  ITEM_OTHER,			/* Misc object			*/
  ITEM_TRASH,			/* Trash - shopkeeps won't buy	*/
  ITEM_TRAP,			/* Unimplemented		*/
  ITEM_CONTAINER,		/* Item is a container		*/
  ITEM_NOTE,			/* Item is note 		*/
  ITEM_DRINKCON,		/* Item is a drink container	*/
  ITEM_KEY,			/* Item is a key		*/
  ITEM_FOOD,			/* Item is food			*/
  ITEM_MONEY,			/* Item is money (gold)		*/
  ITEM_PEN,			/* Item is a pen		*/
  ITEM_BOAT,			/* Item is a boat		*/
  ITEM_FOUNTAIN			/* Item is a fountain		*/
};


/* Take/Wear flags: used by obj_data.obj_flags.wear_flags */
struct can_be_worn_on {
  BIT(ITEM_WEAR_TAKE);		/* Item can be takes		*/
  BIT(ITEM_WEAR_FINGER);	/* Can be worn on finger	*/
  BIT(ITEM_WEAR_NECK);		/* Can be worn around neck 	*/
  BIT(ITEM_WEAR_BODY);		/* Can be worn on body 		*/
  BIT(ITEM_WEAR_HEAD);		/* Can be worn on head 		*/
  BIT(ITEM_WEAR_LEGS);		/* Can be worn on legs		*/
  BIT(ITEM_WEAR_FEET);		/* Can be worn on feet		*/
  BIT(ITEM_WEAR_HANDS);		/* Can be worn on hands		*/
  BIT(ITEM_WEAR_ARMS);		/* Can be worn on arms		*/
  BIT(ITEM_WEAR_SHIELD);	/* Can be used as a shield	*/
  BIT(ITEM_WEAR_ABOUT);		/* Can be worn about body 	*/
  BIT(ITEM_WEAR_WAIST);		/* Can be worn around waist 	*/
  BIT(ITEM_WEAR_WRIST);		/* Can be worn on wrist 	*/
  BIT(ITEM_WEAR_WIELD);		/* Can be wielded		*/
  BIT(ITEM_WEAR_HOLD);		/* Can be held			*/
};

/*
enum wear_flags {
  ITEM_WEAR_TAKE, ITEM_WEAR_FINGER, ITEM_WEAR_NECK, ITEM_WEAR_BODY,
  ITEM_WEAR_HEAD, ITEM_WEAR_LEGS, ITEM_WEAR_FEET, ITEM_WEAR_HANDS,
  ITEM_WEAR_ARMS, ITEM_WEAR_SHIELD, ITEM_WEAR_ABOUT, ITEM_WEAR_WAIST,
  ITEM_WEAR_WRIST, ITEM_WEAR_WIELD, ITEM_WEAR_HOLD
};
*/

/* Extra object flags: used by obj_data.obj_flags.extra_flags */
struct item_is {
  BIT(ITEM_GLOW);		/* Item is glowing		*/
  BIT(ITEM_HUM);		/* Item is humming		*/
  BIT(ITEM_NORENT);		/* Item cannot be rented	*/
  BIT(ITEM_NODONATE);		/* Item cannot be donated	*/
  BIT(ITEM_NOINVIS);		/* Item cannot be made invis	*/
  BIT(ITEM_INVISIBLE);		/* Item is invisible		*/
  BIT(ITEM_MAGIC);		/* Item is magical		*/
  BIT(ITEM_NODROP);		/* Item is cursed: can't drop	*/
  BIT(ITEM_BLESS);		/* Item is blessed		*/
  BIT(ITEM_ANTI_GOOD);		/* Not usable by good people	*/
  BIT(ITEM_ANTI_EVIL);		/* Not usable by evil people	*/
  BIT(ITEM_ANTI_NEUTRAL);	/* Not usable by neutral people	*/
  BIT(ITEM_ANTI_MAGIC_USER);	/* Not usable by mages		*/
  BIT(ITEM_ANTI_CLERIC);	/* Not usable by clerics	*/
  BIT(ITEM_ANTI_THIEF);		/* Not usable by thieves	*/
  BIT(ITEM_ANTI_WARRIOR);	/* Not usable by warriors	*/
  BIT(ITEM_NOSELL);		/* Shopkeepers won't touch it	*/
  BIT(ITEM_ANTI_PSI);
};


/* Modifier constants used with obj affects ('A' fields) */
enum apply {
  APPLY_NONE,			/* No effect			*/
  APPLY_STR,			/* Apply to strength		*/
  APPLY_DEX,			/* Apply to dexterity		*/
  APPLY_INT,			/* Apply to constitution	*/
  APPLY_WIS,			/* Apply to wisdom		*/
  APPLY_CON,			/* Apply to constitution	*/
  APPLY_CHA,			/* Apply to charisma		*/
  APPLY_CLASS,			/* Reserved			*/
  APPLY_LEVEL,			/* Reserved			*/
  APPLY_AGE,			/* Apply to age			*/
  APPLY_CHAR_WEIGHT,		/* Apply to weight		*/
  APPLY_CHAR_HEIGHT,		/* Apply to height		*/
  APPLY_MANA,			/* Apply to max mana		*/
  APPLY_HIT,			/* Apply to max hit points	*/
  APPLY_MOVE,			/* Apply to max move points	*/
  APPLY_GOLD,			/* Reserved			*/
  APPLY_EXP,			/* Reserved			*/
  APPLY_AC,			/* Apply to Armor Class		*/
  APPLY_HITROLL,		/* Apply to hitroll		*/
  APPLY_DAMROLL,		/* Apply to damage roll		*/
  APPLY_SAVING_PARA,		/* Apply to save throw: paralz	*/
  APPLY_SAVING_ROD,		/* Apply to save throw: rods	*/
  APPLY_SAVING_PETRI,		/* Apply to save throw: petrif	*/
  APPLY_SAVING_BREATH,		/* Apply to save throw: breath	*/
  APPLY_SAVING_SPELL,		/* Apply to save throw: spells	*/
  APPLY_SAVING_ALL,             /* Apply to ALL saving throws   */
  APPLY_HND                     /* Apply to both hit & dam rolls */
};


/* Container flags - value[1] */
#define CONT_CLOSEABLE      (1 << 0)	/* Container can be closed	*/
#define CONT_PICKPROOF      (1 << 1)	/* Container is pickproof	*/
#define CONT_CLOSED         (1 << 2)	/* Container is closed		*/
#define CONT_LOCKED         (1 << 3)	/* Container is locked		*/


/* Some different kind of liquids for use in values of drink containers */
enum liquid {
  LIQ_WATER,
  LIQ_BEER,
  LIQ_WINE,
  LIQ_ALE,
  LIQ_DARKALE,
  LIQ_WHISKY,
  LIQ_LEMONADE,
  LIQ_FIREBRT,
  LIQ_LOCALSPC,
  LIQ_SLIME,
  LIQ_MILK,
  LIQ_TEA,
  LIQ_COFFE,
  LIQ_BLOOD,
  LIQ_SALTWATER,
  LIQ_CLEARWATER
};


/* other miscellaneous defines *******************************************/


/* Player conditions */
enum condition {
  DRUNK,
  FULL,
  THIRST
};


/* Sun state for weather_data */
enum sun_state {
  SUN_DARK,
  SUN_RISE,
  SUN_LIGHT,
  SUN_SET
};


/* Sky conditions for weather_data */
enum sky_condition {
  SKY_CLOUDLESS,
  SKY_CLOUDY,
  SKY_RAINING,
  SKY_LIGHTNING
};


/* Rent codes */
enum rent_code {
  RENT_UNDEF,
  RENT_CRASH,
  RENT_RENTED,
  RENT_CRYO,
  RENT_FORCED,
  RENT_TIMEDOUT
};


/* other #defined constants **********************************************/

#define LVL_IMPL	34
#define LVL_GRGOD	33
#define LVL_GOD		32
#define LVL_IMMORT	31

#define LVL_FREEZE	LVL_GRGOD

#define NUM_OF_DIRS	6	/* number of directions in a room (nsewud) */

#define OPT_USEC	100000	/* 10 passes per second */
#define PASSES_PER_SEC	(1000000 / OPT_USEC)
#define RL_SEC		* PASSES_PER_SEC

#define PULSE_ZONE      (10 RL_SEC)
#define PULSE_MOBILE    (10 RL_SEC)
#define PULSE_VIOLENCE  (2 RL_SEC)

#define SMALL_BUFSIZE		1024
#define LARGE_BUFSIZE		(12 * 1024)
#define GARBAGE_SPACE		32

#define MAX_STRING_LENGTH	8192
#define MAX_INPUT_LENGTH	256	/* Max length per *line* of input */
#define MAX_RAW_INPUT_LENGTH	512	/* Max size of *raw* input */
#define MAX_MESSAGES		60
#define MAX_NAME_LENGTH		20  /* Used in char_file_u *DO*NOT*CHANGE* */
#define MAX_PWD_LENGTH		10  /* Used in char_file_u *DO*NOT*CHANGE* */
#define MAX_TITLE_LENGTH	80  /* Used in char_file_u *DO*NOT*CHANGE* */
#define HOST_LENGTH		30  /* Used in char_file_u *DO*NOT*CHANGE* */
#define EXDSCR_LENGTH		240 /* Used in char_file_u *DO*NOT*CHANGE* */
#define MAX_TONGUE		3   /* Used in char_file_u *DO*NOT*CHANGE* */
#define MAX_SKILLS		200 /* Used in char_file_u *DO*NOT*CHANGE* */
#define MAX_AFFECT		32  /* Used in char_file_u *DO*NOT*CHANGE* */
#define MAX_OBJ_AFFECT		6 /* Used in obj_file_elem *DO*NOT*CHANGE* */


/***********************************************************************
 * Structures                                                          *
 **********************************************************************/


typedef signed char		sbyte;
typedef unsigned char		ubyte;
typedef signed short int	sh_int;
typedef unsigned short int	ush_int;
typedef char			bool;
typedef char			byte;

typedef sh_int	room_num;
typedef sh_int	obj_num;


/* Extra description: used in objects, mobiles, and rooms */
struct extra_descr_data {
   char	*keyword;                 /* Keyword in look/examine          */
   char	*description;             /* What to see                      */
   struct extra_descr_data *next; /* Next in list                     */
};


/* object-related structures ******************************************/


/* object flags; used in obj_data */
struct obj_flag_data {
  int	value[4];	/* Values of the item (see list)    */
  byte type_flag;	/* Type of item			    */
  struct can_be_worn_on wear_flags;		/* Where you can wear it	    */
  struct item_is extra_flags;	/* If it hums, glows, etc.	    */
  int	weight;		/* Weigt what else                  */
  int	cost;		/* Value when sold (gp.)            */
  int	cost_per_day;	/* Cost to keep pr. real day        */
  int	timer;		/* Timer for object                 */
  struct affections bitvector;		/* To set chars bits                */
};


/* Used in obj_file_elem *DO*NOT*CHANGE* */
struct obj_affected_type {
   byte location;      /* Which ability to change (APPLY_XXX) */
   sbyte modifier;     /* How much it changes by              */
};


/* ================== Memory Structure for Objects ================== */
struct obj_data {
   obj_num item_number;		/* Where in data-base			*/
   room_num in_room;		/* In what room -1 when conta/carr	*/

   struct obj_flag_data obj_flags;  /* Object information               */
   struct obj_affected_type affected[MAX_OBJ_AFFECT];  /* affects */

   char	*name;                    /* Title of object :get etc.        */
   char	*description;		  /* When in room                     */
   char	*short_description;       /* when worn/carry/in cont.         */
   char	*action_description;      /* What to write when used          */
   struct extra_descr_data *ex_description; /* extra descriptions     */
   struct char_data *carried_by;  /* Carried by :NULL in room/conta   */
   struct char_data *worn_by;	  /* Worn by?			      */
   sh_int worn_on;		  /* Worn where?		      */

   struct obj_data *in_obj;       /* In what object NULL when none    */
   struct obj_data *contains;     /* Contains objects                 */

   struct obj_data *next_content; /* For 'contains' lists             */
   struct obj_data *next;         /* For the object list              */
};
/* ======================================================================= */


/* ====================== File Element for Objects ======================= */
/*                 BEWARE: Changing it will ruin rent files		   */
struct obj_file_elem {
   obj_num item_number;

   int	value[4];
   int	extra_flags;
   int	weight;
   int	timer;
   struct affections	bitvector;
   struct obj_affected_type affected[MAX_OBJ_AFFECT];
};


/* header block for rent files.  BEWARE: Changing it will ruin rent files  */
struct rent_info {
   int	time;
   int	rentcode;
   int	net_cost_per_diem;
   int	gold;
   int	account;
   int	nitems;
   int	spare0;
   int	spare1;
   int	spare2;
   int	spare3;
   int	spare4;
   int	spare5;
   int	spare6;
   int	spare7;
};
/* ======================================================================= */


/* room-related structures ************************************************/


struct room_direction_data {
   char	*general_description;	/* When look DIR.			*/

   char	*keyword;		/* for open/close			*/

   struct exit_info exit;	/* Exit info				*/
   obj_num key;			/* Key's number (-1 for no key)		*/
   room_num to_room;		/* Where direction leads (NOWHERE)	*/
};


/* ================== Memory Structure for room ======================= */
struct room_data {
  room_num number;		/* Rooms number	(vnum)		      */
  sh_int zone;			/* Room zone (for resetting)          */
  int	sector_type;		/* sector type (move/hide)            */
  char	*name;			/* Rooms name 'You are ...'           */
  char	*description;		/* Shown when entered                 */
  struct extra_descr_data *ex_description; /* for examine/look       */
  struct room_direction_data *dir_option[NUM_OF_DIRS]; /* Directions */
  struct roomflags
    room_flags;			/* DEATH,DARK ... etc                 */

  byte light;			/* Number of lightsources in room     */
  SPECIAL(*func);

  struct obj_data *contents;	/* List of items in room              */
  struct char_data *people;	/* List of NPC / PC in room           */
};
/* ====================================================================== */


/* char-related structures ************************************************/


/* memory structure for characters */
struct memory_rec_struct {
   long	id;
   struct memory_rec_struct *next;
};

typedef struct memory_rec_struct memory_rec;


/* This structure is purely intended to be an easy way to transfer */
/* and return information about time (real or mudwise).            */
struct time_info_data {
   byte hours, day, month;
   sh_int year;
};


/* These data contain information about a players time data */
struct time_data {
   time_t birth;    /* This represents the characters age                */
   time_t logon;    /* Time of the last logon (used to calculate played) */
   int	played;     /* This is the total accumulated time played in secs */
};


/* general player-related info, usually PC's and NPC's */
struct char_player_data {
   char	passwd[MAX_PWD_LENGTH+1]; /* character's password      */
   char	*name;	       /* PC / NPC s name (kill ...  )         */
   char	*short_descr;  /* for NPC 'actions'                    */
   char	*long_descr;   /* for 'look'			       */
   char	*description;  /* Extra descriptions                   */
   char	*title;        /* PC / NPC's title                     */
   byte sex;           /* PC / NPC's sex                       */
   byte class;         /* PC / NPC's class		       */
   byte level;         /* PC / NPC's level                     */
   int	hometown;      /* PC s Hometown (zone)                 */
   struct time_data time;  /* PC's AGE in days                 */
   ubyte weight;       /* PC / NPC's weight                    */
   ubyte height;       /* PC / NPC's height                    */
};


/* Char's abilities.  Used in char_file_u *DO*NOT*CHANGE* */
struct char_ability_data {
   sbyte str;
   sbyte str_add;      /* 000 - 100 if strength 18             */
   sbyte intel;
   sbyte wis;
   sbyte dex;
   sbyte con;
   sbyte cha;
};


/* Char's points.  Used in char_file_u *DO*NOT*CHANGE* */
struct char_point_data {
   sh_int mana;
   sh_int max_mana;     /* Max move for PC/NPC			   */
   sh_int hit;
   sh_int max_hit;      /* Max hit for PC/NPC                      */
   sh_int move;
   sh_int max_move;     /* Max move for PC/NPC                     */

   sh_int armor;        /* Internal -100..100, external -10..10 AC */
   int	gold;           /* Money carried                           */
   int	bank_gold;	/* Gold the char has in a bank account	   */
   int	exp;            /* The experience of the player            */

   sbyte hitroll;       /* Any bonus or penalty to the hit roll    */
   sbyte damroll;       /* Any bonus or penalty to the damage roll */
};


/*
 * char_special_data_saved: specials which both a PC and an NPC have in
 * common, but which must be saved to the playerfile for PC's.
 *
 * WARNING:  Do not change this structure.  Doing so will ruin the
 * playerfile.  If you want to add to the playerfile, use the spares
 * in player_special_data.
 */
struct char_special_data_saved {
  int	alignment;		/* +-1000 for alignments                */
  long	idnum;			/* player's idnum; -1 for mobiles	*/
  union characterflags {
    struct playerflags	pc;
    struct mobflags	npc;
  } act;			/* act flag for NPC's; player flag for PC's */
  struct affections affected_by;/* Bitvector for spells/skills affected by */
  sh_int apply_saving_throw[5]; /* Saving throw (Bonuses)		*/
};


/* Special playing constants shared by PCs and NPCs which aren't in pfile */
struct char_special_data {
   struct char_data *fighting;	/* Opponent				*/
   struct char_data *hunting;	/* Char hunted by this char		*/

   byte position;		/* Standing, fighting, sleeping, etc.	*/

   int	carry_weight;		/* Carried weight			*/
   byte carry_items;		/* Number of items carried		*/
   int	timer;			/* Timer for update			*/

   struct char_special_data_saved saved; /* constants saved in plrfile	*/
};


/*
 *  If you want to add new values to the playerfile, do it here.  DO NOT
 * ADD, DELETE OR MOVE ANY OF THE VARIABLES - doing so will change the
 * size of the structure and ruin the playerfile.  However, you can change
 * the names of the spares to something more meaningful, and then use them
 * in your new code.  They will automatically be transferred from the
 * playerfile into memory when players log in.
 */
struct player_special_data_saved {
   byte skills[MAX_SKILLS+1];	/* array of skills plus skill 0		*/
   byte PADDING0;		/* used to be spells_to_learn		*/
   bool talks[MAX_TONGUE];	/* PC s Tongues 0 for NPC		*/
   int	wimp_level;		/* Below this # of hit points, flee!	*/
   byte freeze_level;		/* Level of god who froze char, if any	*/
   sh_int invis_level;		/* level of invisibility		*/
   room_num load_room;		/* Which room to place char in		*/
   struct preferances pref;	/* preference flags for PC's.		*/
   ubyte bad_pws;		/* number of bad password attemps	*/
   sbyte conditions[3];         /* Drunk, full, thirsty			*/

   /* spares below for future expansion.  You can change the names from
      'sparen' to something meaningful, but don't change the order.  */

   ubyte spare0;
   ubyte spare1;
   ubyte spare2;
   ubyte spare3;
   ubyte spare4;
   ubyte spare5;
   int spells_to_learn;		/* How many can you learn yet this level*/
   int spare7;
   int spare8;
   int spare9;
   int spare10;
   int spare11;
   int spare12;
   int spare13;
   int spare14;
   int spare15;
   int spare16;
   long	spare17;
   long	spare18;
   long	spare19;
   long	spare20;
   long	spare21;
};

/*
 * Specials needed only by PCs, not NPCs.  Space for this structure is
 * not allocated in memory for NPCs, but it is for PCs and the portion
 * of it labelled 'saved' is saved in the playerfile.  This structure can
 * be changed freely; beware, though, that changing the contents of
 * player_special_data_saved will corrupt the playerfile.
 */
struct player_special_data {
   struct player_special_data_saved saved;

   char	*poofin;		/* Description on arrival of a god.     */
   char	*poofout;		/* Description upon a god's exit.       */
   struct alias *aliases;	/* Character's aliases			*/
   long last_tell;		/* idnum of last tell from		*/
   void *last_olc_targ;		/* olc control				*/
   int last_olc_mode;		/* olc control				*/
};


/* Specials used by NPCs, not PCs */
struct mob_special_data {
   byte last_direction;     /* The last direction the monster went     */
   int	attack_type;        /* The Attack Type Bitvector for NPC's     */
   byte default_pos;        /* Default position for NPC                */
   memory_rec *memory;	    /* List of attackers to remember	       */
   byte damnodice;          /* The number of damage dice's	       */
   byte damsizedice;        /* The size of the damage dice's           */
   int wait_state;	    /* Wait state for bashed mobs	       */
};


/* An affect structure.  Used in char_file_u *DO*NOT*CHANGE* */
struct affected_type {
   sh_int type;          /* The type of spell that caused this      */
   sh_int duration;      /* For how long its effects will last      */
   sbyte modifier;       /* This is added to apropriate ability     */
   byte location;        /* Tells which ability to change(APPLY_XXX)*/
   struct affections bitvector;/* Tells which bits to set (AFF_XXX) */
   struct affected_type *next;
};


/* Structure used for chars following other chars */
struct follow_type {
   struct char_data *follower;
   struct follow_type *next;
};


/* ================== Structure for player/non-player ===================== */
struct char_data {
   int pfilepos;			 /* playerfile pos		  */
   sh_int nr;                            /* Mob's rnum			  */
   room_num in_room;                     /* Location (real room number)	  */
   room_num was_in_room;		 /* location for linkdead people  */

   struct char_player_data player;       /* Normal data                   */
   struct char_ability_data real_abils;	 /* Abilities without modifiers   */
   struct char_ability_data aff_abils;	 /* Abils with spells/stones/etc  */
   struct char_point_data points;        /* Points                        */
   struct char_special_data char_specials;	/* PC/NPC specials	  */
   struct player_special_data *player_specials; /* PC specials		  */
   struct mob_special_data mob_specials;	/* NPC specials		  */

   struct affected_type *affected;       /* affected by what spells       */
   struct obj_data *equipment[NUM_WEARS];/* Equipment array               */

   struct obj_data *carrying;            /* Head of list                  */
   struct descriptor_data *desc;         /* NULL for mobiles              */

   struct char_data *next_in_room;     /* For room->people - list         */
   struct char_data *next;             /* For either monster or ppl-list  */
   struct char_data *next_fighting;    /* For fighting list               */

   struct follow_type *followers;        /* List of chars followers       */
   struct char_data *master;             /* Who is char following?        */
};
/* ====================================================================== */


/* ==================== File Structure for Player ======================= */
/*             BEWARE: Changing it will ruin the playerfile		  */
struct char_file_u {
   /* char_player_data */
   char	name[MAX_NAME_LENGTH+1];
   char	description[EXDSCR_LENGTH];
   char	title[MAX_TITLE_LENGTH+1];
   byte sex;
   byte class;
   byte level;
   sh_int hometown;
   time_t birth;   /* Time of birth of character     */
   int	played;    /* Number of secs played in total */
   ubyte weight;
   ubyte height;

   char	pwd[MAX_PWD_LENGTH+1];    /* character's password */

   struct char_special_data_saved char_specials_saved;
   struct player_special_data_saved player_specials_saved;
   struct char_ability_data abilities;
   struct char_point_data points;
   struct affected_type affected[MAX_AFFECT];

   time_t last_logon;		/* Time (in secs) of last logon */
   char host[HOST_LENGTH+1];	/* host of last logon */
};
/* ====================================================================== */


/* descriptor-related structures ******************************************/


struct txt_block {
   char	*text;
   int aliased;
   struct txt_block *next;
};


struct txt_q {
   struct txt_block *head;
   struct txt_block *tail;
};


struct descriptor_data {
   int	descriptor;		/* file descriptor for socket		*/
   char	host[HOST_LENGTH+1];	/* hostname				*/
   byte	bad_pws;		/* number of bad pw attemps this login	*/
   int	connected;		/* mode of 'connectedness'		*/
   int	wait;			/* wait for how many loops		*/
   int	desc_num;		/* unique num assigned to desc		*/
   time_t login_time;		/* when the person connected		*/
   char	*showstr_head;		/* for paging through texts		*/
   char	*showstr_point;		/*		-			*/
   char	**str;			/* for the modify-str system		*/
   int	max_str;		/*		-			*/
   long	mail_to;		/* name for mail system			*/
   int	prompt_mode;		/* control of prompt-printing		*/
   char	inbuf[MAX_RAW_INPUT_LENGTH];  /* buffer for raw input		*/
   char	last_input[MAX_INPUT_LENGTH]; /* the last input			*/
   char small_outbuf[SMALL_BUFSIZE];  /* standard output buffer		*/
   char *output;		/* ptr to the current output buffer	*/
   int  bufptr;			/* ptr to end of current output		*/
   int	bufspace;		/* space left in the output buffer	*/
   struct txt_block *large_outbuf; /* ptr to large buffer, if we need it */
   struct txt_q input;		/* q of unprocessed input		*/
   struct char_data *character;	/* linked to char			*/
   struct char_data *original;	/* original char if switched		*/
   struct descriptor_data *snooping; /* Who is this char snooping	*/
   struct descriptor_data *snoop_by; /* And who is snooping this char	*/
   struct descriptor_data *next; /* link to next descriptor		*/
};


/* other miscellaneous structures ***************************************/


struct msg_type {
   char	*attacker_msg;  /* message to attacker */
   char	*victim_msg;    /* message to victim   */
   char	*room_msg;      /* message to room     */
};


struct message_type {
   struct msg_type die_msg;	/* messages when death			*/
   struct msg_type miss_msg;	/* messages when miss			*/
   struct msg_type hit_msg;	/* messages when hit			*/
   struct msg_type god_msg;	/* messages when hit on god		*/
   struct message_type *next;	/* to next messages of this kind.	*/
};


struct message_list {
   int	a_type;			/* Attack type				*/
   int	number_of_attacks;	/* How many attack messages to chose from. */
   struct message_type *msg;	/* List of messages.			*/
};


struct dex_skill_type {
   sh_int p_pocket;
   sh_int p_locks;
   sh_int traps;
   sh_int sneak;
   sh_int hide;
};


struct dex_app_type {
   sh_int reaction;
   sh_int miss_att;
   sh_int defensive;
};


struct str_app_type {
   sh_int tohit;    /* To Hit (THAC0) Bonus/Penalty        */
   sh_int todam;    /* Damage Bonus/Penalty                */
   sh_int carry_w;  /* Maximum weight that can be carrried */
   sh_int wield_w;  /* Maximum weight that can be wielded  */
};


struct wis_app_type {
   byte bonus;       /* how many practices player gains per lev */
};


struct int_app_type {
   byte learn;       /* how many % a player learns a spell/skill */
};


struct con_app_type {
   sh_int hitp;
   sh_int shock;
};


struct weather_data {
   int	pressure;	/* How is the pressure ( Mb ) */
   int	change;	/* How fast and what way does it change. */
   int	sky;	/* How is the sky. */
   int	sunlight;	/* And how much sun. */
};


struct title_type {
   char	*title_m;
   char	*title_f;
   int	exp;
};


/* element in monster and object index-tables   */
struct index_data {
   int	virtual;    /* virtual number of this mob/obj           */
   int	number;     /* number of existing units of this mob/obj	*/
   SPECIAL(*func);
};