/*
 * client.c: The routines for the metaserver reporting.
 * :::NOTE:::
 * If you use this software, send me an email letting me know, so
 * I can keep you informed of updates.
 * The current version is displayed when you telnet to the server,
 * but I'd prefer to be able just to send out one bulk mail to let
 * everyone using it know at the same time.  I make some small attempt to
 * maintain backwards compatibility, but in no way guarantee it, so I do
 * recommend you keep up on the current version.
 *
 * For those who don't understand RCS variable expansion, the version number
 * is here
 *		   vvv
 * $Id: s.client.c 1.12 02/12/09 14:44:57-05:00 mike@intel.velgarin.net $
 */

#include "conf.h"
#include "sysdep.h"

#include "structs.h"
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>

/*
 * Error conditions stored in report.ok
 * For the most part they're to help with debugging, since only
 * REPORT_OK, RETRY, FAILED would suffice, but they help track down
 * where the error occurred from.
 */
typedef enum {
	REPORT_OK, SOCKET, RETRY, GETHOSTBYNAME, CONNECT, UNINITIALIZED
} rpt_err;
#define MINUTES *60

struct report_data {
	rpt_err ok;    /* Is everything set up properly? */
	int socket; /* persistent udp socket */
	time_t update; /* last time the dns information was updated. */
	struct sockaddr_in addr; /* Store this stuff for the sendto() call */
	struct hostent *serv_data;
} report;

/*
 * Uncomment this next line if you want to limit the number of times it will try
 * to re-initialize itself in case of errors.
 */
/* #define MAX_FAILED_UPDATES 5 */
#ifdef MAX_FAILED_UPDATES
byte failed_updates = 0;
#endif

/*
 * This determines if you want a persistent connection and just use send()
 * or just to let sendto() handle the connecting.  If you have some minor
 * hangs when the mud tries to report, you'll want a persistent connection,
 * since it only has to connect one time.
 * If you want a persistent connection, uncomment the following line.
 */
/* #define PERSISTENT */

extern ush_int DFLT_PORT;  /* The default port the mud is running on. */
void nonblock(int);
extern struct descriptor_data *descriptor_list;
void basic_mud_log(const char *, ...);  /* Why include utils.h just for this? */
#define log basic_mud_log

/*
 * For cygwin
 * I don't have cygwin.  I don't know if it even supports this.  If it doesn't,
 * then there's a chance that the send() or sendto() could block.  This COULD
 * create some annoying pauses.  I'm not sure of that since I also set the socket
 * to nonblocking, at least for persistent connections.  If you notice this happening
 * and aren't using a persistent one, try it again with that define uncommented and
 * see if it goes away.
 */
#ifndef MSG_DONTWAIT
#define MSG_DONTWAIT 0x40
#endif

#ifndef MAX
#define MAX(a,b) ((a) > (b) ? (a) : (b))
#define MIN(a,b) ((a) < (b) ? (a) : (b))
#endif

/*
 * Uncomment the following define and replace BOGUS_MUDNAME with the name of your
 * mud.  Failure to uncomment it will stop it from compiling.  Failure to change
 * it from the default will keep you from being listed.
 */
/* #define MUDNAME "BOGUS_MUDNAME" */
#if !defined(MUDNAME)
#error "You must uncomment the above line which defines MUDNAME and replace"
#error "'BOGUS_MUDNAME' with the name of your mud.  It must not contain a '*'"
#endif

/*
 * If you have some dynamic dns you use, set it here.
 * Leaving this as the default value will result in your fqdn being listed as
 * whatever your ip resolves to (which is often pretty ugly looking.)
 * This information IS verified.  First I take the address from the packet your
 * mud sends me.  Then I resolve whatever you fill in here.  If they don't match,
 * you don't get listed.  I don't check for aliases or multiple addresses, so this
 * had better resolve directly back to the ip your mud is running on.
 */
#define MUDDNS "BOGUS.DNS.NET"

/*
 * Do NOT change this.
 */
#define SERVER "velgarian.sytes.net"


void init_report_socket(void)
{
	/* create the socket */
	if((report.socket = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP)) < 0) {
		log("SYSERR: Can't open status socket: %s", strerror(errno));
		report.ok = SOCKET;
#ifdef MAX_FAILED_UPDATES
		failed_updates++;
#endif
		return FALSE;
	}
	/* make sure it's non-blocking */
	nonblock(report.socket);

	return TRUE;
}


void setup_report(void)
{
	static bool first = TRUE;

#ifdef MAX_FAILED_UPDATES
	if(failed_updates > MAX_FAILED_UPDATES) return;
#endif

	memset(&report.addr, 0, sizeof(report.addr));

	/*
	 * If this is the first time it's been called, set report.ok to UNINITIALIZED.
	 */
	if(first) {
		report.ok = UNINITIALIZED;
		first = FALSE;
	}

	report.update = time(0);

	/* get hostent info for the server */
	if(!(report.serv_data = gethostbyname(SERVER))) {
		switch(h_errno) {
		case TRY_AGAIN: /* A temporary error occured. */
		case NO_DATA:  /* Valid name but no IP address. Let's try it again later anyway. */
			if(report.ok != RETRY) { /* Avoid spamming the logs with "resource temp unavail" errors. */
				/* Yeah, hstrerror obsolete but it's more accurate */
				log("SYSERR: Can't get address of status server: %s.  Trying again later", hstrerror(h_errno));
				report.ok = RETRY; /* It is a retryable error. */
			}
#ifdef PERSISTENT
			close(report.socket);
#endif
			break;
		default: /* A non-recoverable error occured */
			log("SYSERR: Fatal error in gethostbyname: %s", hstrerror(h_errno));
			report.ok = GETHOSTBYNAME;
#ifdef PERSISTENT
			close(report.socket);
#endif
			break;
		}
#ifdef MAX_FAILED_UPDATES
		failed_updates++;
#endif
		return;
	}

	/* set up report.addr with all the info */
	report.addr.sin_family = AF_INET;
	report.addr.sin_port = htons(5000);
	memcpy(&report.addr.sin_addr, report.serv_data->h_addr, report.serv_data->h_length);

#ifdef PERSISTENT

	if(!init_report_socket()) return;

	/* Now connect to it. */
	if(connect(report.socket, (struct sockaddr *)&report.addr, sizeof(report.addr)) < 0) {
		log("SYSERR: Can't connect: %s", strerror(errno));
		close(report.socket);
		report.ok = CONNECT;
#ifdef MAX_FAILED_UPDATES
		failed_updates++;
#endif
		return;
	}
#endif

	/* if we got this far, then everything is (hopefully) set up ok */
	report.ok = REPORT_OK;
#ifdef MAX_FAILED_UPDATES
	failed_updates = 0;
#endif
}

void reset_report(void)
{
#ifdef PERSISTENT
	if(report.ok == REPORT_OK) {
		close(report.socket);
	}
#endif

	setup_report();
}

/*
 * This defines the maximum length of the UDP packet sent.
 * 1024 is probably excessive, but it's also the largest one
 * that I'll receive.  The actual size of the packet sent is
 * set to the length of the report string that's built.
 */
#define BUFLEN 1024
#if BUFLEN > 1024
#error "BUFLEN must be 1024 or less."
#endif

void report_status(void)
{
	char stats[BUFLEN];
	extern time_t boot_time;
	struct descriptor_data *d;
	int users = 0;

	/*
	 * if everything isn't set up ok, report.ok is false, so just return
	 * Unless we have gethostbyname() return TRY_AGAIN, then we give it another
	 * shot.  Also, once we're set up and running, if send()/sendto() fails, we set
	 * report.ok to RETRY and set everything up again, in case something changed.
	 * The 60 minute wait is so we don't slow things down with constant calls to
	 * gethostbyname() which tends to annoyingly block.  The only way around that is
	 * to thread the call, and I don't think most people want me threading their muds,
	 * although it's very simple, and does have a number of benefits here & there, but
	 * you also REALLY need to know what you're doing.
	 */
	if(report.ok != REPORT_OK) {
		switch(report.ok) {
		case RETRY:
			if((time(0) - report.update) > (60 MINUTES))
				reset_report();
			break;
		default:
			return;
		}
		return;
	}

	/* count the connected users */
	for(d = descriptor_list; d; d = d->next, users++);

	/* make the status string to send */
	memset(stats, 0, BUFLEN);
	snprintf(stats, BUFLEN, "%s*%ld*%d*%d*%s", MUDNAME, boot_time, users, DFLT_PORT, MUDDNS);
	
	/*
	 * Send the packet.  If we have a persistent connection, just use send.  Otherwise,
	 * use sendto() since it takes care of connecting for us.
	 */
#ifdef PERSISTENT
	if(send(report.socket, stats, (size_t)MIN(1024, strlen(stats)), MSG_DONTWAIT) < 0) {
#else
	if(!init_report_socket()) return;
	
	if(sendto(report.socket, stats, (size_t)MIN(1024, strlen(stats)), MSG_DONTWAIT,
				(struct sockaddr *)&report.addr, sizeof(report.addr)) < 0) {
#endif
		log("SYSERR: report_status sending: %s", strerror(errno));
		if((errno != EAGAIN) && (errno != EWOULDBLOCK)) {
			report.ok = RETRY;
		}
		return;
	}
#ifndef PERSISTENT
	close(report.socket);
#endif
}
