/* ************************************************************************
*  File: act.olc.c                               Part of Death's Gate MUD *
*                                                                         *
*  Usage: Commands for OLC                                                *
*                                                                         *
*  All rights reserved.  See license.doc for complete information.        *
*                                                                         *
*  Death's Gate MUD is based on CircleMUD, Copyright (C) 1993, 94.        *
*  CircleMUD is based on DikuMUD, Copyright (C) 1990, 1991.               *
*                                                                         *
*  $Author: tlo $
*  $Date: 1999/03/04 14:35:31 $
*  $Revision: 6.19 $
************************************************************************ */

#include "conf.h"
#include "sysdep.h"

#include "structs.h"
#include "scripts.h"
#include "index.h"
#include "config.h"
#include "memory.h"
#include "comm.h"
#include "utils.h"
#include "handler.h"
#include "db.h"
#include "interpreter.h"
#include "olc.h"
#include "constants.h"
#include "editor.h"
#include "find.h"
#include "spec.h"
#include "screen.h"
#include "events.h"
#include "shop.h"
#include "spells.h"

void do_show_zone(struct char_data *ch, int zone);
struct olc_save_info *olc_save_list = NULL;
void gen_olc_save(int savenum, int subcmd);

extern int real_shop(int vshop_num);
extern void free_shop(struct shop_data *shop);

extern struct shop_data *shop_index;
extern int top_shop;
extern const char *spells[];
extern const char *mobile_types[];
extern int find_owner_zone(int number);

OLCCMD(olc_create);
OLCCMD(olc_special);

#ifdef GOT_RID_OF_IT
OLCCMD(olc_add);
OLCCMD(olc_delete);
OLCCMD(olc_done);
OLCCMD(olc_exit);
OLCCMD(olc_extradesc);
OLCCMD(olc_help);
OLCCMD(olc_look);
OLCCMD(olc_remove);
OLCCMD(olc_save);
OLCCMD(olc_show);
OLCCMD(olc_trigger);
#endif

const char *olc_modes[] = {
  "ERROR",
  "mobile",
  "object",
  "room",
  "zone",
  "shop",
  "trigger"
};


const char *save_info_msg[] = { 
  "ERROR",
  "Mobiles", 
  "Objects", 
  "Rooms", 
  "Zone info", 
  "Shops", 
  "Triggers"
};

/* OLC Utility Functions */

void *get_current_olc_ptr(char_data *ch, int target, int mode)
{
    index_data *index;
    sh_int real_num;

    if (ch == NULL)
      return NULL;

    if (!OLC_DATA(ch))
      return NULL;      

    switch (mode) {
    case OLC_MEDIT:
	if (OLC_MOB(ch))
	    return OLC_MOB(ch);
	break;
    case OLC_OEDIT:
	if (OLC_OBJ(ch))
	    return OLC_OBJ(ch);
	break;
    case OLC_REDIT:
	if (OLC_ROOM(ch))
	    return OLC_ROOM(ch);
	break;
    case OLC_ZEDIT:
	if (OLC_ZONE(ch))
	    return OLC_ZONE(ch);
	break;
    case OLC_SEDIT:
	if (OLC_SHOP(ch))
	    return OLC_SHOP(ch);
	break;
    case OLC_TRIGEDIT:
	if (OLC_TRIG(ch))
	    return OLC_TRIG(ch);
	break;
    }

    return NULL;
}


void *get_olc_ptr(int target, int mode)
{
    index_data *index;
    sh_int real_num;

    switch (mode) {
    case OLC_MEDIT:
	if ((index = mob_index[target]))
	    return index->proto;
	break;
    case OLC_OEDIT:
	if ((index = obj_index[target]))
	    return index->proto;
	break;
    case OLC_REDIT:
    case OLC_ZEDIT:
	if (target >= 0 && target <= top_of_world)
	    return world[target];
	break;
    case OLC_SEDIT:
        real_num = real_shop(target);
        if (real_num >= 0)
	  return (shop_index + real_num);
	break;
    case OLC_TRIGEDIT:
	if ((index = trig_index[target]))
	    return index->proto;
	break;
    }

    return NULL;
}


/* returns zone (rnum) {targ, mode} falls in, or -1 if none */
int find_target_zone(int target, int mode)
{
  void *ptr;
  int vnum = -1, real_num = -1;

  if ((ptr = get_olc_ptr(target, mode)) == NULL)
    return -1;

  switch (mode) {
  case OLC_MEDIT: vnum = GET_MOB_NUM((struct char_data *) ptr); break;
  case OLC_OEDIT: vnum = GET_OBJ_NUM((struct obj_data *) ptr);  break;
  case OLC_REDIT: vnum = world[target] ? target : -1;           break;
  case OLC_ZEDIT: return target;                                break;
  case OLC_SEDIT: vnum = (real_shop(target) > 0) ? target : -1; break;
  case OLC_TRIGEDIT: vnum = GET_TRIG_RNUM((trig_data *) ptr);   break;
//  case OLC_SEDIT: vnum = real_shop(target);			break;
  }
  
  return find_owner_zone(vnum);
}


/*
 * checks if ch has permission to edit in zone (vnum).  0 if no permission
 */
int check_zone_permission(struct char_data *ch, int zone)
{
  char *s;
  int len;;

  len = strlen(GET_NAME(ch));

  if (zone < 0 || zone > top_of_zone_table)
    return 0;

  else if (GET_TRUST(ch) >= TRUST_ADMIN)
    return 1;

  else if (zone_table[zone].permissions &&
	   !str_cmp(zone_table[zone].permissions, "all"))
    return 1;

  else if (!zone_table[zone].permissions || 
	   (s = str_str(zone_table[zone].permissions, GET_NAME(ch))) == NULL)
    return 0;

  else if ((zone_table[zone].permissions != s && !isspace(*(s - 1))) ||
	   (s[len] && !isspace(s[len])))
    return 0;

  else
    return 1;
}


/* checks if ch has permission to edit targ (rnum).  0 if no permission */
int check_olc_permission(struct char_data *ch, int targ, int mode)
{
  int zone;

  if ((zone = find_target_zone(targ, mode)) < 0)
    return 0;

  return check_zone_permission(ch, zone);
}


int find_olc_target(struct char_data *ch, char *arg, int mode)
{
  int vnum, i = -1;

  if (!*arg) {
    switch (mode) {
    case OLC_MEDIT:
    case OLC_OEDIT:
    case OLC_SEDIT:
    case OLC_TRIGEDIT:
      return -1;
    case OLC_REDIT:
    case OLC_ZEDIT:
      return IN_ROOM(ch);
    }
  } else {
    vnum = atoi(arg);
    switch (mode) {
//     case OLC_MEDIT:  i = vnum;		  break;
//     case OLC_OEDIT:  i = vnum;		  break;
//     case OLC_REDIT:
//     case OLC_ZEDIT:  i = real_room(vnum); break;
    case OLC_TRIGEDIT:
    case OLC_MEDIT:
    case OLC_OEDIT:
    case OLC_REDIT:
    case OLC_SEDIT:  i = vnum;		  break;
    case OLC_ZEDIT:  i = real_room(vnum); break;
//    case OLC_SEDIT:  i = real_shop(vnum); break;
    
    }

    return i;
  }

  return -1;
}


int verify_olc_target(struct char_data *ch, sh_int vnum, int mode)
{
  int i = -1;

  switch (mode) {
    case OLC_TRIGEDIT: i = (trig_index[vnum] ? vnum : -1); break;
    case OLC_MEDIT: i = (mob_index[vnum] ? vnum : -1); break;
    case OLC_OEDIT: i = (obj_index[vnum] ? vnum : -1); break;
    case OLC_REDIT:
    case OLC_ZEDIT:  i = ((real_room(vnum) >= 0) ? vnum : -1); break;
    case OLC_SEDIT:  i = ((real_shop(vnum) >= 0) ? vnum : -1); break;
  }

  return i;
}


int olc_edit_integer(struct char_data *ch, long *result, char *argument,
		     char *name, long min, long max)
{
  long n;

  if (*argument) {
    n = atoi(argument);
    if (min == max || (n <= max && n >= min)) {
      *result = n;
      sprintf(buf, "%s set to %ld.\r\n", name, n);
      send_to_char(CAP(buf), ch);
      return 1;
    }
  }

//   if (min != max)
//     sprintf(buf, "%s must be between %ld and %ld.\r\n", name, min, max);
//   else
//     sprintf(buf, "Enter a number for %s.\r\n", name);

//  send_to_char(CAP(buf), ch);

  return 0;
}


int olc_edit_value(struct char_data *ch, long *result, char *argument,
		   char *name, long min, long max, const char **values, byte diff = FALSE)
{
  long n, i, start;
  
  start = MAX(min, 0);
  if (!max)
    max = 32000;

  if (!*argument || *argument == '?') {
//    sprintf(buf, "Valid values for %s are:\r\n", name);

    *buf = '\0';
    
    /* list values in columns */
    for (i = 0; (*values[i + start] != '\n') && ((i + start) <= max); ++i) {
      sprintf(buf + strlen(buf), "  &cG%2ld&c0) %-17.17s", i + 1, values[i + start]);
      if ((i % 3) == 2)
	strcat(buf, "\r\n");
    }

    if (i % 3)
      strcat(buf, "\r\n");
    page_string(ch->desc, buf, TRUE);
    return 0;
  }

  if ((*argument == '-') || isdigit(*argument)) {
    n = atoi(argument);
    --n;
    for (i = 0; ((*values[i] != '\n') && ((i + start) <= max)); ++i);
    n += start;
    if ((n < min) || (n > (i + start))) {
      sprintf(buf, "Unknown value for %s.\r\n", name);
      send_to_char(buf, ch);
      return 0;
    }
  } else {
    if ((n = search_block(argument, values, FALSE)) < 0) {
    sprintf(buf, "Unknown value for %s.\r\n", name);
    send_to_char(buf, ch);
    return 0;
  }
  }

//  if (n >= 0)
    send_to_charf(ch, "%s set to %s.\r\n", name, values[n]);

  if (diff) {
    n -= min;
  }

  *result = n;

  return 1;
}


// Display function used by olc_edit_bitvector
void olc_display_bitvector(char_data *ch, const char **values, long &mask, sh_int &i)
{
  /* list values */
  sprintf(buf, "Valid flags are:\r\n");
  
  for (i = 0; *values[i] != '\n'; ++i) {
    if (!(mask & (1 << i)))
      sprintf(buf1, "&cG%2d&c0", i + 1);
    else
      sprintf(buf1, "%s", " *");
    sprintf(buf + strlen(buf), "  %s) %-20s%s", buf1,
            values[i], ((i + 1) % 3) ? "" : "\r\n");
  }
  
  if (i % 3)
    strcat(buf, "\r\n");
  
  if (mask)
    strcat(buf, "Selections marked with * cannot be changed.\r\n\n");
    
  send_to_char(buf, ch);
}


// Edit routine designed for Oasis bitvector menus.  Returns 0 for quitting
// a menu, 1 for invalid input, 2 for successful input.
int olc_edit_bitvector(struct char_data *ch, long *result, char *argument,
		       long old_bv, char *name, const char **values, long mask)
{
  sh_int i = 0;
  byte numbers = TRUE, remove = FALSE;
  long flag, bv = 0;
  char *s, buf[MAX_STRING_LENGTH], buf2[MAX_STRING_LENGTH];

  if (!*argument || *argument == '?') {

    olc_display_bitvector(ch, values, mask, i);
    *buf = '\0';

    for (i = 0; i < 32; ++i) {
      if (IS_SET(old_bv, (1 << i))) {
        if (*buf) {
          sprintf(buf + strlen(buf), ", %s", values[i]);
        } else {
          strcpy(buf, values[i]);
        }
      }
    }
    send_to_charf(ch, "\r\nCurrent %s: &cC%s&c0\r\n"
                  "Enter selections separated by spaces, or select 0 to quit: ", 
                  name, (*buf ? buf : "NONE") );

    return 1;
  }

  if (*argument == '0') {
    *result = old_bv;
    return 0;
  }
  
  // Figure out the top value in this bitvector.
  for (i = 0; *values[i] != '\n'; ++i);
  
  half_chop(argument, buf, argument);

  while (*buf) {
    flag = atoi(buf);
    --flag;
    if (flag < 0)
      break;
    if ((flag < i) && (!(mask & (1 << flag))))
      SET_BIT(bv, 1 << flag);

    half_chop(argument, buf, argument);
  }

  /* mask out changes to reserved bits */
  *result = old_bv ^ bv;

  return 2;
}


int olc_edit_string(struct char_data *ch, char **string, char *argument,
		    char *name, EDITFUNC(*func), void *ed_data, int size,
		    int append)
{
  int len = 0, alen;
  char buf[MAX_STRING_LENGTH], *str;



  if (*argument == '+') {	/* append mode */
    if (*string) {
      len = strlen(*string);
      if (len >= size) {
	send_to_char("The string is already at its maximum length.\r\n", ch);
	return 0;
      }

      SET_BIT(append, EDIT_APPEND);

    }

    argument++;
  }

  if (!*argument) {		/* switch to editor */
    if (IS_SET(append, EDIT_APPEND))
      sprintf(buf, "Appending %s (maximum of %d characters).\r\n"
	      "Current string is:\r\n%s\r\n", name, size - len, *string);
    else
      sprintf(buf, "Enter new %s (maximum of %d characters).\r\n",
	      name, size);

    send_to_char(buf, ch);
    string_edit(ch->desc, *string, size, func, ed_data, append);

    act("$n starts editing.", TRUE, ch, NULL, NULL, TO_ROOM);
    
    return 1;
  }

  else {			/* string given with command */
    if (ed_data)
      FREE(ed_data);

    alen = strlen(argument);
    if ((alen + len) >= size)
      argument[size - len - 1] = '\0';

    CREATE(str, char, MIN(size, len + alen + 1));
    if (IS_SET(append, EDIT_APPEND))
      strcpy(str, *string);
    strcpy(str + len, argument);

    if (*string)
      FREE(*string);
    
    *string = str;

    sprintf(buf, "%s %s.\r\n", name, (IS_SET(append, EDIT_APPEND)) ? 
            "appended" : "changed");
    send_to_char(CAP(buf), ch);

    return 1;
  }      
}


int olc_edit_sstring(struct char_data *ch, sstring **string, char *argument,
		     char *name, EDITFUNC(*func), void *ed_data, int size, 
		     int append)
{
  int len = 0, alen;
  char buf[MAX_STRING_LENGTH], *str;

  if (*argument == '+') {	/* append string */
    if (*string) {
      len = strlen(ss_data(*string));
      if (len >= size) {
	send_to_char("The string is already at its maximum length.\r\n", ch);
	return 0;
      }

      SET_BIT(append, EDIT_APPEND);
    }

    argument++;
  }

  if (!*argument) {		/* switch to editor */
    if (IS_SET(append, EDIT_APPEND))
      sprintf(buf, "Appending %s (maximum of %d characters).\r\n"
	      "Current string is:\r\n%s\r\n\r\n", name, size - len,
	      ss_data(*string));
    else
      sprintf(buf, "Enter new %s (maximum of %d characters).\r\n",
	      name, size);

    send_to_char(buf, ch);
    string_edit(ch->desc, ss_data(*string), size, func, ed_data, append);

    act("$n starts editing.", TRUE, ch, NULL, NULL, TO_ROOM);

    return 1;
  }

  else {			/* string given with command */
    if (ed_data)
      FREE(ed_data);

    alen = strlen(argument);
    if ((alen + len) >= size)
      argument[size - len - 1] = '\0';

    CREATE(str, char, MIN(size, len + alen + 1));
    if (IS_SET(append, EDIT_APPEND))
      strcpy(str, ss_data(*string));
    strcpy(str + len, argument);

    ss_free(*string);
    *string = ss_create(str);
    FREE(str);

    sprintf(buf, "%s %s.\r\n", name, (IS_SET(append, EDIT_APPEND)) ? 
            "appended" : "changed");
    send_to_char(CAP(buf), ch);

    return 1;
  }      
}


// Edit object values generically; returns 0 for invalid parameter -DH
int olc_edit_objval(struct char_data *ch, char *argument, char *name)
{
  long n, i, min, max, *result = NULL;
  byte objtype;
  byte objval;
  const char **values;
  
  if ((OLC_DATA(ch) == NULL) || (OLC_OBJ(ch) == NULL)) {
    log ("Serious screwup!  olc_edit_objval called without OLC object loaded!");
    return 0;
  }

  objtype = GET_OBJ_TYPE(OLC_OBJ(ch));
  objval = OLC_MENU(ch) - OEDIT_VALUE_1;
  min = obj_values_tab[objtype][objval].min;
  max = obj_values_tab[objtype][objval].max;
  values = obj_values_tab[objtype][objval].values;
  result = (long *) &OLC_OBJ(ch)->obj_flags.value[objval];

  switch (obj_values_tab[objtype][objval].type) {
  case EDK_INTEGER:
    if (!olc_edit_integer(ch, result, argument, name,
         obj_values_tab[objtype][objval].min,
         obj_values_tab[objtype][objval].max))
      return 0;
    break;
  case EDK_BITVECT:
    if (olc_edit_bitvector(ch, result, argument, *((long *) result), name, values, 
  	obj_values_tab[objtype][objval].max) != 2)
      return 0;
    break;
  case EDK_VALUE:
    if (!olc_edit_value(ch, result, argument, name, min, max, values))
      return 0;
    break;
  case EDK_SPELLVALUE:
    if (*argument == '0') {
      *result = -1;
      return 2;
    } else if (!olc_edit_value(ch, result, argument, name, min, max, values))
      return 0;
    break;
  case EDK_DIFFVALUE:
    if (*argument == '0') {
      *result = -1;
      return 2;
    } else if (!olc_edit_value(ch, result, argument, name, min, max, values, TRUE))
      return 0;
    break;
  default:
    send_to_char("Undefined case in olc_edit_objval!  Tell the imps!\r\n", ch);
  }
    
  return 1;
}


/* OLC Interpreter Functions */

int olc_gen_edit(struct char_data *ch, char *command, char *argument);

void olc_saveinfo(struct char_data *ch)
{ 
  struct olc_save_info *entry;

  if (olc_save_list)
    send_to_char("The following OLC components need saving:-\r\n", ch);
  else
    send_to_char("The database is up to date.\r\n", ch);

  for (entry = olc_save_list; entry; entry = entry->next) { 
    sprintf(buf, " - %s for zone %d.\r\n", 
	save_info_msg[(int)entry->type],
	entry->zone);
    send_to_char(buf, ch);
  }
}


ACMD(do_olc)
{
//  char *s, *p, arg[MAX_INPUT_LENGTH], buf[256], *name = NULL;
//  int targ = -1, old_targ, old_mode;
//  void *ptr;

  sh_int targ = -1, copyfrom = 0;
  char_data *tch;
  void *ptr;
  char arg[MAX_INPUT_LENGTH];

  extern void oedit_setup_new(char_data *ch, int copyfrom);
  extern void oedit_setup_existing(char_data *ch, int copyfrom);
  extern void medit_setup_new(char_data *ch, int copyfrom);
  extern void medit_setup_existing(char_data *ch, int copyfrom);
  extern void redit_setup_new(char_data *ch, int copyfrom);
  extern void redit_setup_existing(char_data *ch, int copyfrom);
  extern void zedit_setup(char_data *ch, int room_num);
  extern void sedit_setup_new(char_data *ch);
  extern void sedit_setup_existing(char_data *ch, int rshop_num);
  extern void trigedit_setup_new (char_data * ch, int copyfrom);
  extern void trigedit_setup_existing (char_data * ch, int rtrg_num);

  if (IS_NPC(ch) || !ch->desc) {
    send_to_char("Sorry, OLC is limited to carbon based life forms.\r\n", ch);
    return;
  }

  if (!subcmd) {
    olc_saveinfo(ch);
    return;
  }

  half_chop(argument, arg, argument);

  if (*arg) {
    while (*arg) {
      if (isdigit(*arg)) {
  	targ = find_olc_target(ch, arg, subcmd);
      } else if (strncmp("copy", arg, 4) == 0) {
        if (subcmd == OLC_ZEDIT || subcmd == OLC_SEDIT) {
  	  send_to_char("Shop and zone copying are unsupported.\r\n", ch);
  	  return;
  	}

  	half_chop(argument, arg, argument);
 
  	if (*arg && isdigit(*arg)) {
  	  copyfrom = MAX(find_olc_target(ch, arg, subcmd), 0);
  	  if (verify_olc_target(ch, copyfrom, subcmd) == -1) {
  	    send_to_char("Source vnum does not exist!\r\n", ch);
  	    return;
  	  }
  	} else {
  	  send_to_char("Copy from which virtual number?\r\n", ch);
  	  return;
  	}
      } else if (strncmp("new", arg, 4) == 0) {
        if (subcmd != OLC_ZEDIT) {
	  send_to_char("You needn't use new, just enter an unused vnum.\r\n", ch);
	  return;
	}
  	half_chop(argument, arg, argument);
	olc_create(ch, OLC_ZEDIT, arg);
	return;
      } else if (strncmp("save", arg, 4) == 0) {
  	half_chop(argument, arg, argument);
 
  	if (*arg) {
	  targ = atoi(arg);
  	  if (real_zone(targ) == -1) {
  	    send_to_char("That zone doesn't exist!\r\n", ch);
  	    return;
  	  }
	  send_to_charf(ch, "Saving %s for zone %d\r\n", 
	                save_info_msg[subcmd], targ);
  	  gen_olc_save(targ, subcmd);
  	  return;
  	} else {
  	  send_to_char("What zone do you wish to save?\r\n", ch);
  	  return;
  	}
      }

      half_chop(argument, arg, argument);
    }
  } else {
    targ = find_olc_target(ch, "", subcmd);
  }
  
  if (targ < 0) {
    send_to_char("Edit what virtual number?\r\n", ch);
    return;
  }

  if (!check_zone_permission(ch, find_owner_zone(targ))) {
    send_to_charf(ch, "You don't have permission to edit %d.\r\n", targ);
    return;
  }

  /*. Check whatever it is isn't already being edited .*/
  for (tch = character_list; tch; tch = tch->next) {
    if (tch->player_specials->olc && (targ == OLC_NUM(tch)) &&
       (subcmd == GET_OLC_MODE(tch))) {
      sprintf(buf, "That %s is currently being edited by %s.\r\n",
  	      olc_modes[GET_OLC_MODE(tch)], GET_NAME(tch));
      send_to_char(buf, ch);
      return;
    }
  }

  /*. Give the character an OLC struct .*/
  CREATE(OLC_DATA(ch), struct olc_data, 1);
  STATE(ch->desc) = CON_MEDIT + subcmd - 1;

  /* go into editing mode */
  
  OLC_NUM(ch) = targ;
  GET_OLC_MODE(ch) = subcmd;
  
  OLC_SCRIPT(ch) = NULL;

  ptr = get_olc_ptr(targ, subcmd);
  
  switch (subcmd) {
  case OLC_MEDIT: 
    OLC_ITEM_TYPE(ch) = MOB_TRIGGER;
    if (ptr) {
      OLC_DATA(ch)->func = mob_index[targ]->func;
      if (mob_index[targ]->farg)
        OLC_DATA(ch)->farg = strdup(mob_index[targ]->farg);
    }
    break;
  case OLC_OEDIT: 
    OLC_ITEM_TYPE(ch) = OBJ_TRIGGER; 
    if (ptr) {
      OLC_DATA(ch)->func = obj_index[targ]->func;
      if (obj_index[targ]->farg)
        OLC_DATA(ch)->farg = strdup(obj_index[targ]->farg);
    }
    break;
  case OLC_REDIT: 
    OLC_ITEM_TYPE(ch) = WLD_TRIGGER; 
    if (ptr) {
      OLC_DATA(ch)->func = world[targ]->func;
      if (world[targ]->farg)
        OLC_DATA(ch)->farg = strdup(world[targ]->farg);
    }
    break;
  default: 
    OLC_ITEM_TYPE(ch) = -1;
    OLC_DATA(ch)->func = 0;
    OLC_DATA(ch)->farg = NULL;
  }

  SET_BIT(PLR_FLAGS(ch), PLR_WRITING);

  act("$n starts using OLC.", TRUE, ch, 0, 0, TO_ROOM);

  if (targ >= 0) {
    switch (subcmd) {
    case OLC_OEDIT: 
      if (ptr)
        oedit_setup_existing(ch, copyfrom);
      else
        oedit_setup_new(ch, copyfrom); 
      break;
    case OLC_MEDIT: 
      if (ptr)
        medit_setup_existing(ch, copyfrom);
      else
        medit_setup_new(ch, copyfrom); 
      break;
    case OLC_REDIT:
      if (ptr)
        redit_setup_existing(ch, copyfrom);
      else
        redit_setup_new(ch, copyfrom); 
      break;
    case OLC_ZEDIT:
      zedit_setup(ch, OLC_NUM(ch)); 
      break;
    case OLC_SEDIT:
      if (ptr)
        sedit_setup_existing(ch, real_shop(targ));
      else
        sedit_setup_new(ch); 
      break;
    case OLC_TRIGEDIT:
      if (ptr)
        trigedit_setup_existing(ch, copyfrom);
      else
        trigedit_setup_new(ch, copyfrom); 
      break;
    default:
      send_to_char("Unsupported.\r\n", ch);
      break;
    }
  }

}


struct olc_cmd_tab {
  char *cmd;
  void (*cmdptr)(struct char_data *ch, int subcmd, char *argument);
  byte cmd_valid[4];
  int  subcmd;
};


int olc_interpreter(struct char_data *ch, char *argument)
{
#ifdef GOT_RID_OF_IT
  return 0;

  char command[MAX_INPUT_LENGTH];
  int i, mode, length, old_mode, old_targ;

  /*
   * The command table for olc commands.  Format as follows:
   *   1:  command name             :  string,
   *   2:  command pointer          :  function pointer,
   *   3:  Array of modes valid for :  array of bytes,
   *          { MEDIT, OEDIT, REDIT, ZEDIT }
   *   4:  subcommand		    :  integer
   */
  struct olc_cmd_tab olccmdtab[] = {
    { "north"    , olc_exit     , { TRUE,  TRUE,  TRUE , TRUE  }, NORTH },
    { "east"     , olc_exit     , { TRUE,  TRUE,  TRUE , TRUE  }, EAST },
    { "south"    , olc_exit     , { TRUE,  TRUE,  TRUE , TRUE  }, SOUTH },
    { "west"     , olc_exit     , { TRUE,  TRUE,  TRUE , TRUE  }, WEST },
    { "up"       , olc_exit     , { TRUE,  TRUE,  TRUE , TRUE  }, UP },
    { "down"     , olc_exit     , { TRUE,  TRUE,  TRUE , TRUE  }, DOWN },
    { "add"	 , olc_add	, { FALSE, FALSE, FALSE, TRUE  }, 0 },
    { "create"   , olc_create   , { TRUE , TRUE , TRUE , TRUE  }, 0 },
    { "delete"   , olc_delete   , { TRUE , TRUE , TRUE , TRUE  }, 0 },
    { "done"     , olc_done     , { TRUE , TRUE , TRUE , TRUE  }, 0 },
    { "ed"       , olc_extradesc, { FALSE, TRUE , TRUE , FALSE }, 0 },
    { "extradesc", olc_extradesc, { FALSE, TRUE , TRUE , FALSE }, 0 },
    { "help"     , olc_help     , { TRUE , TRUE , TRUE , TRUE  }, 0 },
    { "?"        , olc_help     , { TRUE , TRUE , TRUE , TRUE  }, 0 },
    { "look"     , olc_look     , { TRUE , TRUE , TRUE , TRUE  }, 0 },
    { "remove"	 , olc_remove	, { FALSE, FALSE, FALSE, TRUE  }, 0 },
    { "save"     , olc_save     , { TRUE , TRUE , TRUE , TRUE  }, 0 },
    { "show"     , olc_show     , { TRUE , TRUE , TRUE , TRUE  }, 0 },
    { "special"  , olc_special  , { TRUE , TRUE , TRUE , FALSE }, 0 },
    { "trigger"  , olc_trigger  , { TRUE , TRUE , FALSE, FALSE }, 0 },
    { "quit"     , olc_done     , { TRUE , TRUE , TRUE , TRUE  }, 0 },

    { "\n", NULL, { FALSE, FALSE, FALSE, FALSE }}
  };

  if (!GET_OLC_MODE(ch)) {
    send_to_char("Error:  No OLC mode found!\r\n", ch);
    return 1;
  }

  skip_spaces(&argument);
  
  if (*argument == '\\') { /* command escape */
    /* save old mode/targ */
    old_mode = GET_OLC_MODE(ch);
    old_targ = OLC_NUM(ch);

    GET_OLC_MODE(ch) = 0;		/* To prevent recursion */
    command_interpreter(ch, argument + 1);

    /* restore old mode/targ */
    GET_OLC_MODE(ch) = old_mode;
    OLC_NUM(ch) = old_targ;

    return 1;
  }

  argument = any_one_arg(argument, command);
  mode = GET_OLC_MODE(ch) - 1;

  for (i = 0, length = strlen(command); *olccmdtab[i].cmd != '\n'; i++)
    if (!strncmp(olccmdtab[i].cmd, command, length) &&
	olccmdtab[i].cmd_valid[mode])
      break;

  if (*olccmdtab[i].cmd == '\n')
    return olc_gen_edit(ch, command, argument);
  else
    (olccmdtab[i].cmdptr)(ch, olccmdtab[i].subcmd, argument);

#endif
  return 1;
}

#define OFFSET(field, type)	((long) &((type *) 0)->field)
#define OBJOFF(field)		OFFSET(field, obj_data)
#define TRIGOFF(field)		OFFSET(field, trig_data)
#define ROOMOFF(field)		OFFSET(field, struct room_data)
#define ZONEOFF(field)		OFFSET(field, struct zone_data)
#define SHOPOFF(field)		OFFSET(field, struct shop_data)
#define CHAROFF(field)		OFFSET(field, struct char_data)
#define CDPLAYEROFF(field)	OFFSET(field, struct char_player_data)
#define CDPOINTOFF(field)	OFFSET(field, struct char_point_data)
#define CDCHSPECOFF(field)	OFFSET(field, struct char_special_data)
#define CDCHABILOFF(field)	OFFSET(field, struct char_ability_data)
#define ODFLAGOFF(field)	OFFSET(field, struct obj_flag_data)
#define MOBSPECOFF(field)       OFFSET(field, struct mob_special_data)
#define EXITOFF(field)          OFFSET(field, struct room_direction_data)

struct oasis_gen_cmds {
  char *name;			/* command name				  */
  int menu;			// Oasis Menu referred to.
  byte mode;			/* olc mode of command			  */
  byte type;			/* type of variable (int/long...) EDT_x   */
  byte where;			/* where the offset is from, EDW_x	  */
  byte what;			/* kind of var (bv, int, string), EDK_x   */
  byte flags;			/* Type of flags to pass to the editor    */
  long displacement;		/* displacement of var from where	  */
  long min;			/* min value (integers only)		  */
  long max;			/* max value (int), leng (str), mask (bv) */
  const char **values;		/* list of values, terminated with '\n'   */
};


int oasis_gen_edit(struct char_data *ch, char *command, int menu, char *argument)
{
  long result;
  void *ptr, *bptr;
  int i, editmode = 0;
  size_t len;
  struct finish_olcstr_data *std;
  struct finish_olcshr_data *shd;
  struct finish_olcdoor_data *doordat;

  extern const char *trig_types[];
  extern const char *otrig_types[];
  extern const char *wtrig_types[];
  extern const char *ocmd_place_vector[];

  const struct oasis_gen_cmds cmds[] = {
    { "affectflags",	MEDIT_AFF_FLAGS, OLC_MEDIT,	0,	EDW_CDCHSPEC,		EDK_BITVECT,
      0, CDCHSPECOFF(saved.affected_by),	0, AFF_GROUP | AFF_CHARM,	(const char **)affected_bits
    },
    { "attacktype", MEDIT_ATTACK, OLC_MEDIT, EDT_INT, EDW_CDMOBSPEC, EDK_DIFFVALUE,
      0, MOBSPECOFF(attack_type),  FIRST_COMBAT_MESSAGE, TYPE_GARROTE,  (const char **) spells
    },
    { "bodytype", MEDIT_BODYTYPE, OLC_MEDIT, EDT_UBYTE, EDW_CDPLAYER, EDK_VALUE,
      0, CDPLAYEROFF(bodytype),  0, 0, (const char **) bodytypes
    },
    { "description", REDIT_DESC,	OLC_REDIT,	0,	EDW_BASE,	EDK_STRING,
      EDIT_APPEND, ROOMOFF(description),	0,	MAX_ROOM_DESC,		NULL
    },
    { "description", MEDIT_D_DESC,	OLC_MEDIT,	0,	EDW_CDPLAYER,	EDK_SSTRING,
      EDIT_APPEND, CDPLAYEROFF(description),	0,	MAX_MOB_DESC,		NULL
    },
    { "actdescription", OEDIT_ACTDESC,	OLC_OEDIT,	0,	EDW_BASE,	EDK_SSTRING,
      EDIT_APPEND, OBJOFF(action_description),	0,	MAX_OBJ_DESC,		NULL
    },
    { "exitdescription", REDIT_EXIT_DESCRIPTION,	OLC_REDIT,	0,	EDW_BASE,	EDK_DOORSTRING,
      EDIT_APPEND, EXITOFF(general_description),	0,	MAX_DOORDESC_LENGTH,		NULL
    },
    { "sector", REDIT_SECTOR,	OLC_REDIT, EDT_SHINT, EDW_BASE,	EDK_VALUE,
      0, ROOMOFF(sector_type), 0, 0,	(const char **) sector_types
    },
    // { "exitflags", REDIT_EXIT_DOORFLAGS, OLC_REDIT, 0, EDW_ROOMEXIT, EDK_BITVECT,
    //   0, EXITOFF(exit_info), 0, 0, (const char **) exit_bits
    // },
    { "extraflags", OEDIT_EXTRAS, OLC_OEDIT,	0,	EDW_ODFLAG,		EDK_BITVECT,
      0, ODFLAGOFF(extra_flags),	0, ITEM_OLC,	(const char **)extra_bits
    },
    { "flags",	REDIT_FLAGS, OLC_REDIT,	EDT_INT,	EDW_BASE,	EDK_BITVECT,
      0, ROOMOFF(room_flags),	0,	
      ROOM_HOUSE | ROOM_HOUSE_CRASH | ROOM_ATRIUM | ROOM_OLC | ROOM_BFS_MARK,
      (const char **)room_bits
    },

    { "alignment", MEDIT_ALIGNMENT, OLC_MEDIT, EDT_INT, EDW_CDCHSPEC, EDK_INTEGER,
      0, CDCHSPECOFF(saved.alignment), -1000, 1000, NULL
    },
    { "hitroll", MEDIT_HITROLL, OLC_MEDIT, EDT_SBYTE, EDW_CDPOINT, EDK_INTEGER,
      0, CDPOINTOFF(hitroll), 0, 50, NULL
    },
    { "damroll", MEDIT_DAMROLL, OLC_MEDIT, EDT_SBYTE, EDW_CDPOINT, EDK_INTEGER,
      0, CDPOINTOFF(damroll), 0, 50, NULL
    },
    { "disposition", MEDIT_DISPOSITION, OLC_MEDIT, EDT_SHINT, EDW_CDMOBSPEC, EDK_INTEGER,
      0, MOBSPECOFF(disposition), -1000, 1000, NULL
    },
    { "level", MEDIT_LEVEL, OLC_MEDIT, EDT_LONG, EDW_CDPLAYER, EDK_INTEGER,
      0, CDPLAYEROFF(level), 0, MAX_LEVEL, 0
    },
    { "gold", MEDIT_GOLD, OLC_MEDIT, EDT_INT, EDW_CDPOINT, EDK_INTEGER,
      0, CDPOINTOFF(gold), 0, 2000000000, NULL
    },
    { "maxriders", MEDIT_MAX_RIDERS, OLC_MEDIT, EDT_SBYTE, EDW_CDMOBSPEC, EDK_INTEGER,
      0, MOBSPECOFF(max_riders), 0, 9, NULL
    },
    { "numdice", MEDIT_NUM_HP_DICE, OLC_MEDIT, EDT_SHINT, EDW_CDPOINT, EDK_INTEGER,
      0, CDPOINTOFF(hit), 0, 30, NULL
    },
    { "sizedice", MEDIT_SIZE_HP_DICE, OLC_MEDIT, EDT_SHINT, EDW_CDPOINT, EDK_INTEGER,
      0, CDPOINTOFF(mana), 0, 1000, NULL
    },
    { "addhp", MEDIT_ADD_HP, OLC_MEDIT, EDT_SHINT, EDW_CDPOINT, EDK_INTEGER,
      0, CDPOINTOFF(move), 1, 30000, NULL
    },
    { "dr", MEDIT_AC, OLC_MEDIT, EDT_BYTE, EDW_CDPOINT, EDK_INTEGER,
      0, CDPOINTOFF(damage_resistance), 0, 50, NULL
    },
    { "expbonus", MEDIT_EXP, OLC_MEDIT, EDT_SHINT, EDW_CDCHSPEC, EDK_INTEGER,
      0, CDCHSPECOFF(saved.exp_modifier), -32000, 32000, NULL
    },
    { "totalpoints", MEDIT_TOTALPTS, OLC_MEDIT, EDT_LONG, EDW_CDPOINT, EDK_INTEGER,
      0, CDPOINTOFF(totalpts), -500, 500000, NULL
    },
    { "type", MEDIT_MOBTYPE, OLC_MEDIT, EDT_SBYTE, EDW_BASE, EDK_VALUE,
      0, CHAROFF(type), 0, 0, (const char **)mobile_types
    },

    { "npcflags", MEDIT_NPC_FLAGS, OLC_MEDIT,	  0,	  EDW_CDCHSPEC, 	  EDK_BITVECT,
      0, CDCHSPECOFF(saved.act),	0, MOB_SPEC | MOB_ISNPC,	(const char **)action_bits
    },
    { "wearflags", OEDIT_WEAR, OLC_OEDIT,	0,	EDW_ODFLAG,		EDK_BITVECT,
      0, ODFLAGOFF(wear_flags),	0,	0,	(const char **)wear_bits
    },
    { "objtype", OEDIT_TYPE, OLC_OEDIT, EDT_BYTE, EDW_ODFLAG, EDK_VALUE,
      0, ODFLAGOFF(type_flag),  0, 0, (const char **)item_types
    },
    { "objmaterial", OEDIT_MATERIAL, OLC_OEDIT, EDT_BYTE, EDW_ODFLAG, EDK_VALUE,
      0, ODFLAGOFF(material),  0, 0, (const char **)material_types
    },
    { "objpd", OEDIT_PD, OLC_OEDIT, EDT_BYTE, EDW_ODFLAG, EDK_INTEGER,
      0, ODFLAGOFF(passive_defense),  0, 120, NULL
    },
    { "objdr", OEDIT_DR, OLC_OEDIT, EDT_BYTE, EDW_ODFLAG, EDK_INTEGER,
      0, ODFLAGOFF(damage_resistance),  0, 120, NULL
    },
    { "objmaxhit", OEDIT_MAXHIT, OLC_OEDIT, EDT_SHINT, EDW_ODFLAG, EDK_INTEGER,
      0, ODFLAGOFF(max_hit),  0, 32000, NULL
    },
    { "objhit", OEDIT_HIT, OLC_OEDIT, EDT_SHINT, EDW_ODFLAG, EDK_INTEGER,
      0, ODFLAGOFF(hit),  0, 32000, NULL
    },
    { "objmaxmana", OEDIT_MAXMANA, OLC_OEDIT, EDT_SHINT, EDW_ODFLAG, EDK_INTEGER,
      0, ODFLAGOFF(max_mana),  0, 32000, NULL
    },
    { "objmana", OEDIT_MANA, OLC_OEDIT, EDT_SHINT, EDW_ODFLAG, EDK_INTEGER,
      0, ODFLAGOFF(mana),  0, 32000, NULL
    },
    { "position", MEDIT_POS, OLC_MEDIT, EDT_BYTE, EDW_CDCHSPEC, EDK_INTEGER,
      0, CDCHSPECOFF(position), 0, 0, (const char **) position_types
    },
    { "defpos", MEDIT_DEFAULT_POS, OLC_MEDIT, EDT_BYTE, EDW_CDMOBSPEC, EDK_INTEGER,
      0, MOBSPECOFF(default_pos), 0, 0, (const char **) position_types
    },
    { "shopflags", SEDIT_SHOP_FLAGS, OLC_SEDIT, 0, EDW_BASE, EDK_BITVECT,
      0, SHOPOFF(bitvector), 0, 0,	(const char **)shop_bits
    },
    { "tradeflags", SEDIT_NOTRADE, OLC_SEDIT, 0, EDW_BASE, EDK_BITVECT,
      0, SHOPOFF(with_who), 0, 0,	(const char **)trade_letters
    },
    { "value1", OEDIT_VALUE_1, OLC_OEDIT, EDT_INT, EDW_ODFLAG, EDK_OBJVAL, 0, 
      0,	0, 0, NULL
    },
    { "value2", OEDIT_VALUE_2, OLC_OEDIT, EDT_INT, EDW_ODFLAG, EDK_OBJVAL, 0, 
      0,	0, 0, NULL
    },
    { "value3", OEDIT_VALUE_3, OLC_OEDIT, EDT_INT, EDW_ODFLAG, EDK_OBJVAL, 0, 
      0,	0, 0, NULL
    },
    { "value4", OEDIT_VALUE_4, OLC_OEDIT, EDT_INT, EDW_ODFLAG, EDK_OBJVAL, 0, 
      0,	0, 0, NULL
    },
    { "value5", OEDIT_VALUE_5, OLC_OEDIT, EDT_INT, EDW_ODFLAG, EDK_OBJVAL, 0, 
      0,	0, 0, NULL
    },
    { "value6", OEDIT_VALUE_6, OLC_OEDIT, EDT_INT, EDW_ODFLAG, EDK_OBJVAL, 0, 
      0,	0, 0, NULL
    },
    { "value7", OEDIT_VALUE_7, OLC_OEDIT, EDT_INT, EDW_ODFLAG, EDK_OBJVAL, 0, 
      0,	0, 0, NULL
    },
    { "value8", OEDIT_VALUE_8, OLC_OEDIT, EDT_INT, EDW_ODFLAG, EDK_OBJVAL, 0, 
      0,	0, 0, NULL
    },
    { "race", MEDIT_RACE, OLC_MEDIT, EDT_BYTE, EDW_CDPLAYER, EDK_VALUE,
      0, CDPLAYEROFF(race),  0, NUM_RACES - 1,  (const char **) race_types
    },
    { "class", MEDIT_CLASS, OLC_MEDIT, EDT_BYTE, EDW_CDPLAYER, EDK_VALUE,
      0, CDPLAYEROFF(clss),  0, CLASS_PSIONICIST,  (const char **) class_types
    },
    { "sex", MEDIT_SEX, OLC_MEDIT, EDT_BYTE, EDW_CDPLAYER, EDK_VALUE,
      0, CDPLAYEROFF(sex), 0, 2, (const char **) genders
    },
    { "strength", MEDIT_STR, OLC_MEDIT, EDT_SBYTE, EDW_CDCHABIL, EDK_INTEGER,
      0, CDCHABILOFF(str), 1, 80, NULL
    },
    { "intelligence", MEDIT_INT, OLC_MEDIT, EDT_SBYTE, EDW_CDCHABIL, EDK_INTEGER,
      0, CDCHABILOFF(intel), 1, 80, NULL
    },
    { "wisdom", MEDIT_WIS, OLC_MEDIT, EDT_SBYTE, EDW_CDCHABIL, EDK_INTEGER,
      0, CDCHABILOFF(wis), 1, 80, NULL
    },
    { "dexterity", MEDIT_DEX, OLC_MEDIT, EDT_SBYTE, EDW_CDCHABIL, EDK_INTEGER,
      0, CDCHABILOFF(dex), 1, 80, NULL
    },
    { "constitution", MEDIT_CON, OLC_MEDIT, EDT_SBYTE, EDW_CDCHABIL, EDK_INTEGER,
      0, CDCHABILOFF(con), 1, 80, NULL
    },
    { "charisma", MEDIT_CHA, OLC_MEDIT, EDT_SBYTE, EDW_CDCHABIL, EDK_INTEGER,
      0, CDCHABILOFF(cha), 1, 80, NULL
    },

    { "name", TRIGEDIT_NAME, OLC_TRIGEDIT, 0, EDW_BASE, EDK_SSTRING,
      0, TRIGOFF(name),  0, 40,  NULL
    },
    { "argument", TRIGEDIT_ARGUMENT, OLC_TRIGEDIT, 0, EDW_BASE, EDK_SSTRING,
      0, TRIGOFF(arglist),  0, 120,  NULL
    },
    { "mtypes", TRIGEDIT_MOB_TYPES, OLC_TRIGEDIT, EDT_LONG, EDW_BASE, EDK_BITVECT,
      0, TRIGOFF(trigger_type_mob), 0, 0, (const char **) trig_types
    },
    { "otypes", TRIGEDIT_OBJ_TYPES, OLC_TRIGEDIT, EDT_LONG, EDW_BASE, EDK_BITVECT,
      0, TRIGOFF(trigger_type_obj), 0, 0, (const char **) otrig_types
    },
    { "wtypes", TRIGEDIT_WLD_TYPES, OLC_TRIGEDIT, EDT_LONG, EDW_BASE, EDK_BITVECT,
      0, TRIGOFF(trigger_type_wld), 0, 0, (const char **) wtrig_types
    },
    { "numarg", TRIGEDIT_NARG, OLC_TRIGEDIT, EDT_INT, EDW_BASE, EDK_INTEGER,
      0, TRIGOFF(narg), 0, 0, NULL
    },
    { "numargbit", TRIGEDIT_NARG_BITV, OLC_TRIGEDIT, EDT_INT, EDW_BASE, EDK_BITVECT,
      0, TRIGOFF(narg), 0, 0, (const char **) ocmd_place_vector
    },
    
    { "\n", -1, 0, 0, 0, 0, 0, 0, 0, 0, NULL }
  };

  editmode = GET_OLC_MODE(ch);

  if (*command) {
    for (i = 0, len = strlen(command); *cmds[i].name != '\n'; ++i)
      if (editmode == cmds[i].mode &&
	  !strncmp(command, cmds[i].name, len))
        break;
  
    if (*cmds[i].name == '\n')
      return 0;
  } else {
    for (i = 0; cmds[i].menu >= 0; ++i)
      if (editmode == cmds[i].mode && (cmds[i].menu == menu))
        break;
  
    if (cmds[i].menu == -1) {
      log("Invalid menu number in oasis_gen_edit.  Tell the imps!");
      return 0;
  }
  }

  /* find memory location */
  if ((bptr = get_current_olc_ptr(ch, OLC_NUM(ch), editmode)) == NULL) {
    send_to_char("No target found to edit.\r\n", ch);
    return 1;
  }

  ptr = (void *)((long) bptr + cmds[i].displacement);

  switch (cmds[i].where) {
  case EDW_BASE: break;
  case EDW_CDPLAYER:
    ptr = (void *)((long) ptr + OFFSET(player, char_data));
    break;
  case EDW_CDPOINT:
    ptr = (void *)((long) ptr + OFFSET(points, char_data));
    break;
  case EDW_CDCHSPEC:
    ptr = (void *)((long) ptr + OFFSET(char_specials, char_data));
    break;
  case EDW_CDCHABIL:
    ptr = (void *)((long) ptr + OFFSET(aff_abils, char_data));
    break;
  case EDW_ODFLAG:
    ptr = (void *)((long) ptr + OFFSET(obj_flags, obj_data));
    break;
  case EDW_CDMOBSPEC:
    ptr = (void *)((long) ptr + OFFSET(mob_specials, char_data));
    break;

  default:
    break;
  }

  /* parse arg and get value */

  skip_spaces(&argument);

  switch (cmds[i].what) {
  case EDK_INTEGER:
    if (!olc_edit_integer(ch, &result, argument, cmds[i].name,
			  cmds[i].min, cmds[i].max))
      return 0;
    break;
  case EDK_BITVECT:
    if (olc_edit_bitvector(ch, &result, argument, *((long *) ptr),
			    cmds[i].name, cmds[i].values, cmds[i].max) != 2)
      return 0;
    break;
  case EDK_VALUE:
    if (!olc_edit_value(ch, &result, argument, cmds[i].name, cmds[i].min, cmds[i].max, cmds[i].values))
      return 0;
    break;
  case EDK_SPELLVALUE:
    if (!olc_edit_value(ch, &result, argument, cmds[i].name, cmds[i].min, cmds[i].max, cmds[i].values))
      return 0;
    break;
  case EDK_DIFFVALUE:
    if (!olc_edit_value(ch, &result, argument, cmds[i].name, cmds[i].min, cmds[i].max, cmds[i].values, TRUE))
      return 0;
    break;
  case EDK_OBJVAL:
    return (olc_edit_objval(ch, argument, cmds[i].name));
  case EDK_STRING:
    CREATE(std, struct finish_olcstr_data, 1);
    std->string = (char **) ptr;
    std->mode = editmode;
    std->target = OLC_NUM(ch);
    std->ptr = bptr;

    olc_edit_string(ch, (char **) ptr, argument, cmds[i].name,
		    finish_olc_string, std, cmds[i].max, (byte) cmds[i].flags);
    return 1;
    break;
  case EDK_SSTRING:
    CREATE(shd, struct finish_olcshr_data, 1);
    shd->string = (sstring **) ptr;
    shd->mode = editmode;
    shd->target = OLC_NUM(ch);
    shd->ptr = bptr;

    olc_edit_sstring(ch, (sstring **) ptr, argument, cmds[i].name,
		     finish_olc_sstring, shd, cmds[i].max, (byte) cmds[i].flags);
    return 1;
    break;
  case EDK_DOORSTRING:
    CREATE(doordat, struct finish_olcdoor_data, 1);
    doordat->string = &OLC_EXIT(ch)->general_description;
    doordat->target = OLC_NUM(ch);
    doordat->door = OLC_VAL(ch);
    doordat->ptr = (void *) OLC_ROOM(ch)->dir_option[OLC_VAL(ch)];

    olc_edit_string(ch, doordat->string, argument, cmds[i].name,
		    finish_olc_door, doordat, cmds[i].max, (byte) cmds[i].flags);
    return 1;
    break;
  }

  /* assign value */
  switch (cmds[i].type) {
  case EDT_LONG:	*(long *) ptr = (long) result;		break;
  case EDT_INT:		*(int *) ptr = (int) result;		break;
  case EDT_SHINT:	*(sh_int *) ptr = (sh_int) result;	break;
  case EDT_BYTE:	*(byte *) ptr = (byte) result;		break;
  case EDT_SBYTE:	*(sbyte *) ptr = (sbyte) result;		break;
  case EDT_UBYTE:	*(ubyte *) ptr = (ubyte) result;		break;
  }

  return 1;
}


#ifdef GOT_RID_OF_IT
/* OLC Command Functions */

OLCCMD(olc_add)
{
  char name[MAX_INPUT_LENGTH], buf[500];
  int zn;

  if (GET_TRUST(ch) < TRUST_IMPL) {
    send_to_char("Sorry, you can not change the access control list for a zone.\r\n", ch);
    return;
  }

  one_argument(argument, name);

  if (!*name)
    send_to_char("Usage:  add [ all | <person> ]\r\n", ch);

  else if (str_cmp(name, "all") && player_index->get_id_by_name(name) < 0)
    send_to_char("That person does not exist.\r\n", ch);

  else if (get_olc_ptr(OLC_NUM(ch), OLC_ZEDIT) == NULL)
    send_to_char("That zone does not exist.\r\n", ch);

  else {
    zn = OLC_NUM(ch);

    if (!str_cmp(name, "all")) {
      if (zone_table[zn].permissions)
	FREE(zone_table[zn].permissions);
      
      zone_table[zn].permissions = str_dup("all");

      sprintf(buf, "Anyone can now edit zone %d.\r\n", zone_table[zn].number);
      send_to_char(buf, ch);
    }

    else if (zone_table[zn].permissions &&
	     (str_str(zone_table[zn].permissions, name) ||
	     !str_cmp(zone_table[zn].permissions, "all")))
      send_to_char("That person already has permission to edit that zone.\r\n",
		   ch);

    else {
      if (zone_table[zn].permissions) {
	sprintf(buf, "%s %s", zone_table[zn].permissions,  name);
	FREE(zone_table[zn].permissions);
	zone_table[zn].permissions = str_dup(buf);
      }

      else
	zone_table[zn].permissions = str_dup(name);

      sprintf(buf, "%s added to access list for zone %d.\r\n", name,
	      zone_table[zn].number);
      send_to_char(CAP(buf), ch);
    }
  }
}


OLCCMD(olc_remove)
{
  char name[MAX_INPUT_LENGTH], buf[500], *s;
  int zn, len, all;
  
  if (GET_TRUST(ch) < TRUST_IMPL) {
    send_to_char("Sorry, you can not change the access control list for a zone.\r\n", ch);
    return;
  }

  one_argument(argument, name);

  len = strlen(name);
  zn = OLC_NUM(ch);
  all = !str_cmp(name, "all");

  if (!*name)
    send_to_char("Usage:  remove [ all | <person> ]\r\n", ch);

  else if (get_olc_ptr(zn, OLC_ZEDIT) == NULL)
    send_to_char("That zone does not exist.\r\n", ch);

  else if (!all && (!zone_table[zn].permissions ||
	   (s = str_str(zone_table[zn].permissions, name)) == NULL))
    send_to_char("That person is not on the access list for that zone.\r\n", ch);

  else if (!all && ((zone_table[zn].permissions != s && !isspace(*(s - 1))) ||
	   (s[len] && !isspace(s[len]))))
    send_to_char("That person is not on the access list for that zone.\r\n", ch);
  
  else  if (all) {
      if (zone_table[zn].permissions)
	FREE(zone_table[zn].permissions);
      zone_table[zn].permissions = NULL;

      sprintf(buf, "All OLC access removed from zone %d.\r\n",
	      zone_table[zn].number);
      send_to_char(buf, ch);
  }

  else {
    *s = '\0';
    strcpy(buf, zone_table[zn].permissions);
    if (s[len])
      strcat(buf, s + len + 1);

    FREE(zone_table[zn].permissions);
    if (!*buf)
      zone_table[zn].permissions = NULL;
    else
      zone_table[zn].permissions = str_dup(buf);

    sprintf(buf, "%s removed from access list for zone %d.\r\n", name,
	    zone_table[zn].number);
    send_to_char(CAP(buf), ch);
  }
}
#endif


OLCCMD(olc_create)
{
  int vnum, i = -1;
  char buf[256];

  skip_spaces(&argument);

  if (!*argument) {
    send_to_char("You must give a vnum to create.\r\n", ch);
    return;
  }

  vnum = atoi(argument);

  if (vnum < 0 || vnum > SHRT_MAX) {
    send_to_char("That is not a valid number!\r\n", ch);
    return;
  }
    
  switch (subcmd) {
  case OLC_MEDIT: i = mob_index[vnum] ? 0 : -1; break;
  case OLC_OEDIT: i = obj_index[vnum] ? 0 : -1; break;
  case OLC_REDIT: i = real_room(vnum);          break;
  case OLC_ZEDIT: i = real_zone(vnum);          break;
  }

  if (i >= 0) {
    send_to_char("That vnum already exists.\r\n", ch);
    return;
  }

  switch (subcmd) {
  case OLC_MEDIT:
  case OLC_OEDIT:
  case OLC_REDIT:
    if (!check_zone_permission(ch, find_owner_zone(vnum))) {
      send_to_char("Sorry, you can't build there.\r\n", ch);
      return;
    }
    break;

  case OLC_ZEDIT:
    break;
  }

  /* all checks done, create the thing */
  switch (subcmd) {
  case OLC_MEDIT:
    create_mob_proto(vnum);
    break;

  case OLC_OEDIT:
    create_obj_proto(vnum);
    break;

  case OLC_ZEDIT:
    /* only imps can create new zones */
    if (GET_TRUST(ch) < TRUST_IMPL) {
      send_to_char("Ask an implementor to create a zone for you.\r\n", ch);
      return;
    }
    if (vnum > (SHRT_MAX / 100)) {
      send_to_char("That is not a valid number!\r\n", ch);
      return;
    }

    create_zone(vnum);
    break;

  case OLC_REDIT:
    create_room(vnum);
    break;
  }

  send_to_charf(ch, "%s %d created.\r\n", olc_modes[subcmd], vnum);
  olc_add_to_save_list(vnum, subcmd);

  // OLC_NUM(ch) = vnum; /* switch target to new thing */
}


#ifdef GOT_RID_OF_IT
OLCCMD(olc_delete)
{
  char_data *mob, *next_mob;
  obj_data *obj, *next_obj;
  void *ptr;
  int targ;
  byte delete_other = FALSE;

  skip_spaces(&argument);

  /* find what to delete */

  if (*argument) {
    if ((targ = find_olc_target(ch, argument, GET_OLC_MODE(ch))) < 0) {
      send_to_char("You can't delete what doesn't exist!\r\n", ch);
      return;
    }
    else if (!check_olc_permission(ch, targ, GET_OLC_MODE(ch))) {
      send_to_char("You don't have permission to delete that!\r\n", ch);
      return;
    }
    delete_other = TRUE;
  } else
    targ = OLC_NUM(ch);

  if ((ptr = get_olc_ptr(targ, GET_OLC_MODE(ch))) == NULL) {
    send_to_char("What do you wish to delete?\r\n", ch);
    return;
  }


  /* check if deleteable */

  if (targ == 0) {
    /* don't screw with system stuff! */
    send_to_char("Sorry, you can't delete that!\r\n", ch);
    return;
  }

  /*    and the special cases... */
  if (GET_OLC_MODE(ch) == OLC_REDIT &&
      (targ == r_mortal_start_room ||
       targ == r_immort_start_room ||
       targ == r_frozen_start_room)) {
    send_to_char("Sorry, you can't delete that room!\r\n", ch);
    return;
  }

  
  /* clean up whatever mess necessary before removing */

  switch (GET_OLC_MODE(ch)) {
  case OLC_MEDIT:
  case OLC_OEDIT:
    break;

  case OLC_REDIT:
    /* everything to void! */

    act("$n hurls you into the void!\r\n", TRUE, ch, NULL, NULL, TO_ROOM);
    send_to_char("With a sweep of your hand, you hurl the room into oblivion.\r\n", ch);

    for (mob = ((struct room_data *) ptr)->people; mob; mob = next_mob) {
      next_mob = mob->next_in_room;

      char_from_room(mob);
      char_to_room(mob, 0);
    }

    for (obj = ((struct room_data *) ptr)->contents; obj; obj = next_obj) {
      next_obj = obj->next_content;

      obj_from_room(obj);
      obj_to_room(obj, 0);
    }
    break;

  case OLC_ZEDIT:
    break;
  }


  /* get rid of of the thing */

  switch (GET_OLC_MODE(ch)) {
  case OLC_MEDIT:
    mob = (char_data *) ptr;
    sprintf(buf, "Mobile #%d, '%s' deleted.\r\n",
	    GET_MOB_NUM(mob), GET_SHORT(mob));
    send_to_char(buf, ch);
    mob_index.remove(GET_MOB_NUM((char_data *) ptr));
    break;

  case OLC_OEDIT:
    obj = (obj_data *) ptr;
    sprintf(buf, "Object #%d, '%s' deleted.\r\n",
	    GET_OBJ_NUM(obj), ss_data(obj->short_description));
    send_to_char(buf, ch);
    obj_index.remove(GET_OBJ_NUM((obj_data *) ptr));
    break;

  case OLC_REDIT:
    sprintf(buf, "Room #%d, '%s' deleted.\r\n",
	    ((struct room_data *) ptr)->number,
	    ((struct room_data *) ptr)->name);
    send_to_char(buf, ch);
    delete (room_data *) ptr;
    world[targ] = NULL;
    break;

  case OLC_ZEDIT:
    sprintf(buf, "Zone #%d, '%s' deleted.\r\n",
	    ((struct zone_data *) ptr)->number,
	    ((struct zone_data *) ptr)->name);
    send_to_char(buf, ch);
    delete_zone(real_zone(((struct zone_data *) ptr)->number));
    break;
  }    

  if (!delete_other)
    OLC_NUM(ch) = -1;
}


OLCCMD(olc_done)
{
  GET_OLC_MODE(ch) = 0;
  OLC_NUM(ch) = 0;

  send_to_char(OK, ch);
}


#define DOOR_DELETE	1
#define DOOR_DIG	2
#define DOOR_LINK	3
#define DOOR_ROOM	4
#define DOOR_KEY	5
#define DOOR_NAME	6
#define DOOR_DESCR	7
#define DOOR_FLAGS	8
#define DOOR_HELP	9

struct door_cmdinfo {
  char *name;
  int  cmd;
  char *arg;
  byte flags;
  char *help;
};

OLCCMD(olc_exit)
{
  int door = subcmd, i, cmd, len, n;
  long flags, newflags;
  struct room_data *room;
  char command[MAX_INPUT_LENGTH];
  char buf[MAX_STRING_LENGTH], buf2[MAX_INPUT_LENGTH];
  struct finish_olcdoor_data *ed;

  const struct door_cmdinfo cmds[] = {
    { "dig"        , DOOR_DIG   , "<room>"      ,
      0, 
      "Creates room and links opposite exits."
    },
    { "link"       , DOOR_LINK  , "<room>"      ,
      0,
      "Links opposite exits of edited room and target room."
    },
    { "room"       , DOOR_ROOM  , "<room>"      ,
      0,
      "Sets the exit to go to the target room."
    },
    { "key"        , DOOR_KEY   , "[<key>|none]",
      0,
      "Sets the key to the exit."
    },
    { "name"       , DOOR_NAME  , "<name>"      ,
      EDIT_ONELINE,
      "Sets the door name."
    },
    { "description", DOOR_DESCR , "<desc>"      ,
      EDIT_APPEND,
      "Sets the exit description."
    },
    { "flags"      , DOOR_FLAGS , "<flags>"     ,
      0,
      "Changes flags on exit."
    },
    { "door"       , DOOR_FLAGS , "<flags>"     ,
      0,
      "Changes flags on exit."
    },
    { "help"       , DOOR_HELP  , ""            ,
      0,
      "Gives help on available commands."
    },
    { "delete"     , DOOR_DELETE, ""            ,
      0,
      "Deletes this exit." 
    },

    { "\n", 0, "", 0, "" },
  };

  ACMD(do_move);

  argument = any_one_arg(argument, command);
  if (!*command || GET_OLC_MODE(ch) != OLC_REDIT) {
    do_move(ch, command, argument, door + 1, 0);
    return;
  }

  if ((room = (struct room_data *) get_olc_ptr(OLC_NUM(ch),
					       GET_OLC_MODE(ch))) == NULL) {
    send_to_char("There must be a room to edit!\r\n", ch);
    return;
  }

  len = strlen(command);
  for (i = 0; *cmds[i].name != '\n'; i++)
    if (!strncmp(command, cmds[i].name, len))
      break;

  if (*cmds[i].name == '\n') {
    sprintf(buf, "Unknown exit command.  \"%s help\" for more information.\r\n",
	    dirs[door]);
    send_to_char(buf, ch);
    return;
  }

  skip_spaces(&argument);

  /* help on individual commands */
  if (!str_cmp("help", argument)) {
    sprintf(buf2, "%s %s", cmds[i].name, cmds[i].arg);
    sprintf(buf, "%-19s %s\r\n", buf2, cmds[i].help);
    send_to_char(buf, ch);
    return;
  }

  cmd = cmds[i].cmd;

  /* complete help command */
  if (cmd == DOOR_HELP) {
    strcpy(buf, "Valid exit commands are:\r\n");
    for (i = 0; *cmds[i].name != '\n'; i++) {
      sprintf(buf2, "%s %s", cmds[i].name, cmds[i].arg);
      sprintf(buf, "%s  %-19s %s\r\n", buf, buf2, cmds[i].help);
    }

    page_string(ch->desc, buf, TRUE);
    return;
  }

  /* complete delete command */
  if (cmd == DOOR_DELETE) {
    if (!room->dir_option[door])
      send_to_char("There is no exit there!\r\n", ch);
    else {
      free_exit(room->dir_option[door]);
      room->dir_option[door] = NULL;
      send_to_char("Exit deleted.\r\n", ch);
    }

    return;
  }

  /* Error checking for remainder of commands */

  if (!*argument) {
    if (cmd == DOOR_KEY) {
      send_to_char("Key number or 'non' expected.\r\n", ch);
      return;
    } else if (cmd == DOOR_DIG || cmd == DOOR_LINK || cmd == DOOR_ROOM) {
      send_to_char("Target room number expected.\r\n", ch);
      return;
    }
  }

  n = atoi(argument);

  if (cmd == DOOR_DIG) {
    if (n < 0 || n > SHRT_MAX) {
      send_to_char("That is not a valid room.\r\n", ch);
      return;
    }
    else if (real_room(n) >= 0) {
      send_to_char("That room already exists.\r\n", ch);
      return;
    }
    else if (!check_zone_permission(ch, find_owner_zone(n))) {
      sprintf(buf, "You do not have permission to modify room %d.\r\n", n);
      send_to_char(buf, ch);
      return;
    }

    create_room(n);
  }

  else if (cmd == DOOR_LINK) {
    if (real_room(n) < 0) {
      send_to_char("That room does not exist.\r\n", ch);
      return;
    }
    else if (!check_olc_permission(ch, real_room(n), OLC_REDIT)) {
      sprintf(buf, "You do not have permission to modify room %d.\r\n", n);
      send_to_char(buf, ch);
      return;
    }
  }

  else if (cmd == DOOR_FLAGS) {
    flags = 0;
    if (room->dir_option[door])
      flags = room->dir_option[door]->exit_info;
    if (!olc_edit_bitvector(ch, &newflags, argument, flags, "exit flags",
			     exit_bits, 0))
      return;
  }


  /* create the exit (for all commands) */
  if (!room->dir_option[door]) {
    CREATE(room->dir_option[door], struct room_direction_data, 1);
    room->dir_option[door]->to_room = -1;
  }

  /* finish assignments */

  if (cmd == DOOR_DIG || cmd == DOOR_LINK || cmd == DOOR_ROOM) {
    room->dir_option[door]->to_room = n;

    if (cmd == DOOR_DIG || cmd == DOOR_LINK) {
      if (!world[n]->dir_option[rev_dir[door]])
	CREATE(world[n]->dir_option[rev_dir[door]],
	       struct room_direction_data, 1);
      
      world[n]->dir_option[rev_dir[door]]->to_room = room->number;
    }

    switch (cmd) {
    case DOOR_DIG:
      sprintf(buf, "Room %d created; %s exit linked with %s of room %d.\r\n",
	      n, dirs[rev_dir[door]], dirs[door], room->number);
      break;
    case DOOR_LINK:
      sprintf(buf, "%s exit of room %d linked with %s exit of room %d.\r\n",
	      dirs[door], room->number, dirs[rev_dir[door]], n);
      break;
    case DOOR_ROOM:
      sprintf(buf, "%s exit of room %d connected to room %d.\r\n",
	      dirs[door], room->number, n);
      break;
    }
    send_to_char(CAP(buf), ch);
  }

  else if (cmd == DOOR_KEY) {
    one_argument(argument, buf);

    if (!str_cmp(buf, "none"))
      n = -1;
    
    room->dir_option[door]->key = n;

    if (n == -1)
      sprintf(buf, "%s exit key set to none.\r\n", dirs[door]);
    else
      sprintf(buf, "%s exit key set to %d.\r\n", dirs[door], n);
    send_to_char(CAP(buf), ch);
  }

  else if (cmd == DOOR_FLAGS)
    room->dir_option[door]->exit_info = (sh_int) newflags;

  else if (cmd == DOOR_NAME || cmd == DOOR_DESCR) {
    CREATE(ed, struct finish_olcdoor_data, 1);
    ed->target = OLC_NUM(ch);
    ed->door = door;
    ed->ptr = (void *) room->dir_option[door];

    if (cmd == DOOR_NAME) {
      ed->string = &room->dir_option[door]->keyword;
      olc_edit_string(ch, ed->string, argument, "exit name", finish_olc_door,
		      ed, MAX_DOORNAME_LENGTH, (int) cmds[i].flags);
    } else {
      ed->string = &room->dir_option[door]->general_description;
      olc_edit_string(ch, ed->string, argument, "exit description",
		      finish_olc_door, ed, MAX_DOORDESC_LENGTH, (int) cmds[i].flags);
    }
  }
}
#endif


void edit_extradesc(char_data *ch, struct extra_descr_data *ed)
{
  struct finish_olcextr_data *fd = NULL;

  CREATE(fd, struct finish_olcextr_data, 1);
  fd->string = &ed->description;
  fd->target = OLC_NUM(ch);
  fd->mode = GET_OLC_MODE(ch);
  fd->ptr = ed;

  olc_edit_sstring(ch, fd->string, "+", "extra description", 
        	  finish_olc_exdesc, fd, MAX_EXDESC_LENGTH, 0);
}


#ifdef GOT_RID_OF_IT
OLCCMD(olc_extradesc)
{
  struct extra_descr_data **extra, *ed = NULL, **tmp;
  char command[MAX_INPUT_LENGTH], buf[MAX_INPUT_LENGTH];
  int editor = FALSE;
  void *ptr;


  if ((ptr = get_olc_ptr(OLC_NUM(ch), GET_OLC_MODE(ch))) == NULL) {
    send_to_char("No target found to edit.\r\n", ch);
    return;
  }

  switch (GET_OLC_MODE(ch)) {
  case OLC_OEDIT:
    extra = &((struct obj_data *) ptr)->ex_description;
    break;
  case OLC_REDIT:
    extra = &((struct room_data *) ptr)->ex_description;
    break;
  default:
    return;
  }

  argument = any_one_arg(argument, command);
  skip_spaces(&argument);

  if (!*command) {
    send_to_char("Valid commands for editing extra descriptions are:\r\n"
"  extradesc add <keywords>      Add a new extra description.\r\n"
"  extradesc delete <keywords>   Delete an existing extra description.\r\n"
"  extradesc edit <keywords>     Edits desc of existing extra desciption.\r\n",
		 ch);
    return;
  }

  if (!*argument) {
    send_to_char("You must give a keyword for the extra description.\r\n", ch);
    return;
  }

  if (is_abbrev(command, "add")) {
    CREATE(ed, struct extra_descr_data, 1);
    ed->keyword = ss_create(argument);
    ed->next = *extra;
    *extra = ed;

    editor = TRUE;
  }

  else if (is_abbrev(command, "delete")) {
    if ((ed = find_extra_desc(argument, *extra)) == NULL) {
      send_to_char("Extra description not found.\r\n", ch);
      return;
    }

    /* extra descr must be on the list at this point */
   
    for (tmp = extra; *tmp != ed; tmp = &(*tmp)->next);

    *tmp = (*tmp)->next;
    sprintf(buf, "Extra description '%s' deleted.\r\n", ss_data(ed->keyword));
    send_to_char(buf, ch);

    free_exdesc(ed);
  }

  else if (is_abbrev(command, "edit")) {
    if ((ed = find_extra_desc(argument, *extra)) == NULL) {
      send_to_char("Extra description not found.\r\n", ch);
      return;
    }

    /* extra descr must be on the list at this point */
    
    for (tmp = extra; *tmp != ed; tmp = &(*tmp)->next);

    editor = TRUE;
  }

  else {
    send_to_char("Unknown extradesc command.\r\n", ch);
    return;
  }

  if (editor) {
    edit_extradesc(ch, ed);
  }
}


OLCCMD(olc_help)
{
  ACMD(do_help);
  const char *helps[] = { "", "medit", "oedit", "redit", "zedit" };
  char buf[MAX_INPUT_LENGTH];

  strcpy(buf, helps[GET_OLC_MODE(ch)]);
  do_help(ch, "wizhelp", buf, 0, SCMD_WHELP);
}


OLCCMD(olc_look)
{
  ACMD(do_look);

  do_look(ch, "look", argument, 0, SCMD_LOOK);
}


OLCCMD(olc_show)
{
  void *ptr = NULL;
  char buf[256];

  if ((ptr = get_olc_ptr(OLC_NUM(ch), GET_OLC_MODE(ch))) == NULL) {
    send_to_char("No target found to edit.\r\n", ch);
    return;
  }

  switch (GET_OLC_MODE(ch)) {

  case OLC_ZEDIT:
    do_show_zone(ch, real_zone(((struct zone_data *) ptr)->number));
    break;

  case OLC_OEDIT:
    do_stat_object(ch, (struct obj_data *) ptr, STAT_OLC);
    break;

  case OLC_REDIT:
    do_stat_room(ch, (struct room_data *) ptr, STAT_OLC);
    break;

  case OLC_MEDIT:
    do_stat_character(ch, (struct char_data *) ptr, STAT_OLC);
    break;

  default: 
    send_to_char("Sorry, not implemented yet.\n\r", ch);
    break;
  } 
}


OLCCMD(olc_special)
{
    int n, i;
    char buf[MAX_STRING_LENGTH], **farg = NULL;
    struct spec_type *spec_table = NULL;
    struct finish_olcstr_data *fd;
    SpecProc proc;
    void *ptr = NULL;
    index_data *index;

    n = OLC_NUM(ch);
    switch (GET_OLC_MODE(ch)) {
    case OLC_MEDIT:
	n = mob_index[n] ? n : -1;
	spec_table = (struct spec_type *) spec_mob_table;
	break;
    case OLC_OEDIT:
	n = obj_index[n] ? n : -1;
	spec_table = (struct spec_type *) spec_obj_table;
	break;
    case OLC_REDIT:
	n = real_room(n);		/* check room exists */
	spec_table = (struct spec_type *) spec_room_table;
	break;
    }

    argument = any_one_arg(argument, buf);
    if (!*buf) {
	strcpy(buf, "Usage:  special [<name>| none ] [<argument>]\r\n"
	       "Available special procedures are:\r\n");
	for (i = 0; spec_table[i].name; i++)
	    sprintf(buf, "%s  %-20s%s", buf, spec_table[i].name,
		    ((i % 3) == 2) ? "\r\n" : "");

	if ((i % 3) != 0)
	    strcat(buf, "\r\n");

	page_string(ch->desc, buf, TRUE);
	return;
    }

    if (n < 0) {
	send_to_char("No target found.\r\n", ch);
	return;
    }

    else if (!str_cmp(buf, "none"))
	proc = NULL;

    else if ((proc = spec_lookup(buf, spec_table)) == NULL) {
	send_to_char("Special procedure not found.\r\n", ch);
	return;
    }

    switch (GET_OLC_MODE(ch)) {
    case OLC_MEDIT:
	index = mob_index[n];
	index->func = proc;
	farg = &index->farg;
	ptr = index;
	if (proc)
	    SET_BIT(MOB_FLAGS((char_data *) index->proto), MOB_ISNPC);
	else
	    REMOVE_BIT(MOB_FLAGS((char_data *) index->proto), MOB_ISNPC);
	break;
    case OLC_OEDIT:
	index = obj_index[n];
	index->func = proc;
	farg = &index->farg;
	ptr = index;
	break;
    case OLC_REDIT:
	world[n]->func = proc;
	farg = &world[n]->farg;
	ptr = world[n];
	break;
    }

    if (proc == NULL) {
	if (*farg)
	    FREE(*farg);
	send_to_char("Special procedure set to none.\r\n", ch);
    }

    else {
	sprintf(buf, "Special procedure set to %s.\r\n",
		spec_name(proc, spec_table));
	send_to_char(buf, ch);
    
	CREATE(fd, struct finish_olcstr_data, 1);
	fd->string = farg;
	fd->mode = GET_OLC_MODE(ch);
	fd->target = OLC_NUM(ch);
	fd->ptr = ptr;
    
	skip_spaces(&argument);
	olc_edit_string(ch, farg, argument, "special procedure argument", 
			finish_olc_farg, fd, MAX_FARG_LENGTH, EDIT_ONELINE);
    }
}

OLCCMD(olc_trigger)
{
    int n, loc, tn = 0;
    char arg1[MAX_INPUT_LENGTH], trigvnum[MAX_INPUT_LENGTH];
    char locnum[MAX_INPUT_LENGTH];
    index_data *index = NULL;
    index_data *tindex;
    trig_data *trig;
    struct int_list *trn, *trp;

    n = OLC_NUM(ch);
    switch (GET_OLC_MODE(ch)) {
    case OLC_MEDIT:
	n = mob_index[n] ? n : -1;
	break;
    case OLC_OEDIT:
	n = obj_index[n] ? n : -1;
	break;
    }

    if (n < 0 ) {
	send_to_char("No target found. \r\n", ch);
	return;
    } 

    switch (GET_OLC_MODE(ch)) {
    case OLC_MEDIT:
	index = mob_index[n];
	break;
    case OLC_OEDIT:
	index = obj_index[n];
	break;
    }


    half_chop(argument, arg1, trigvnum);
    two_arguments(trigvnum, trigvnum, locnum);

    if (!*arg1 || (*trigvnum && !isdigit(*trigvnum))) {
	send_to_char("Usage: trigger { add } { trigger vnum } [location] \r\n", ch);
	send_to_char("       trigger { delete } { trigger vnum } \r\n", ch);
	send_to_char("       trigger { show } \r\n", ch);
	return;
    }
    if (*trigvnum)
	tn = atoi(trigvnum);
    loc = (*locnum) ? atoi(locnum) : -1;

    if (tn && trig_index[tn] == NULL) {
	sprintf(buf,"Trigger Vnum %d does not exist. \r\n", tn);
	send_to_char(buf, ch);
	return;
    }
    if (is_abbrev(arg1, "add")) {
	CREATE(trn, struct int_list, 1);
	trn->i = tn;
	for (n = loc, trp = index->triggers; trp && trp->next && (n != 0); n--, trp = trp->next);

	if (!loc) {
	    trn->next = index->triggers;
	    index->triggers = trn;
	}
	else if (!trp)
	    index->triggers = trn;
	else {
	    trn->next = trp->next;
	    trp->next = trn;
	}
	sprintf(buf, "Trigger %d added.\r\n", tn);
	send_to_char(buf, ch);
    }
    if (is_abbrev(arg1, "delete")) {
	if (!index->triggers) {
	    send_to_char("No triggers to remove. \r\n", ch);
	    return;
	}
	for (n = 0, trn = NULL, trp = index->triggers; trp; trn = trp, trp = trp->next) {
	    if (tn == trp->i)
		break;
	}

	if (trp) {
	    if (trn) {
		trn->next = trp->next;
	    }
	    else {
		index->triggers = trp->next;
	    }
	}
	sprintf(buf, "Trigger %d removed.", tn);
	send_to_char(buf, ch);
    }
    if (is_abbrev(arg1, "show")) {
	n = 0;
	if (!index->triggers) {
	    send_to_char("No triggers attached. \r\n", ch);
	    return;
	}
	trp = index->triggers;
	tindex = trig_index[n];
	trig = (trig_data *) index->proto;
	while (trp) {
	    sprintf(buf, "Vnum. [%5d] %s \r\n", trig_index[n]->virt,
		    GET_TRIG_NAME(trig));
	    send_to_char(buf, ch);
	    trp = trp->next;
	}
    }
}
#endif


// More stuff imported from Oasis by Daniel Houghton
void olc_add_to_save_list(int zone, byte type)
{ struct olc_save_info *adding;

  /*. Return if it's already in the list .*/
  for(adding = olc_save_list; adding; adding = adding->next)
    if ((adding->zone == zone) && (adding->type == type))
      return;

  CREATE(adding, struct olc_save_info, 1);
  adding->zone = zone;
  adding->type = type;
  adding->next = olc_save_list;
  olc_save_list = adding;
}

/*. Remove an entry from the 'to be saved' list .*/

void olc_remove_from_save_list(int zone, byte type)
{ struct olc_save_info **entry;
  struct olc_save_info *temp;

  for(entry = &olc_save_list; *entry; entry = &(*entry)->next)
    if (((*entry)->zone == zone) && ((*entry)->type == type))
    { temp = *entry;
      *entry = temp->next;
      free(temp);
      return;
    }
}

/*. This procedure removes the '\r\n' from a string so that it may be
    saved to a file.  Use it only on buffers, not on the oringinal
    strings.*/

void strip_string(char *buffer)
{ register char *ptr, *str;

  ptr = buffer;
  str = ptr;

  while((*str = *ptr))
  { str++;
    ptr++;
    if (*ptr == '\r')
      ptr++;
  }
}


/*. This procdure frees up the strings and/or the structures
    attatched to a descriptor, sets all flags back to how they
    should be .*/

void cleanup_olc(char_data *ch, byte cleanup_type)
{ 
  struct event_list *el, *next_el;
  byte i;

  extern struct char_data *purged_chars;
  extern struct obj_data *purged_objs;
  extern struct trig_data *purged_trigs;
  extern struct script_data *purged_scripts;
  extern void extract_script(struct script_data *sc);

  if (OLC_DATA(ch))
  {
    if (OLC_STORAGE(ch)) {
      free (OLC_STORAGE(ch));
      OLC_STORAGE(ch) = NULL;
    }

    if (OLC_DATA(ch)->farg) {
      free (OLC_DATA(ch)->farg);
      OLC_DATA(ch)->farg = NULL;
    }

    /*. Check for room .*/
    if(OLC_ROOM(ch))
    { /*. free_room performs no sanity checks, must be carefull here .*/
      switch(cleanup_type)
      { 
        case CLEANUP_ALL: 
          FREE (OLC_ROOM(ch));
          break;
        case CLEANUP_STRUCTS:
          FREE(OLC_ROOM(ch));
          break;
        default:
          /*. Caller has screwed up .*/
          break;
      }

      if (OLC_EXIT(ch)) {
        if (OLC_EXIT(ch)->keyword)
	  FREE(OLC_EXIT(ch)->keyword);
        if (OLC_EXIT(ch)->general_description)
	  FREE(OLC_EXIT(ch)->general_description);
	FREE(OLC_EXIT(ch));
        OLC_EXIT(ch) = NULL;
      }
    }
  
    /*. Check for object .*/
    if (OLC_OBJ(ch)) { /*. free_obj checks strings arn't part of proto .*/
      if (GET_OBJ_TIMER(OLC_OBJ(ch))) {
  	cancel_event(GET_OBJ_TIMER(OLC_OBJ(ch)));
  	GET_OBJ_TIMER(OLC_OBJ(ch)) = NULL;
      }

      if (SCRIPT(OLC_OBJ(ch))) {
  	extract_script(SCRIPT(OLC_OBJ(ch)));
  	SCRIPT(OLC_OBJ(ch)) = NULL;
      }

      PURGED(OLC_OBJ(ch)) = TRUE;
      OLC_OBJ(ch)->next = purged_objs;
      purged_objs = OLC_OBJ(ch);

      OLC_OBJ(ch) = NULL;
    }

    /*. Check for mob .*/
    if(OLC_MOB(ch))
    { /*. free_char checks strings arn't part of proto .*/

  /* cancel any casting event */
  if (GET_ACTION(OLC_MOB(ch))) {
    cancel_event(GET_ACTION(OLC_MOB(ch)));
    GET_ACTION(OLC_MOB(ch)) = NULL;
  }

  /* cancel point updates */
  for (i = 0; i <= EVT_MOVE; i++)
    if (GET_POINTS_EVENT(ch, i)) {
      cancel_event(GET_POINTS_EVENT(ch, i));
      GET_POINTS_EVENT(ch, i) = NULL;
    }

  /* cancel damage events */
  for (el = GET_DAMAGE_EVENTS(OLC_MOB(ch)); el; el = next_el) {
    next_el = el->next;
    cancel_event(el->event);
    FREE(el);
  }
  GET_DAMAGE_EVENTS(OLC_MOB(ch)) = NULL;

  /* don't bother canceling affect events... affect_event() can handle it */
  
  if (SCRIPT(OLC_MOB(ch))) {
    extract_script(SCRIPT(OLC_MOB(ch)));
    SCRIPT(OLC_MOB(ch)) = NULL;
  }

      PURGED(OLC_MOB(ch)) = TRUE;
      OLC_MOB(ch)->next = purged_chars;
      purged_chars = OLC_MOB(ch);

      OLC_MOB(ch) = NULL;
    }

    /*. Check for zone .*/
    if(OLC_ZONE(ch))
    { /*. cleanup_type is irrelevant here, free everything .*/
      FREE(OLC_ZONE(ch)->name);
      FREE(OLC_ZONE(ch)->cmd);
      FREE(OLC_ZONE(ch));
    }

    /*. Check for shop .*/
    if(OLC_SHOP(ch)) {
    /*. free_shop performs no sanity checks, must be carefull here .*/
      switch(cleanup_type) {
        case CLEANUP_ALL:
          free_shop(OLC_SHOP(ch));
          break;
        case CLEANUP_STRUCTS:
          free_shop(OLC_SHOP(ch));
          break;
        default:
          /*. Caller has screwed up .*/
          break;
      }
    }

    if (OLC_SCRIPT(ch))
      delete OLC_SCRIPT(ch);

    /*. Restore desciptor playing status .*/
    REMOVE_BIT(PLR_FLAGS(ch), PLR_WRITING);
    STATE(ch->desc) = CON_PLAYING;
    act("$n stops using OLC.", TRUE, ch, 0, 0, TO_ROOM);
    FREE (OLC_DATA(ch));
    OLC_DATA(ch) = NULL;
  }

  GET_OLC_MODE(ch) = 0;
  GET_OLC_TARG(ch) = 0;
}


// Takes a zone rnum and an OLC state as arguments, and saves the file.
void gen_olc_save(int savenum, int subcmd)
{
  // void *ptr;
  int i, zone, rec = 0, bot, top;;
  FILE *fl;
  char filename[256], buf[256];

  const char *suffix[] = {
    "TMP",
    "mob",
    "obj",
    "wld",
    "zon",
    "shp",
    "trg",
    "ERROR"
  };
  const char *prefix[] = {
    "TMP",
    MOB_PREFIX,
    OBJ_PREFIX,
    WLD_PREFIX,
    ZON_PREFIX,
    SHP_PREFIX,
    TRG_PREFIX,
    "ERROR"
  };

  extern int add_to_index(int zone, const char *type);
  extern void fprint_shop(FILE *stream, shop_data *shop);

  zone = real_zone(savenum);

//   sprintf(filename, "%s/tmp%d.%s", prefix[subcmd],
// 	  zone_table[zone].number, suffix[subcmd]);

  sprintf(filename, "%s/tmp%d.%s", prefix[subcmd],
	  zone_table[zone].number, suffix[subcmd]);

  sprintf(buf, "OLC:  Writing data for zone v%d, r%d.", 
          zone_table[zone].number, zone);
  log(buf);

  if ((fl = fopen(filename, "w")) == NULL) {
    sprintf(buf, "OLC:  Unable to open %s for writing.", filename);
    log(buf);
    return;
  }

  /* write the file */
  switch (subcmd) {
  case OLC_MEDIT:
    bot = zone ? zone_table[zone - 1].top + 1 : 0;
//    bot = zone ? zone_table[zone].number * 100 : 0;
    top = zone_table[zone].top;
    for (i = mob_index.next(-1), rec = 0; i >= 0; i = mob_index.next(i))
	if (i >= bot && i <= top) {
	    fprint_mobile(fl, (char_data *) mob_index[i]->proto);
	    ++rec;
	}
    break;

  case OLC_OEDIT:
    bot = zone ? zone_table[zone - 1].top + 1 : 0;
    top = zone_table[zone].top;
    for (i = obj_index.next(-1), rec = 0; i >= 0; i = obj_index.next(i))
	if (i >= bot && i <= top) {
	    fprint_object(fl, (obj_data *) obj_index[i]->proto);
	    ++rec;
	}
    break;

  case OLC_REDIT:
    for (i = 0, rec = 0; i <= top_of_world; ++i)
      if (world[i] && world[i]->zone == zone) {
	fprint_room(fl, world[i]);
	++rec;
      }
    break;

  case OLC_ZEDIT:
    fprint_zone(fl, &zone_table[zone]);
    rec = 1;
    break;

  case OLC_SEDIT:
    bot = zone ? zone_table[zone - 1].top + 1 : 0;
    top = zone_table[zone].top;
    for (i = 0; i < top_shop; ++i)
      if ((SHOP_NUM(i) >= bot) && (SHOP_NUM(i) <= top)) {
        fprint_shop(fl, shop_index + i);
	++rec;
      }
    break;

  case OLC_TRIGEDIT:
    bot = zone ? zone_table[zone - 1].top + 1 : 0;
    top = zone_table[zone].top;
    for (i = trig_index.next(-1), rec = 0; i >= 0; i = trig_index.next(i))
	if (i >= bot && i <= top) {
	    fprint_trig(fl, (trig_data *) trig_index[i]->proto);
	    ++rec;
	}
    break;
  }

  fprintf(fl, "#99999\n$~\n");

  if (fclose(fl) == EOF)
    perror("fclose");

  if (!rec)
    sprintf(buf, "OLC:  No %ss found to write!", suffix[subcmd]);
  else
    sprintf(buf, "OLC:  %d %s%s written to file %s.",
	    rec, suffix[subcmd], (rec > 1) ? "s" : "", filename);
  log(buf);

  sprintf(buf2, "mv %s ", filename);
  
  sprintf(filename, "%s/%d.%s", prefix[subcmd],
	  zone_table[zone].number, suffix[subcmd]);
  strcat(buf2, filename);
	  
  if (system(buf2) == -1)
    mudlog("Error on port open for OLC write!", CMP, TRUST_IMPL, TRUE);
	  
  add_to_index(zone_table[zone].number, (const char *) suffix[subcmd]);

  olc_remove_from_save_list(zone_table[zone].number, (byte) subcmd);
}


ACMD(do_zstat)
{
  int i, zn = -1, realzn = -1, pos = 0, none = TRUE;
  struct builder_list *bldr;

  char *rmode[] =
    {  "Never resets",
       "Resets only when deserted",
       "Always resets"  };

  one_argument(argument, buf);

  if ((*buf) && (isdigit(*buf)))
    zn = atoi(buf);
  else {
    zn = world[IN_ROOM(ch)]->number / 100;
  }

  realzn = real_zone(zn);

  if (realzn == -1) {
    sprintf(buf, "SYSERR: Zone #%d doesn't seem to exist.",
      zn);
    log(buf);
    send_to_char("Specified zone does not exist.\r\n", ch);
    return;
  }

  do_show_zone(ch, realzn);
  
}


ACMD(do_zallow)
{
  char name[MAX_INPUT_LENGTH], buf[500];
  int zn, realzn;

  if (GET_TRUST(ch) < TRUST_SUBIMPL) {
    send_to_char("Sorry, you can not change the access control list for a zone.\r\n", ch);
    return;
  }

  two_arguments(argument, buf2, name);

  if ((!*name) || ((!*buf2) || (!isdigit(*buf2))))
    send_to_char("Usage:  zallow <zone vnum> [ all | <person> ]\r\n", ch);

  else if (str_cmp(name, "all") && player_index->get_id_by_name(name) < 0)
    send_to_char("That person does not exist.\r\n", ch);

  else {
    zn = atoi(buf2);
    realzn = real_zone(zn);

    if (realzn == -1) {
      send_to_char("That zone doesn't exist!\r\n", ch);
      return;
    }
    
    if (!str_cmp(name, "all")) {
      if (zone_table[realzn].permissions)
	FREE(zone_table[realzn].permissions);
      
      zone_table[realzn].permissions = str_dup("all");

      sprintf(buf, "Anyone can now edit zone %d.\r\n", zn);
      send_to_char(buf, ch);
    }

    else if (zone_table[realzn].permissions &&
	     (str_str(zone_table[realzn].permissions, name) ||
	     !str_cmp(zone_table[realzn].permissions, "all")))
      send_to_char("That person already has permission to edit that zone.\r\n",
		   ch);

    else {
      if (zone_table[realzn].permissions) {
	sprintf(buf, "%s %s", zone_table[realzn].permissions,  name);
	FREE(zone_table[realzn].permissions);
	zone_table[realzn].permissions = str_dup(buf);
      }

      else
	zone_table[realzn].permissions = str_dup(name);

      // gen_olc_save(zn, OLC_ZEDIT);
      olc_add_to_save_list(zn, OLC_ZEDIT);

      sprintf(buf, "%s added to access list for zone %d.\r\n", name, zn);
      send_to_char(CAP(buf), ch);
    }
  }
}


ACMD(do_zdeny)
{
  char name[MAX_INPUT_LENGTH], buf[500], *s;
  int zn, realzn, len, all;

  if (GET_TRUST(ch) < TRUST_SUBIMPL) {
    send_to_char("Sorry, you can not change the access control list for a zone.\r\n", ch);
    return;
  }

  two_arguments(argument, buf2, name);

  if ((!*name) || ((!*buf2) || (!isdigit(*buf2)))) {
    send_to_char("Usage:  zdeny <zone vnum> [ all | <person> ]\r\n", ch);
    return;
  }

  len = strlen(name);
  all = !str_cmp(name, "all");

  zn = atoi(buf2);
  realzn = real_zone(zn);

  if (realzn == -1) {
    send_to_char("That zone doesn't exist!\r\n", ch);
    return;
  }
  
  if (!all && (!zone_table[realzn].permissions ||
	   (s = str_str(zone_table[realzn].permissions, name)) == NULL))
    send_to_char("That person is not on the access list for that zone.\r\n", ch);

  else if (!all && ((zone_table[realzn].permissions != s && !isspace(*(s - 1))) ||
	   (s[len] && !isspace(s[len]))))
    send_to_char("That person is not on the access list for that zone.\r\n", ch);
  
  else {

    if (all) {
      if (zone_table[realzn].permissions)
	FREE(zone_table[realzn].permissions);
      zone_table[realzn].permissions = NULL;

      sprintf(buf, "All OLC access removed from zone %d.\r\n", zn);
      send_to_char(buf, ch);
    } else {
      *s = '\0';
      strcpy(buf, zone_table[realzn].permissions);
      if (s[len])
    	strcat(buf, s + len + 1);

      FREE(zone_table[realzn].permissions);
      if (!*buf)
    	zone_table[realzn].permissions = NULL;
      else
    	zone_table[realzn].permissions = str_dup(buf);

      // gen_olc_save(zn, OLC_ZEDIT);
      olc_add_to_save_list(zn, OLC_ZEDIT);

      sprintf(buf, "%s removed from access list for zone %d.\r\n", name, zn);
      send_to_char(CAP(buf), ch);
    }
  }
}


/* Ray's dig code for Oasis OLC, added by Daniel Houghton */
ACMD(do_rdig)
{
  /* Only works if you have Oasis OLC */
  extern void redit_save_internally(char_data *ch);
  extern void olc_add_to_save_list(int zone, byte type);
  extern const int rev_dir[];

  char buf2[120];
  char buf3[120];
  char buf[120];
  extern const char *dirs[];
  int iroom = 0, rroom = 0;
  int dir = 0;
  bool newroom = FALSE, savedoor = FALSE;
  /*  struct room_data *room; */

  two_arguments(argument, buf2, buf3);
  /* buf2 is the direction, buf3 is the room */

        iroom = atoi(buf3);
        rroom = real_room(iroom);

  if (!*buf2) {
    send_to_char("Format: rdig <dir> <room number>\r\n", ch);
    return;
  } else if (!*buf3) {
    send_to_char("Format: rdig <dir> <room number>\r\n", ch);
    return;
  } else if (!isdigit(*buf3)) {
    send_to_char("Format: rdig <dir> <room number>\r\n", ch);
    return;
  }

  if (find_owner_zone(iroom) == -1) {
    send_to_char("There is no zone for that number!\r\n", ch);
    return;
  }

  if (GET_LEVEL(ch) < TRUST_SUBIMPL) {
    if (!check_zone_permission(ch, find_owner_zone(IN_ROOM(ch)))) {
      sprintf(buf, "You don't have builder rights here in zone #%i\r\n", 
                   find_owner_zone(IN_ROOM(ch)));
      send_to_char(buf, ch);
      return;
    }
  }

  CREATE(OLC_DATA(ch), struct olc_data, 1);

  if (rroom <= 0) {
    if (!check_zone_permission(ch, find_owner_zone(iroom))) { 
      sprintf(buf, "You don't have builder rights to create rooms in zone #%i\r\n", 
                    find_owner_zone(iroom));
      send_to_char(buf, ch);
      return;
    }
    OLC_NUM(ch) = iroom;
    OLC_ROOM(ch) = new room_data;
    init_room(OLC_ROOM(ch));
    newroom = TRUE;
  } else {
    if ((GET_TRUST(ch) < TRUST_IMPL) && 
       (ROOM_FLAGGED(rroom, ROOM_PRIVATE | ROOM_GODROOM))) {
      send_to_char("You aren't godly enough to make an exit to that room.\r\n", ch);
      return;
    }
  }

  /* Main stuff */
  switch (*buf2) {
    case 'n':
    case 'N':
      dir = NORTH;
      break;
    case 'e':
    case 'E':
      dir = EAST;
      break;
    case 's':
    case 'S':
      dir = SOUTH;
      break;
    case 'w':
    case 'W':
      dir = WEST;
      break;
    case 'u':
    case 'U':
      dir = UP;
      break;
    case 'd':
    case 'D':
      dir = DOWN;
      break;
  }

  if (world[IN_ROOM(ch)]->dir_option[dir])
    free(world[IN_ROOM(ch)]->dir_option[dir]);

  CREATE(world[IN_ROOM(ch)]->dir_option[dir], struct room_direction_data,1);
  world[IN_ROOM(ch)]->dir_option[dir]->general_description = NULL;
  world[IN_ROOM(ch)]->dir_option[dir]->keyword = NULL;
  world[IN_ROOM(ch)]->dir_option[dir]->to_room = iroom; 

  if (!newroom) {

    /* Daniel Houghton's change:  Didn't want to risk overwriting the zone! */
    if ((!world[rroom]->dir_option[rev_dir[dir]]) && 
       (check_zone_permission(ch, find_owner_zone(rroom)))) {
      free (world[rroom]->dir_option[rev_dir[dir]]);
      savedoor = TRUE;

      CREATE(world[rroom]->dir_option[rev_dir[dir]], struct room_direction_data,1);
      world[rroom]->dir_option[rev_dir[dir]]->general_description = NULL;
      world[rroom]->dir_option[rev_dir[dir]]->keyword = NULL;
      world[rroom]->dir_option[rev_dir[dir]]->to_room = IN_ROOM(ch);
    } else {
      send_to_char("No reverse exit created.\r\n", ch);
    }

  } else {
    CREATE(OLC_ROOM(ch)->dir_option[rev_dir[dir]], struct room_direction_data, 1);
    OLC_ROOM(ch)->dir_option[rev_dir[dir]]->general_description = NULL;
    OLC_ROOM(ch)->dir_option[rev_dir[dir]]->keyword = NULL;
    OLC_ROOM(ch)->dir_option[rev_dir[dir]]->to_room = IN_ROOM(ch);
  }

    /* Only works if you have Oasis OLC */
  if (newroom)
    redit_save_internally(ch);

  world[IN_ROOM(ch)]->dir_option[dir]->to_room = iroom;

  /* save world data for destination room */
  if ((!newroom) && (savedoor))
    olc_add_to_save_list(zone_table[find_owner_zone(rroom)].number, OLC_REDIT);

  /* save world data for starting room */
  
  olc_add_to_save_list(zone_table[find_owner_zone(IN_ROOM(ch))].number, OLC_REDIT);
   
  free(OLC_ROOM(ch));
  free(OLC_DATA(ch));
  OLC_DATA(ch) = NULL;
  sprintf(buf, "You make an exit %s from room %d to room %d.\r\n", dirs[dir], IN_ROOM(ch), iroom);
  send_to_char(buf, ch);
}


/*
 * For extra descriptions.
 */
void olc_disp_specproc_menu(char_data *ch)
{
  struct spec_type *spec_table = NULL;
  int i;

  switch (GET_OLC_MODE(ch)) {
  case OLC_MEDIT:
    spec_table = (struct spec_type *) spec_mob_table;
    break;
  case OLC_OEDIT:
    spec_table = (struct spec_type *) spec_obj_table;
    break;
  case OLC_REDIT:
    spec_table = (struct spec_type *) spec_room_table;
    break;
  }
  
  *buf = '\0';

#if defined(CLEAR_SCREEN)
  strcat(buf, "[H[J");
#endif

  for (i = 0; spec_table[i].name; i++)
    sprintf(buf + strlen(buf), "%-20s%5s", spec_table[i].name,
  	    ((i % 3) == 2) ? "\r\n" : "");

  if ((i % 3) != 0)
    strcat(buf, "\r\n");
    
  strcat(buf, "Enter your choice (\"none\" clears) : ");

  send_to_char(buf, ch);
}


void olc_edit_special(char_data *ch, char *argument)
{
    char buf[MAX_STRING_LENGTH];
    struct spec_type *spec_table = NULL;
    int n = 0, i;
    SpecProc proc;

    char **farg = NULL;
    struct finish_olcstr_data *fd;
    void *ptr = NULL;
    index_data *index;

    if (OLC_DATA(ch) == NULL)
      return;

    switch (GET_OLC_MODE(ch)) {
    case OLC_MEDIT:
	spec_table = (struct spec_type *) spec_mob_table;
	break;
    case OLC_OEDIT:
	spec_table = (struct spec_type *) spec_obj_table;
	break;
    case OLC_REDIT:
	spec_table = (struct spec_type *) spec_room_table;
	break;
    }

    argument = any_one_arg(argument, buf);
    if (!*buf) {
	strcpy(buf, "Usage:  special [<name>| none ] [<argument>]\r\n"
	       "Available special procedures are:\r\n");
	for (i = 0; spec_table[i].name; i++)
	    sprintf(buf, "%s  %-20s%s", buf, spec_table[i].name,
		    ((i % 3) == 2) ? "\r\n" : "");

	if ((i % 3) != 0)
	    strcat(buf, "\r\n");
	    
	send_to_char(buf, ch);
	// page_string(ch->desc, buf, TRUE);
	return;
    }

    if (n < 0) {
	send_to_char("No target found.\r\n", ch);
	return;
    }

    else if (!str_cmp(buf, "none"))
	proc = NULL;

    else if ((proc = spec_lookup(buf, spec_table)) == NULL) {
	send_to_char("Special procedure not found.\r\n", ch);
	return;
    }

    switch (GET_OLC_MODE(ch)) {
    case OLC_MEDIT:
    case OLC_OEDIT:
    case OLC_REDIT:
      OLC_DATA(ch)->func = proc;
      break;
    }

    if (proc == NULL) {
      if (OLC_DATA(ch)->farg)
	FREE(OLC_DATA(ch)->farg);
      send_to_char("Special procedure set to none.\r\n", ch);
    }

    else {
	sprintf(buf, "Special procedure set to %s.\r\n",
		spec_name(proc, spec_table));
	send_to_char(buf, ch);
    
	CREATE(fd, struct finish_olcstr_data, 1);
	fd->string = &OLC_DATA(ch)->farg;
	fd->mode = GET_OLC_MODE(ch);
	fd->target = OLC_NUM(ch);
	fd->ptr = get_current_olc_ptr(ch, OLC_NUM(ch), GET_OLC_MODE(ch));
    
	skip_spaces(&argument);
	olc_edit_string(ch, &(OLC_DATA(ch)->farg), argument, "special procedure argument", 
			finish_olc_farg, fd, MAX_FARG_LENGTH, EDIT_ONELINE);
    }
}


void erase_exdesc(struct extra_descr_data **head, struct extra_descr_data **lookfor) 
{
  struct extra_descr_data *prev = NULL, *place = *head, *target = *lookfor;
  
  while (place && (place != target)) {
    prev = place;
    place = place->next;
  }
  
  if (!place) {
    return;
  }

  if (prev)
    prev->next = target->next;
  else
    *head = target->next;
  
  ss_free(target->keyword);
  ss_free(target->description);
  
  FREE(target);
  *lookfor = NULL;
  
}
