/* ************************************************************************
*  File: editor.c                                Part of Death's Gate MUD *
*                                                                         *
*  Usage: functions for string editing                                    *
*                                                                         *
*  All rights reserved.  See license.doc for complete information.        *
*                                                                         *
*  Death's Gate MUD is based on CircleMUD, Copyright (C) 1993, 94.        *
*  CircleMUD is based on DikuMUD, Copyright (C) 1990, 1991.               *
*                                                                         *
*  $Author: tlo $
*  $Date: 1998/12/31 11:00:50 $
*  $Revision: 6.8 $
************************************************************************ */

#include "conf.h"
#include "sysdep.h"


#include "structs.h"
#include "scripts.h"
#include "index.h"
#include "memory.h"
#include "utils.h"
#include "comm.h"
#include "db.h"
#include "boards.h"
#include "mail.h"
#include "interpreter.h"
#include "olc.h"
#include "editor.h"
#include "find.h"

#ifdef REDHAT
#include <std/bastring.h>
typedef basic_string < char >String;
#else
#include "String.h"

// Single replacement of a pattern - DH
void String::sub(const char* pat, const char* r)
{
  String After(this.after(pat));

  this.del(this.index(pat), this.length() - this.index(pat));
  
  cat(*this, r, *this);
  cat(*this, After, *this);
}

#endif

// action modes for parse_action
#define PARSE_FORMAT		0 
#define PARSE_REPLACE		1 
#define PARSE_HELP		2 
#define PARSE_DELETE		3
#define PARSE_INSERT		4
#define PARSE_LIST_NORM		5
#define PARSE_LIST_NUM		6
#define PARSE_EDIT		7

int replace_str(char **string, char *pattern, char *replacement, int rep_all,
		unsigned int max_size);
void format_text(char **ptr_string, int mode, descriptor_data * d,
		 unsigned int maxlen, unsigned int rmargin = 78);
int string_freq(const char *searchin, const char *searchfor);

/* ************************************************************************
   *  modification of malloc'ed strings                                      *
   ************************************************************************ */


int replace_str(char **string, const char *pattern, const char *replacement, int rep_all,
		unsigned int max_size)
{
  String tempstring;
  int pos, replaced = 0;
  const int patternlen = strlen(pattern);
  unsigned int begin = 0, end = 0;
  const char *tmpptr = *string;

  tempstring = tmpptr;

#ifdef REDHAT
  pos = tempstring.find(pattern, end);

  while (pos != -1) {
    begin = pos;
    end = begin + patternlen;

    tempstring.replace(begin, patternlen, replacement);

    if (rep_all)
      pos = tempstring.find(pattern, end);
    else
      pos = -1;			// 

    ++replaced;
  }

  strncpy(*string, tempstring.data(), max_size - 1);
#else
  if (rep_all) {
    tempstring.gsub(pattern, replacement);
  } else {
    tempstring.sub(pattern, replacement);
  }

  strncpy(*string, tempstring.chars(), max_size - 1);
#endif

  return replaced;
}


void string_edit(struct descriptor_data *d, char *str, size_t size,
		 EDITFUNC(*func), void *obj, byte flags)
{
  struct editor_data *ed;

  if (d->ed) {
    SEND_TO_Q("You are already editing something!\r\n", d);
    return;
  }
  CREATE(ed, struct editor_data, 1);
  CREATE(ed->str, char, size);

  ed->max_str = size;
  ed->func = func;
  ed->obj = obj;
  ed->flags = flags;

  if ((IS_SET(ed->flags, EDIT_APPEND)) && str)
    strncpy(ed->str, str, size - 4);
  else
    *ed->str = '\0';
  
  d->ed = ed;
  SEND_TO_Q("(/s saves /h for help)\r\n", d);

  if (!d->connected && d->character && !IS_NPC(d->character))
    SET_BIT(PLR_FLAGS(d->character), PLR_WRITING);
}


/*  handle some editor commands */
void parse_action(int command, char *string, struct descriptor_data *d)
{
   int indent = 0, rep_all = 0, flags = 0, replaced = 0;
   unsigned int total_len;
   register int j = 0;
   int i, line_low = 1, line_high = 999999;
   char *s = 0, *t = 0, temp;
   

   // For greater efficiency, setting of line_low and line_high has been
   // placed here.

   switch (command) { 
    case PARSE_DELETE:
    case PARSE_LIST_NORM:
    case PARSE_LIST_NUM:
      *buf3 = '\0';
      /* note: my buf,buf1,buf2 vars are defined at 32k sizes so they
       * are prolly ok fer what i want to do here. */
      if (*string != '\0')
	switch (sscanf(string, " %d - %d ", &line_low, &line_high)) {
         case 0:
           if (command == PARSE_DELETE) {
	     SEND_TO_Q("You must specify a line number or range to delete.\r\n", d);
	     return;
	   }
	   break;
	 case 1:
	   line_high = line_low;
	   break;
         case 2:
	   if (line_high < line_low) {
	     SEND_TO_Q("That range is invalid.\r\n", d);
	     return;
	   }
	   break;
	}
      if (line_low < 1) {
	 SEND_TO_Q("Line numbers must be greater than 0.\r\n", d);
	 return;
      }
      if (line_high < line_low) {
	 SEND_TO_Q("That range is invalid.\r\n", d);
	 return;
      }
      break;
    case PARSE_INSERT:
    case PARSE_EDIT:
      half_chop(string, buf3, buf4);
      if (*buf3 == '\0') {
	 switch (command) {
	 case PARSE_INSERT:
	   SEND_TO_Q("You must specify a line number before which to insert text.\r\n", d);
	 case PARSE_EDIT:
	   SEND_TO_Q("You must specify a line number at which to change text.\r\n", d);
	 }
	 return;
      }
      line_low = atoi(buf3);
      if (line_low <= 0) {
        SEND_TO_Q("Line number must be higher than 0.\r\n", d);
        *buf3 = '\0';
        *buf4 = '\0';
        return;
      }
      strcat(buf4, "\r\n");
      *buf3 = '\0';
      break;
    default:
      break;
   }


   // We've established the line numbers.  Now actually do the processing!

   switch (command) { 
    case PARSE_HELP: 
      sprintf(buf3, 
	      "Editor command formats: /<letter>\r\n\r\n"
	      "/a         -  aborts editor\r\n"
	      "/c         -  clears buffer\r\n"
	      "/d#        -  deletes a line #\r\n"
	      "/e# <text> -  changes the line at # with <text>\r\n"
	      "/f         -  formats text\r\n"
	      "/fi        -  indented formatting of text\r\n"
	      "/h         -  list text editor commands\r\n"
	      "/i# <text> -  inserts <text> before line #\r\n"
	      "/l         -  lists buffer\r\n"
	      "/n         -  lists buffer with line numbers\r\n"
	      "/r 'a' 'b' -  replace 1st occurance of text <a> in buffer with text <b>\r\n"
	      "/ra 'a' 'b'-  replace all occurances of text <a> within buffer with text <b>\r\n"
	      "              usage: /r[a] 'pattern' 'replacement'\r\n"
	      "/s         -  saves text\r\n");
      SEND_TO_Q(buf3, d);
      break;
    case PARSE_FORMAT: 
      while (isalpha(string[j]) && j < 2) {
	 switch (string[j]) {
	  case 'i':
	    if (!indent) {
	       indent = 1;
	       flags += FORMAT_INDENT;
	    }             
	    break;
	  default:
	    break;
	 }     
	 j++;
      }

      *buf3 = '\0';
    format_text(&d->ed->str, flags, d, d->ed->max_str, 79);
      *buf3 = '\0';

      sprintf(buf3, "Text formatted with%s indent.\r\n", (indent ? "" : "out")); 
      SEND_TO_Q(buf3, d);
      break;    
    case PARSE_REPLACE: 
      while (isalpha(string[j]) && j < 2) {
	 switch (string[j]) {
	  case 'a':
	    if (!indent) {
	       rep_all = 1;
	    }             
	    break;
	  default:
	    break;
	 }     
	 j++;
      }

      // Now, extract the pattern and replacement into t and s
      s = strtok(string, "'");
      if (s == NULL) {
	 SEND_TO_Q("Invalid format.\r\n", d);
	 return;
      }
      s = strtok(NULL, "'");
      if (s == NULL) {
	 SEND_TO_Q("Target string must be enclosed in single quotes.\r\n", d);
	 return;
      }
      t = strtok(NULL, "'");
      if (t == NULL) {
	 SEND_TO_Q("No replacement string.\r\n", d);
	 return;
      }
      t = strtok(NULL, "'");
      if (t == NULL) {
	 SEND_TO_Q("Replacement string must be enclosed in single quotes.\r\n", d);
	 return;
      }

      total_len = (strlen(t) - strlen(s));
      // CHANGEPOINT:  We need a function to count the number of occurrences
      // of a string within a string
      replaced = string_freq(d->ed->str, s);

      if (replaced == 0) {
        sprintf(buf3, "String '%s' not found.\r\n", s); 
        SEND_TO_Q(buf3, d);
        return;
      }
    if (rep_all)
      total_len = (total_len * replaced);

        total_len += strlen(d->ed->str);
      
    if (total_len > (d->ed->max_str - 4)) {
	SEND_TO_Q("Not enough space left in buffer.\r\n", d);
	return;
      }
    replace_str(&d->ed->str, (const char *) s, (const char *) t, rep_all, d->ed->max_str);
    // sprintf(buf, "d->ed->str length %d, ", strlen(d->ed->str));
    // log (buf);
      
    // This ensures that the string is terminated properly.
    *((char *) (d->ed->str + d->ed->max_str - 1)) = '\0';
      
      break;
    case PARSE_DELETE:
      i = 1;
      total_len = 1;

      if (strlen(d->ed->str) <= 0) {
	 SEND_TO_Q("Buffer is empty.\r\n", d);
	 return;
      }
      s = d->ed->str;

      while (s && (i < line_low))
        if ((s = strchr(s, '\n')) != NULL) {
          i++;
          s++;
	}

      if ((i < line_low) || (s == NULL)) {
	SEND_TO_Q("Line(s) out of range; not deleting.\r\n", d);
	return;
      }
      t = s;

      while (s && (i < line_high))
        if ((s = strchr(s, '\n')) != NULL) {
          i++;
          total_len++;
          s++;
        }
      if ((s) && ((s = strchr(s, '\n')) != NULL)) {
        s++;
      while (*s != '\0')
	*(t++) = *(s++);
      } else 
        total_len--;

      *t = '\0';

      sprintf(buf3, "%d line%s deleted.\r\n", total_len,
	    ((total_len != 1) ? "s" : ""));
      SEND_TO_Q(buf3, d);
      break;

    case PARSE_LIST_NORM:
      /* note: my buf,buf1,buf2 vars are defined at 32k sizes so they
       * are prolly ok fer what i want to do here. */

      i = 1;
      total_len = 0;
      s = d->ed->str;
      while (s && (i < line_low))
	if ((s = strchr(s, '\n')) != NULL) {
	   i++;
	   s++;
	}
      if ((i < line_low) || (s == NULL)) {
	 SEND_TO_Q("Line(s) out of range; no buffer listing.\r\n", d);
	 return;
      }
      t = s;
      while (s && (i <= line_high))
	if ((s = strchr(s, '\n')) != NULL) {
	   i++;
	   total_len++;
	   s++;
	}
      if (s)	{
	 temp = *s;
	 *s = '\0';
	 strcat(buf3, t);
	 *s = temp;
    } else
      strcat(buf3, t);
      /* this is kind of annoying.. will have to take a poll and see..
      sprintf(buf3, "%s\r\n%d line%sshown.\r\n", buf3, total_len,
	      ((total_len != 1)?"s ":" "));
       */
      page_string(d, buf3, TRUE);
      break;
    case PARSE_LIST_NUM:
      /* note: my buf,buf1,buf2 vars are defined at 32k sizes so they
       * are prolly ok fer what i want to do here. */

      i = 1;
      total_len = 0;
      s = d->ed->str;
      while (s && (i < line_low))
	if ((s = strchr(s, '\n')) != NULL) {
	   i++;
	   s++;
	}
      if ((i < line_low) || (s == NULL)) {
	 SEND_TO_Q("Line(s) out of range; no buffer listing.\r\n", d);
	 return;
      }
      t = s;
      while (s && (i <= line_high))
	if ((s = strchr(s, '\n')) != NULL) {
	   i++;
	   total_len++;
	   s++;
	   temp = *s;
	   *s = '\0';
	sprintf(buf3, "%s%4d:\r\n", buf3, (i - 1));
	   strcat(buf3, t);
	   *s = temp;
	   t = s;
	}
      if (s && t) {
	 temp = *s;
	 *s = '\0';
	 strcat(buf3, t);
	 *s = temp;
    } else if (t)
      strcat(buf3, t);
      /* this is kind of annoying .. seeing as the lines are #ed
      sprintf(buf3, "%s\r\n%d numbered line%slisted.\r\n", buf3, total_len,
	      ((total_len != 1)?"s ":" "));
       */
      page_string(d, buf3, TRUE);
      break;

    case PARSE_INSERT:
      i = 1;
      total_len = strlen(d->ed->str);

      if (!total_len) {
	 SEND_TO_Q("Buffer is empty, nowhere to insert.\r\n", d);
	 return;
      }
    if ((total_len + strlen(buf4)) > (d->ed->max_str - 4)) {
	SEND_TO_Q("Insert text causes buffer to exceed maximum length, aborted.\r\n", d);
	return;
      }
      CREATE(t, char, d->ed->max_str);
      strcat(t, d->ed->str);

      if ((s = t) == NULL) {
	 SEND_TO_Q("Buffer is empty, nowhere to insert.\r\n", d);
	 return;
      }

      while (s && (i < line_low))
        if ((s = strchr(s, '\n')) != NULL) {
          i++;
          s++;
	}

        if ((i < line_low) || (s == NULL)) {
          SEND_TO_Q("Line number out of range; insert aborted.\r\n", d);
          return;
        }
        temp = *s;
        *s = '\0';

    if ((strlen(d->ed->str) + strlen(buf4) + strlen(s + 1) + 3) > d->ed->max_str) {
	  *s = temp;
	  SEND_TO_Q("Insert text pushes buffer over maximum size, insert aborted.\r\n", d);
	  return;
	}

        strcat(buf3, t);
        *s = temp;
        strcat(buf3, buf4);
        if (s && (*s != '\0')) 
          strcat(buf3, s);

        strcpy(d->ed->str, buf3);
	SEND_TO_Q("Line inserted.\r\n", d);
      break;

    case PARSE_EDIT:
      i = 1;
      s = d->ed->str;
      if (s == NULL) {
	 SEND_TO_Q("Buffer is empty, nothing to change.\r\n", d);
	 return;
      }
      if (line_low > 0) {
	 /* loop through the text counting /n chars till we get to the line */
      	 while (s && (i < line_low))
	   if ((s = strchr(s, '\n')) != NULL) {
	      i++;
	      s++;
	   }
	 /* make sure that there was a THAT line in the text */
	 if ((i < line_low) || (s == NULL)) {
	    SEND_TO_Q("Line number out of range; change aborted.\r\n", d);
	    return;
	 }
	 /* if s is the same as d->ed->str that means im at the beginning of the
	  * message text and i dont need to put that into the changed buffer */
	 if (s != d->ed->str) {
	    /* first things first .. we get this part into buf. */
	    temp = *s;
	    *s = '\0';
	    /* put the first 'good' half of the text into storage */
	    strcat(buf3, d->ed->str);
	    *s = temp;
	 }
	 /* put the new 'good' line into place. */
	 strcat(buf3, buf4);
	 if ((s = strchr(s, '\n')) != NULL) {
	    /* this means that we are at the END of the line we want outta there. */
	    /* BUT we want s to point to the beginning of the line AFTER
	     * the line we want edited */
	    s++;
	    /* now put the last 'good' half of buffer into storage */
	    strcat(buf3, s);
	 }
	 /* check for buffer overflow */
      if (strlen(buf3) > (d->ed->max_str - 4)) {
	    SEND_TO_Q("Change causes new length to exceed buffer maximum size, aborted.\r\n", d);
	    return;
	 }
	 /* change the size of the REAL buffer to fit the new text */
         strcpy(d->ed->str, buf3);
	 SEND_TO_Q("Line changed.\r\n", d);
    } else {
	 SEND_TO_Q("Line number must be higher than 0.\r\n", d);
	 return;
      }
      break;
    default:
      SEND_TO_Q("Invalid option.\r\n", d);
      mudlog("SYSERR: invalid command passed to parse_action", BRF, TRUST_IMPL, TRUE);
      return;
   }
}


/* Add user input to the 'current' string (as defined by d->ed->str) */
void string_add(struct descriptor_data *d, char *str)
{
   int terminator = 0, action = 0;
   register int i = 2, j = 0;
   char actions[MAX_INPUT_LENGTH];
   char *ptr;

  /* determine if this is the terminal string, and truncate if so */
  /* changed to only accept '@' at the beginning of line - J. Elson 1/17/94 */

  /* changed to accept '/<letter>' style editing commands - instead */
  /* of solitary '@' to end - (modification of improved_edit patch) */
  /*   M. Scott 10/15/96 */

  delete_doubledollar(str);

   if ((action = (*str == '/'))) {
      while (str[i] != '\0') {
	 actions[j] = str[i];              
	 i++;
	 j++;
      }
      actions[j] = '\0';
      *str = '\0';
      switch (str[1]) {
       case 'a':
	 terminator = 2; /* working on an abort message */
	 break;
       case 'c':
	 if (strlen(d->ed->str) > 0) {
	   *d->ed->str = '\0';
           SEND_TO_Q("Current buffer cleared.\r\n", d);
         } else {
           SEND_TO_Q("Current buffer empty.\r\n", d);
         }
//	 FREE(d->ed->str);
      //       CREATE(d->ed->str, char, d->ed->max_str);
      //       if (*(d->ed->str)) {
      //          d->ed->str = '\0';
      //            FREE(d->ed->str);
      //          SEND_TO_Q("Current buffer cleared.\r\n", d);
      //       }
      //       else
	 break;
       case 'd':
	 parse_action(PARSE_DELETE, actions, d);
	 break;
       case 'e':
	 parse_action(PARSE_EDIT, actions, d);
	 break;
       case 'f': 
	 if (*(d->ed->str))
	   parse_action(PARSE_FORMAT, actions, d);
	 else
	   SEND_TO_Q("Current buffer empty.\r\n", d);
	 break;
       case 'i':
	 if (*(d->ed->str))
	   parse_action(PARSE_INSERT, actions, d);
	 else
	   SEND_TO_Q("Current buffer empty.\r\n", d);
	 break;
       case 'h': 
	 parse_action(PARSE_HELP, actions, d);
	 break;
       case 'l':
	 if (d->ed->str)
	   parse_action(PARSE_LIST_NORM, actions, d);
      else
	SEND_TO_Q("Current buffer empty.\r\n", d);
	 break;
       case 'n':
	 if (d->ed->str)
	   parse_action(PARSE_LIST_NUM, actions, d);
      else
	SEND_TO_Q("Current buffer empty.\r\n", d);
	 break;
       case 'r':   
	 parse_action(PARSE_REPLACE, actions, d);
	 break;
       case 's':
	 terminator = 1;
	 break;
       default:
	 SEND_TO_Q("Invalid option.\r\n", d);
	 break;
      }
    *str = '\0';
   }

  if (IS_SET(d->ed->flags, EDIT_ONELINE)) {
    // Strip carriage return.
    if ((ptr = strchr(str, '\r')) != NULL)
      *ptr = '\0';
    if ((ptr = strchr(str, '\n')) != NULL)
      *ptr = '\0';
    terminator = 1;
  }
  // sprintf(buf3, "String length: %d  Input length: %d  Max length %d\r\n", 
  //              strlen(d->ed->str), strlen(str), d->ed->max_str);
  // log(buf3);

  if (!(d->ed->str)) {
    if (strlen(str) > (d->ed->max_str - 4)) {
      send_to_char("String too long - Truncated.\r\n", d->character);
      *(str + d->ed->max_str) = '\0';
    }
    strcpy(d->ed->str, str);
  } else {
    if ((strlen(str) + strlen(d->ed->str)) > (d->ed->max_str - 4)) {
      send_to_char("String too long, limit reached on message.  Last line ignored.\r\n",
		    d->character);
    } else {
      strcat(d->ed->str, str);
      if (!action)
        strcat(d->ed->str, "\r\n");
    }
  }

  if (terminator)
    string_finish(d, terminator);
}


/* finishes the edit on a string */
void string_finish(struct descriptor_data *d, int terminator)
{
  if (IS_SET(d->ed->flags, EDIT_ONELINE)) {
  }
  (d->ed->func) (d, d->ed->str, d->ed->obj, terminator);

  if (!d->connected && d->character && !IS_NPC(d->character))
    REMOVE_BIT(PLR_FLAGS(d->character), PLR_WRITING);

  FREE(d->ed->str);
  FREE(d->ed);
  d->ed = NULL;

  switch (STATE(d)) {
    /*. OLC states . */
  case CON_OEDIT: 
    if (d->character)
      oedit_parse(d->character, "");
    break;
  case CON_MEDIT: 
    if (d->character)
      medit_parse(d->character, "");
    break;
  case CON_REDIT: 
    if (d->character)
      redit_parse(d->character, "");
    break;
  case CON_ZEDIT: 
    if (d->character)
      zedit_parse(d->character, "");
    break;
  case CON_SEDIT: 
    if (d->character)
      sedit_parse(d->character, "");
    break;
  case CON_TRIGEDIT: 
    if (d->character)
      trigedit_parse(d->character, "");
    break;
    /*. End of OLC states . */
  default:
    break;
  }
}


/* finishing funcs */
EDITFUNC(finish_mail)
{
  if (terminator != 2) {
    store_mail(*((long *) obj), GET_IDNUM(d->character), str);
  SEND_TO_Q("Message sent!\r\n", d);
  if (!IS_NPC(d->character))
    REMOVE_BIT(PLR_FLAGS(d->character), PLR_MAILING | PLR_WRITING);
    } else {
      SEND_TO_Q("Message aborted!\r\n", d);
    }

  FREE(obj);
}


EDITFUNC(finish_desc)
{
  sstring **descr = ((sstring **) obj);

  extern char *MENU;

  if (terminator != 2) {
  ss_free(*descr);
    *descr = ss_create(str);
  }

  SEND_TO_Q(MENU, d);
  d->connected = CON_MENU;
}


EDITFUNC(finish_note)
{
  struct finish_note_data *nd = (struct finish_note_data *) obj;

  if (!find_obj(nd->id))
    SEND_TO_Q("Your paper is gone!\r\n", d);

  else {
    if (terminator != 2) {
    ss_free(*nd->desc);
      *nd->desc = ss_create(str);
    }
    SEND_TO_Q("Note written!\r\n", d);
  }

  FREE(obj);
}

EDITFUNC(finish_olc_string)
{
  struct finish_olcstr_data *od = (struct finish_olcstr_data *) obj;

//  if (od->ptr != get_olc_ptr(od->target, od->mode))
  if (od->ptr != get_current_olc_ptr(d->character, od->target, od->mode))
    SEND_TO_Q("The thing you were editing is gone!\r\n", d);
    
  else {
    if (terminator != 2) {
    if (*od->string)
      FREE(*od->string);

    *od->string = str_dup(str);
    }

    act("$n finishes editing.", TRUE, d->character, NULL, NULL, TO_ROOM);
  }
	
  FREE(obj);

}


EDITFUNC(finish_olc_sstring)
{
  struct finish_olcshr_data *od = (struct finish_olcshr_data *) obj;

//  if (od->ptr != get_olc_ptr(od->target, od->mode))
  if (od->ptr != get_current_olc_ptr(d->character, od->target, od->mode))
    SEND_TO_Q("The thing you were editing is gone!\r\n", d);

  else {
    if (terminator != 2) {
    ss_free(*od->string);
      *od->string = ss_create(str);
    }

    act("$n finishes editing.", TRUE, d->character, NULL, NULL, TO_ROOM);
  }

  FREE(obj);

}


EDITFUNC(finish_olc_door)
{
  struct finish_olcdoor_data *od = (struct finish_olcdoor_data *) obj;
  struct room_data *room;

  if ((room = (struct room_data *) get_current_olc_ptr(d->character, od->target, OLC_REDIT)) == NULL)
    SEND_TO_Q("The thing you were editing is gone!\r\n", d);
  
  else if (room->dir_option[od->door] != od->ptr)
    SEND_TO_Q("The door you were editing is gone!\r\n", d);

  else {
    if (terminator != 2) {
    if (*od->string)
      FREE(*od->string);

    *od->string = str_dup(str);
    }

    act("$n finishes editing.", TRUE, d->character, NULL, NULL, TO_ROOM);
  }
	
  FREE(obj);
}


EDITFUNC(finish_olc_exdesc)
{
  struct finish_olcextr_data *od = (struct finish_olcextr_data *) obj;
  void *ptr;
  struct extra_descr_data *extra = NULL, *i;

  if ((ptr = get_current_olc_ptr(d->character, od->target, od->mode)) == NULL) {
    SEND_TO_Q("The thing you were editing is gone!\r\n", d);
    FREE(obj);
    return;
  }
  switch (od->mode) {
  case OLC_OEDIT:
    extra = ((struct obj_data *) ptr)->ex_description;
    break;
  case OLC_REDIT:
    extra = ((struct room_data *) ptr)->ex_description;
    break;
  }

  for (i = extra; i && i != od->ptr; i = i->next);

  if (!i) {
    SEND_TO_Q("The extra description you were editing is gone!\r\n", d);
    FREE(obj);
    return;
  } else {
    if (terminator != 2) {
    ss_free(*od->string);
      *od->string = ss_create(str);
    }

    act("$n finishes editing.", TRUE, d->character, NULL, NULL, TO_ROOM);
  }
	
  FREE(obj);
}


EDITFUNC(finish_olc_farg)
{
  struct finish_olcstr_data *od = (struct finish_olcstr_data *) obj;
  int same = FALSE, exist = FALSE;

  switch (od->mode) {
  case OLC_MEDIT:
    exist = mob_index[od->target] ? TRUE : FALSE;
    break;
  case OLC_OEDIT:
    exist = obj_index[od->target] ? TRUE : FALSE;
    break;
  case OLC_REDIT:
    exist = world[od->target] ? TRUE : FALSE;
    break;
  default:
    log("SYSERR:  Illegal mode for finish_olc_farg (%d).", od->mode);
    FREE(obj);
    return;
  }

  if (!exist) {
    SEND_TO_Q("The thing you were editing is gone!\r\n", d);
    FREE(obj);
    return;
  }
  switch (od->mode) {
  case OLC_MEDIT:
    same = (od->ptr == mob_index[od->target]);
    break;
  case OLC_OEDIT:
    same = (od->ptr == obj_index[od->target]);
    break;
  case OLC_REDIT:
    same = (world[od->target] && od->ptr == world[od->target]);
    break;
  }

  if (!same)
    SEND_TO_Q("The thing you were editing is gone!\r\n", d);
    
  else {
    if (terminator != 2) {
    if (*od->string)
      FREE(*od->string);

    *od->string = str_dup(str);
    }

    act("$n finishes editing.", TRUE, d->character, NULL, NULL, TO_ROOM);
  }
	
  FREE(obj);
}


#ifdef GOT_RID_OF_IT
/* string manipulation fucntion originally by Darren Wilson */
/* (wilson@shark.cc.cc.ca.us) improved and bug fixed by Chris (zero@cnw.com) */
/* completely re-written again by M. Scott 10/15/96 (scottm@workcommn.net), */
/* substitute appearances of 'pattern' with 'replacement' in string */
/* and return the # of replacements */
int replace_str(char **string, char *pattern, char *replacement, int rep_all,
		unsigned int max_size)
{
   char *replace_buffer = NULL;
  char *flow, *jetsam, temp;
   int len, i;
   
   if ((strlen(*string) - strlen(pattern)) + strlen(replacement) > max_size)
     return -1;
   
   CREATE(replace_buffer, char, max_size);
   i = 0;
   jetsam = *string;
   flow = *string;
   *replace_buffer = '\0';
   if (rep_all) {
    while ((flow = (char *) strstr(flow, pattern)) != NULL) {
	 i++;
	 temp = *flow;
	 *flow = '\0';
	 if ((strlen(replace_buffer) + strlen(jetsam) + strlen(replacement)) > max_size) {
	    i = -1;
	    break;
	 }
	 strcat(replace_buffer, jetsam);
	 strcat(replace_buffer, replacement);
	 *flow = temp;
	 flow += strlen(pattern);
	 jetsam = flow;
      }
      strcat(replace_buffer, jetsam);
  } else {
    if ((flow = (char *) strstr(*string, pattern)) != NULL) {
	 i++;
	 flow += strlen(pattern);  
      len = ((char *) flow - (char *) *string) - strlen(pattern);
   
	 strncpy(replace_buffer, *string, len);
	 strcat(replace_buffer, replacement);
	 strcat(replace_buffer, flow);
      }
   }
  if (i == 0)
    return 0;
   if (i > 0) {
    RECREATE(*string, char, max_size);

      strcpy(*string, replace_buffer);
   }
   free(replace_buffer);
   return i;
}
#endif


/* re-formats message type formatted char * */
/* (for strings edited with d->str) (mostly olc and mail)     */
void format_text(char **ptr_string, int mode, descriptor_data * d,
		 unsigned int maxlen, unsigned int rmargin = 78)
{
  unsigned int total_chars, cap_next = TRUE, cap_next_next = FALSE;
  int return_found = 0, found_paragraph = FALSE, found_char = FALSE;
   char *flow, *start = NULL, temp;

   /* warning: do not edit messages with max_str's of over this value */
   char formatted[MAX_STRING_LENGTH];
   
  flow = *ptr_string;
   if (!flow) 
     return;

   if (IS_SET(mode, FORMAT_INDENT)) {
      strcpy(formatted, "   ");
      total_chars = 3;
  } else {
      *formatted = '\0';
      total_chars = 0;
   } 

   while (*flow != '\0') {

    found_paragraph = 0;

    found_char = FALSE;
    while (found_char == FALSE) {
      switch (*flow) {
      case '\n':
	++found_paragraph;
	++flow;
	break;
      case '\r':
      case '\f':
      case '\t':
      case '\v':
      case ' ':
	++flow;
	break;
      default:
	found_char = TRUE;
	break;
      }
    }

    if (found_paragraph >= 2) {
      found_paragraph = 0;
      strcat(formatted, "\r\n\r\n");
      total_chars = 0;
      if (IS_SET(mode, FORMAT_INDENT)) {
	strcat(formatted, "   ");
	total_chars += 3;
      }
    }
      if (*flow != '\0') {

	 start = flow++;
	 while ((*flow != '\0') &&
		(*flow != '\n') &&
		(*flow != '\r') &&
		(*flow != '\f') &&
		(*flow != '\t') &&
		(*flow != '\v') &&
		(*flow != ' ') &&
		(*flow != '.') &&
		(*flow != '?') &&
	     (*flow != '!'))
	flow++;

	 if (cap_next_next) {
	    cap_next_next = FALSE;
	    cap_next = TRUE;
	 }
      /* this is so that if we stopped on a sentance .. we move off the sentance delim. */
	 while ((*flow == '.') || (*flow == '!') || (*flow == '?')) {
	    cap_next_next = TRUE;
	    flow++;
	 }
	 
	 temp = *flow;
	 *flow = '\0';

      if ((total_chars + strlen(start) + 1) > rmargin) {
	    strcat(formatted, "\r\n");
	    total_chars = 0;
	 }
      if (cap_next == FALSE) {
	    if (total_chars > 0) {
	       strcat(formatted, " ");
	       total_chars++;
	    }
      } else {
	    cap_next = FALSE;
	    *start = UPPER(*start);
	 }

	 total_chars += strlen(start);
	 strcat(formatted, start);

	 *flow = temp;
      }

      if (cap_next_next) {
	 if ((total_chars + 3) > rmargin) {
	    strcat(formatted, "\r\n");
	    total_chars = 0;
      } else {
	    strcat(formatted, "  ");
	    total_chars += 2;
	 }
      }
   }
   strcat(formatted, "\r\n");

   if (strlen(formatted) > maxlen) 
    formatted[maxlen - 1] = '\0';

  strcpy(*ptr_string, formatted);
}


int string_freq(const char *searchin, const char *searchfor)
{
  int found = 0, len;
  const char *ptr;
  
  ptr = searchin;
  len = strlen(searchfor);
  
  while (strchr(ptr, *searchfor)) {
    if (strncmp(ptr, searchfor, len)) {
      ++found;
      ptr += len;
    } else {
      ++ptr;
    }
  }

  return found;
}
