/**************************************************************************
*  File: scriptcmd.c                                                      *
*  Usage: contains the command_interpreter for objects,                   *
*         object commands.                                                *
*                                                                         *
*  $Author: tlo $
*  $Date: 1999/03/06 18:56:42 $
*  $Revision: 1.1 $
**************************************************************************/

#include "conf.h"
#include "sysdep.h"

#include "structs.h"
#include "screen.h"
#include "scripts.h"
#include "index.h"
#include "utils.h"
#include "comm.h"
#include "interpreter.h"
#include "find.h"
#include "handler.h"
#include "db.h"
#include "events.h"
#include "spells.h"
#include "constants.h"

extern void die(struct char_data *ch, struct char_data *killer);
extern int find_owner_zone(int number);
extern void extract_script(struct script_data *sc);
extern void extract_trigger(trig_data * trig);
extern int remove_trigger(struct script_data *sc, char *name);
extern int find_skill_num(char *name, sh_int index = 1, sh_int top = LAST_SKILLNUM);
extern void add_var(struct trig_var_data **var_list, char *name, char *value, long context);

#define SCRIPTCMD(name)  \
    void (name)(game_data *thing, char *argument, int cmd, int subcmd, int room)

// Locates a room by number or by thing's lookup routines.
void find_script_target_room(game_data * thing, char *arg,
			     char_data ** c, obj_data ** o, room_data ** r)
{
  int tmp;
  sh_int location;

  *c = NULL;
  *o = NULL;
  *r = NULL;

  if (!*arg)
    return;

  if (isdigit(*arg) && !strchr(arg, '.')) {
    tmp = atoi(arg);
    if (real_room(tmp) < 0)
      return;
    *r = world[tmp];
  }
  else if ((*c = thing->AcquireCharacter(arg))) {
    if (PURGED(*c) || ((*c)->InRoom() == NOWHERE))
      c = NULL;
  }
  else if ((*o = thing->AcquireObject(arg))) {
    if (PURGED(*o) || ((*o)->InRoom() == NOWHERE))
      o = NULL;
  }
}


// A more object-oriented log.  Can log rooms, objects, mobs, and any other 
// type derived from game_data.
void scripterr_log(game_data * thing, char *msg)
{
  char buf[MAX_INPUT_LENGTH + 100];

  void script_log(char *msg);

  switch (thing->ClassType()) {	// Dunno why, I keep getting nulls out of datatypes_strings!
  case DATATYPE_ROOM_DATA:
    strcpy(buf, "Room");
    break;
  case DATATYPE_OBJ_DATA:
    strcpy(buf, "Obj");
    break;
  case DATATYPE_CHAR_DATA:
    strcpy(buf, "Mob");
    break;
  default:
    break;
  }  

  sprintf(buf + strlen(buf), " (%s, VNum %d): %s",
	  thing->Name(), thing->Vnum(), msg);
  script_log(buf);
}


void generic_damage_char(game_data * thing, char_data * ch, int dam)
{
  alter_hit(ch, dam);
  update_pos(ch);
  switch (GET_POS(ch)) {
  case POS_MORTALLYW:
    act("$n is mortally wounded, and will die soon, if not aided.", TRUE, ch, 0, 0, TO_ROOM);
    send_to_char("You are mortally wounded, and will die soon, if not aided.\r\n", ch);
    break;
  case POS_INCAP:
    act("$n is incapacitated and will slowly die, if not aided.", TRUE, ch, 0, 0, TO_ROOM);
    send_to_char("You are incapacitated and will slowly die, if not aided.\r\n", ch);
    break;
  case POS_STUNNED:
    act("$n is stunned, but will probably regain consciousness again.", TRUE, ch, 0, 0, TO_ROOM);
    send_to_char("You're stunned, but will probably regain consciousness again.\r\n", ch);
    break;
  case POS_DEAD:
    act("$n is dead!  R.I.P.", FALSE, ch, 0, 0, TO_ROOM);
    send_to_char("You are dead!  Sorry...\r\n", ch);
    break;

  default:                    /* >= POSITION SLEEPING */
    if (dam > (GET_MAX_HIT(ch) >> 2))
      act("That really did HURT!", FALSE, ch, 0, 0, TO_CHAR);
    if (GET_HIT(ch) < (GET_MAX_HIT(ch) >> 2)) {
      sprintf(buf2, "%sYou wish that your wounds would stop BLEEDING so much!%s\r\n",
              CCRED(ch, C_SPR), CCNRM(ch, C_SPR));
      send_to_char(buf2, ch);
    }
  }
  if (GET_POS(ch) == POS_DEAD) {
    if (!IS_NPC(ch)) {
      sprintf(buf2, "%s killed by %s at %s", ch->Name(),
              thing->Name(), world[ch->in_room]->name);
      mudlog(buf2, BRF, 0, TRUE);
    }
    die(ch, (thing->ClassType() == DATATYPE_CHAR_DATA) ?
        (char_data *) thing : NULL);
  }
}


SCRIPTCMD(do_scriptecho)
{
  skip_spaces(&argument);

  if (!*argument)
    scripterr_log(thing, "echo called with no args");

  if (world[room]->people)
    thing->Echo(argument, world[room]->people);
}


SCRIPTCMD(do_scriptasound)
{
  skip_spaces(&argument);

  if (!*argument)
    scripterr_log(thing, "asound called with no args");

  if (world[room]->people)
    thing->EchoDistance(argument, world[room]->people);
}


SCRIPTCMD(do_scriptzoneecho)
{
    int zone;
    char zone_name[MAX_INPUT_LENGTH], buf[MAX_INPUT_LENGTH], *msg;
  
    msg = any_one_arg(argument, zone_name);
    skip_spaces(&msg);

    if (!*zone_name || !*msg)
	scripterr_log(thing, "zoneecho called with too few args");

    else if ((zone = find_owner_zone(atoi(zone_name) * 100)) < 0)
	scripterr_log(thing, "zoneecho called for nonexistant zone");

    else {
	sprintf(buf, "%s\r\n", msg);
	send_to_zone(buf, zone);
    }
}


SCRIPTCMD(do_scriptforce)
{
  char_data *ch, *next_ch;
  obj_data *obj, *next_obj;
  char *line, arg1[MAX_INPUT_LENGTH];
  int afk_flag;

  line = one_argument(argument, arg1);

  if (!*arg1 || !*line) {
    scripterr_log(thing, "force called with too few args");
    return;
  }

  switch (*arg1) {
  case 'a':
    if (!str_cmp(arg1, "all")) {
      for (obj = world[room]->contents; obj; obj = next_obj) {
	next_obj = obj->next_content;
	obj->Command(line);
      }
    }
  case 'm':
    if (!str_cmp(arg1, "mobs")) {
      for (ch = world[room]->people; ch; ch = next_ch) {
	next_ch = ch->next_in_room;
	if (!IS_GOD(ch)) {
	  afk_flag = PLR_FLAGS(ch);
	  REMOVE_BIT(PLR_FLAGS(ch), PLR_AFK);
	  ch->Command(line);
	  PLR_FLAGS(ch) = afk_flag;
	}
      }
      return;
    }
    break;
  case 'o':
    if (!str_cmp(arg1, "objs")) {
      for (obj = world[room]->contents; obj; obj = next_obj) {
	next_obj = obj->next_content;
	obj->Command(line);
      }
      return;
    }
    break;
  case 'r':
    if (!str_cmp(arg1, "room")) {
      world[room]->Command(line);
      return;
    }
    break;
  }

  ch = thing->AcquireCharacterRoom(arg1);
  if (ch) {
    if (!IS_GOD(ch)) {
      afk_flag = PLR_FLAGS(ch);
      REMOVE_BIT(PLR_FLAGS(ch), PLR_AFK);
      ch->Command(line);
      PLR_FLAGS(ch) = afk_flag;
    }
  }
  else {
    obj = thing->AcquireObjectList(arg1, world[room]->contents);
    if (obj) {
      obj->Command(line);
    }
  }

  if (!ch && !obj) {
    scripterr_log(thing, "force: no target found");
  }
}


SCRIPTCMD(do_scriptsend)
{
  char_data *ch;
  char buf[MAX_INPUT_LENGTH], *msg;

  msg = any_one_arg(argument, buf);

  if (!*buf) {
    scripterr_log(thing, "send called with no args");
    return;
  }

  skip_spaces(&msg);

  if (!*msg) {
    scripterr_log(thing, "send called without a message");
    return;
  }

  ch = thing->AcquireCharacterRoom(buf);

  if (ch) {
    if (subcmd == SCMD_SEND)
      thing->EchoAt(msg, ch);
    else if (subcmd == SCMD_ECHOAROUND)
      thing->EchoAround(msg, (game_data *) ch, world[ch->InRoom()]->people);
  }
  else
    scripterr_log(thing, "no target found for send");
}


SCRIPTCMD(do_scriptat)
{
    char arg[MAX_INPUT_LENGTH], *p;
    char_data *c;
    obj_data *o;
    room_data *r;
    int target = NOWHERE;
  
    p = one_argument(argument, arg);
  
    if (!*arg) {
	scripterr_log(thing, "at: bad argument");
	return;
    }

    find_script_target_room(thing, arg, &c, &o, &r);

    if (c)
      target = c->InRoom();
    else if (o)
      target = o->InRoom();
    else if (r)
      target = r->Vnum();

    if (target == NOWHERE) {
      scripterr_log(thing, "at: target not found");
      return;
    }

    /* a room has been found.  Check for permission */
    if (ROOM_FLAGGED(target, ROOM_GODROOM) ||
        ROOM_FLAGGED(target, ROOM_HOUSE) ||
        ROOM_FLAGGED(target, ROOM_PRIVATE)) {
      scripterr_log(thing, "at target is a private room");
      return;
    }

    thing->AtCommand(target, p);
}


/* increases the target's exp */
SCRIPTCMD(do_scriptexp)
{
  char_data *ch;
  char name[MAX_INPUT_LENGTH], amount[MAX_INPUT_LENGTH];

  extern int exp_from_level(long chlevel, long fromlevel, int modifier);

  two_arguments(argument, name, amount);

  if (!*name || !*amount) {
    scripterr_log(thing, "exp: too few arguments");
    return;
  }
  if ((ch = thing->AcquireCharacterRoom(name)))
    gain_exp(ch, exp_from_level(GET_LEVEL(ch), atoi(amount), 0));
  else {
    scripterr_log(thing, "exp: target not found");
    return;
  }
}


/* transform into a different object */
/* note: this shouldn't be used with containers unless both objects */
/* are containers! */


SCRIPTCMD(do_scripttransform)
{
  char arg[MAX_INPUT_LENGTH];
  extern int obj_transform(obj_data * obj, int newnumber);

  one_argument(argument, arg);

  if (!*arg) {
    scripterr_log(thing, "transform: missing argument");
    return;
  }
  else if (!isdigit(*arg)) {
    scripterr_log(thing, "transform: bad argument");
    return;
  }

  switch (thing->ClassType()) {
  case DATATYPE_OBJ_DATA:
    obj_transform((obj_data *) thing, atoi(arg));
    break;
  case DATATYPE_ROOM_DATA:
  case DATATYPE_CHAR_DATA:
  default:
    scripterr_log(thing, "transform: unsupported type");
    return;
  }
}


/* purge all objects an npcs in room, or specified object or mob */
SCRIPTCMD(do_scriptpurge)
{
  char arg[MAX_INPUT_LENGTH];
  char_data *ch, *next_ch;
  obj_data *o, *next_obj;

  one_argument(argument, arg);

  if (!*arg) {
    for (ch = world[room]->people; ch; ch = next_ch) {
      next_ch = ch->next_in_room;
      if ((ch != thing) && IS_NPC(ch))
	extract_char(ch);
    }

    for (o = world[room]->contents; o; o = next_obj) {
      next_obj = o->next_content;
      if (o != thing)
	extract_obj(o);
    }
    return;
  }

  ch = thing->AcquireCharacterRoom(arg);
  if (ch && thing != ch) {
    if (!IS_NPC(ch)) {
      scripterr_log(thing, "purge: purging a PC");
      return;
    }
    extract_char(ch);
  }
  else {
    o = thing->AcquireObjectList(arg, world[room]->contents);
    if (o && thing != o) {
      extract_obj(o);
    }
  }

  if (!ch && !o) {
    scripterr_log(thing, "purge: no target found");
  }
}


SCRIPTCMD(do_scriptteleport)
{
  char_data *ch, *next_ch, *c;
  obj_data *obj, *next_obj, *o;
  room_data *r;
  int target = NOWHERE;
  char arg1[MAX_INPUT_LENGTH], arg2[MAX_INPUT_LENGTH];
  bool objs = FALSE, mobs = FALSE;

  two_arguments(argument, arg1, arg2);

  if (!*arg1 || !*arg2) {
    scripterr_log(thing, "teleport called with too few args");
    return;
  }

  find_script_target_room(thing, arg2, &c, &o, &r);

  if (c)
    target = c->InRoom();
  else if (o)
    target = o->InRoom();
  else if (r)
    target = r->Vnum();

  if (target == NOWHERE) {
    scripterr_log(thing, "teleport target not found");
    return;
  }

  /* a room has been found.  Check for permission */
  if (ROOM_FLAGGED(target, ROOM_GODROOM) ||
      ROOM_FLAGGED(target, ROOM_HOUSE) ||
      ROOM_FLAGGED(target, ROOM_PRIVATE)) {
    scripterr_log(thing, "teleport target is a private room");
    return;
  }

  if (!strcmp(arg1, "all")) {
    objs = TRUE;
    mobs = TRUE;
  }
  else if (!strcmp(arg1, "mobs")) {
    mobs = TRUE;
  }
  else if (!strcmp(arg1, "objs")) {
    objs = TRUE;
  }
  
  if (mobs) {
    for (ch = world[room]->people; ch; ch = next_ch) {
      next_ch = ch->next_in_room;
      if (ch == thing)
	continue;
      char_from_room(ch);
      char_to_room(ch, target);
    }
  }

  if (objs) {
    for (obj = world[room]->contents; obj; obj = next_obj) {
      next_obj = obj->next_content;
      if (obj == thing)
	continue;
      obj_from_room(obj);
      obj_to_room(obj, target);
    }
  }

  if (!mobs && !objs) {
    if ((ch = thing->AcquireCharacterRoom(arg1))) {
      char_from_room(ch);
      char_to_room(ch, target);
    }
    else if ((obj = thing->AcquireObjectList(arg1, world[room]->contents))) {
      obj_from_room(obj);
      obj_to_room(obj, target);
    }
  }

}


SCRIPTCMD(do_scriptload)
{
  char arg1[MAX_INPUT_LENGTH], arg2[MAX_INPUT_LENGTH], *p;
  int number = 0;
  char_data *mob, *target;
  obj_data *object;

  p = two_arguments(argument, arg1, arg2);

  if (!*arg1 || !*arg2 || !is_number(arg2) || ((number = atoi(arg2)) < 0)) {
    scripterr_log(thing, "load: bad syntax");
    return;
  }
  if (!strcmp(arg1, "mob")) {
    if ((mob = read_mobile(number)) == NULL) {
      scripterr_log(thing, "load: bad mob vnum");
      return;
    }
    char_to_room(mob, room);
    load_mtrigger(mob);
  }
  else if (!strcmp(arg1, "obj")) {
    if ((object = read_object(number)) == NULL) {
      scripterr_log(thing, "load: bad object vnum");
      return;
    }
    skip_spaces(&p);
    if (*p) {
      target = thing->AcquireCharacterRoom(p);
    } else if (thing->ClassType() == DATATYPE_CHAR_DATA) {
      target = (char_data *) thing;
    }
    
    if (CAN_WEAR(object, ITEM_WEAR_TAKE)) {
      obj_to_char(object, target);
    } else {
      obj_to_room(object, room);
    } 

    load_otrigger(object);
  }
  else
    scripterr_log(thing, "load: bad type");
}

SCRIPTCMD(do_scriptdamage)
{
  char name[MAX_INPUT_LENGTH], amount[MAX_INPUT_LENGTH];
  int dam = 0;
  char_data *ch;

  two_arguments(argument, name, amount);

  if (!*name || !*amount || !isdigit(*amount)) {
    scripterr_log(thing, "damage: bad syntax");
    return;
  }

  dam = atoi(amount);

  if ((ch = thing->AcquireCharacterRoom(name))) {
    if (PURGED(ch)) {
      return;
    }
    if (IS_IMMORTAL(ch)) {
      send_to_char("Being the cool immortal you are, you sidestep a trap, obviously placed to kill you.", ch);
      return;
    }
    generic_damage_char(thing, ch, dam);
  }
  else
    scripterr_log(thing, "damage: target not found");
}


SCRIPTCMD(do_scriptskillset)
{
  extern char *spells[];
  char_data *vict;
  int skill, value, theory, i, qend, add;
  char help[MAX_INPUT_LENGTH];

  argument = one_argument(argument, buf);

  if (!*buf) {			/* no arguments. print an informative text */
    scripterr_log(thing, "skillset: no arguments");
    return;
  }

  if (!(vict = thing->AcquireCharacterRoom(buf))) {
    scripterr_log(thing, "skillset: target not found");
    return;
  }

  skip_spaces(&argument);

  /* If there is no chars in argument */
  if (!*argument) {
    scripterr_log(thing, "skillset: no skill specified");
    return;
  }
  if (*argument != '\'') {
    scripterr_log(thing, "skillset: skill must be enclosed in ''");
    return;
  }
  /* Locate the last quote && lowercase the magic words (if any) */

  for (qend = 1; *(argument + qend) && (*(argument + qend) != '\''); qend++)
    *(argument + qend) = LOWER(*(argument + qend));

  if (*(argument + qend) != '\'') {
    scripterr_log(thing, "skillset: skill must be enclosed in ''");
    return;
  }
  strcpy(help, (argument + 1));
  help[qend - 1] = '\0';
  if ((skill = find_skill_num(help, FIRST_SKILLNUM, LAST_SKILL)) <= 0) {
    scripterr_log(thing, "skillset: unrecognized skill");
    return;
  }
  argument += qend + 1;		/* skip to next parameter */
  argument = one_argument(argument, buf);

  if (!*buf) {
    scripterr_log(thing, "skillset: no points value");
    return;
  }

  value = atoi(buf);
  value = MAX(MIN(value, 250), 0);
  if (*argument) {
    theory = atoi(argument);
    theory = MAX(MIN(value, 250), 0);
  }

  add = CHAR_SKILL(vict).SetSkill(skill, NULL, value, theory);
  if (!SKL_IS_CSPHERE(skill))
    GET_TOTALPTS(vict) += add;

}


SCRIPTCMD(do_scriptsetchar)
{
  int field, setto, setdiff;
  char fieldarg[MAX_INPUT_LENGTH];
  char_data *ch;

  const char *wset_table[] =
  {
    "hit",
    "maxhit",
    "mana",
    "maxmana",
    "move",
    "maxmove",
    "alignment",
    "lawfulness",
    "gold",
    "var",
    "\n"
  };

  half_chop(argument, buf1, argument);
  
  ch = thing->AcquireCharacterRoom(buf1);

  if (IS_IMMORTAL(ch))
    return;

  if (PURGED(ch) || (IN_ROOM(ch) == NOWHERE))
    return;

  half_chop(argument, fieldarg, argument);
  
  field = search_block(fieldarg, wset_table, TRUE);
  
  // Didn't find the argument?
  if (field == -1) {
    scripterr_log(thing, "set:  Invalid parameter field");
    return;
  }
  half_chop(argument, buf1, argument);
  setto = atoi(buf1);
  
  switch (field) {
  case 0: // hit
    setto = MAX(setto, -10);
    setdiff = GET_HIT(ch) - setto;
    generic_damage_char(thing, ch, setdiff);
	    break;
  case 1: // maxhit
    setto = MAX(setto, -10);
    setdiff = GET_HIT(ch) - setto;
    generic_damage_char(thing, ch, setdiff);
    GET_MAX_HIT(ch) = setto;
    	    break;
  case 2: // mana
    setto = MAX(setto, 0);
    setdiff = GET_MANA(ch) - setto;
    alter_mana(ch, setdiff);
    	    break;
  case 3: // maxmana
    setto = MAX(setto, 0);
    setdiff = GET_MANA(ch) - setto;
    alter_mana(ch, setdiff);
    GET_MAX_MANA(ch) = setto;
    	    break;
  case 4: // move
    setto = MAX(setto, 0);
    setdiff = GET_MOVE(ch) - setto;
    alter_move(ch, setdiff);
    break;
  case 5: // maxmove
    setto = MAX(setto, 0);
    setdiff = GET_MOVE(ch) - setto;
    alter_move(ch, setdiff);
    GET_MAX_MOVE(ch) = setto;
    break;
  case 6: // alignment
    setto = MAX(MIN(setto, 1000), -1000);
    GET_ALIGNMENT(ch) = setto;
    break;
  case 7: // lawfulness
    setto = MAX(MIN(setto, 1000), -1000);
    GET_LAWFULNESS(ch) = setto;
    break;
  case 8: // gold
    GET_GOLD(ch) = setto;
    break;
  case 9: // var
    if (!SCRIPT(thing))
      SCRIPT(thing) = new script_data;
    half_chop(argument, buf2, argument);
    add_var(&(SCRIPT(thing)->global_vars), buf1, buf2, 0);
    break;
  default:
    sprintf(buf, "set:  Invalid field (%s) in switch", fieldarg);
    scripterr_log(thing, buf);
    return;
  }
}


SCRIPTCMD(do_scriptattach)
{
  char_data *targ_ch;
  obj_data *targ_obj;
  room_data *targ_room;
  script_data **targ_script = NULL;
  trig_data *trig, *t;

  char *p = argument;
  char type[MAX_INPUT_LENGTH], name[MAX_INPUT_LENGTH], trigger[MAX_INPUT_LENGTH];

  int tn, rn;

  p = two_arguments(argument, type, name);
  one_argument(p, trigger);

  if (!*type || !*trigger || !*name) {
    if (!*type)
      scripterr_log(thing, "attach: missing mob|obj|room");
    if (!*name)
      scripterr_log(thing, "attach: missing target name/ID");
    if (!*trigger)
      scripterr_log(thing, "attach: missing trigger vnum");
    return;   
  }

  tn = atoi(trigger);
  rn = trig_index[tn] ? tn : -1;

  if (rn <= 0) {
    sprintf(buf, "attach: trigger %d does not exist.", tn);
    scripterr_log(thing, buf);
    return;
  }

  if (!str_cmp(type, "mob")) {
    if ((targ_ch = thing->AcquireCharacterRoom(name))) {
      if (!IS_NPC(targ_ch))
        return;
        
      targ_script = &SCRIPT(targ_ch);
    } else {
      sprintf(buf, "attach: target mob '%s' does not exist.", name);
      scripterr_log(thing, buf);
      return;
    }
  }
  else if (!str_cmp(type, "obj")) {
    if ((targ_obj = thing->AcquireObjectList(name, world[room]->contents))) {
      targ_script = &SCRIPT(targ_obj);
    } else {
      sprintf(buf, "attach: target obj '%s' does not exist.", name);
      scripterr_log(thing, buf);
      return;
    }
  }
  else if (!str_cmp(type, "room")) {
    if ((targ_room = thing->AcquireRoom(name))) {
      targ_script = &SCRIPT(targ_room);
    } else {
      sprintf(buf, "attach: target room '%s' does not exist.", name);
      scripterr_log(thing, buf);
      return;
    }
  }
  
  if (!targ_script)
    return;

  if (*targ_script) {
    for (t = TRIGGERS(*targ_script); t; t = t->next) {
      if (GET_TRIG_VNUM(t) == tn) {
        // sprintf(buf, "attach: trigger '%d' already attached to target.", tn);
        // scripterr_log(thing, buf);
        return;
      }
    }
  } else {
    *targ_script = new script_data;
  }
  
  trig = read_trigger(rn);
  add_trigger(*targ_script, trig, -1);
  
#ifdef JD_TESTING_OUT
          sprintf(buf, "attach: Trigger %d (%s) attached to %s.\r\n",
                  tn, GET_TRIGGER(trig), name)
          scripterr_log(thing, buf);
#endif
}


SCRIPTCMD(do_scriptdetach) 
{
  char_data *targ_ch = NULL;
  obj_data *targ_obj = NULL;
  room_data *targ_room = NULL;
  script_data **targ_script = NULL;
  trig_data *trig, *prev_trig = NULL;
  int tn;

  char type[MAX_INPUT_LENGTH], name[MAX_INPUT_LENGTH], trigger[MAX_INPUT_LENGTH];

  argument = two_arguments(argument, type, name);
  one_argument(argument, trigger);

  if (!*type || !*name || !*trigger) {
    if (!*type)
      scripterr_log(thing, "detach: missing mob|obj|room");
    if (!*name)
      scripterr_log(thing, "detach: missing target name/ID");
    if (!*trigger)
      scripterr_log(thing, "detach: missing trigger vnum");
    return;   
  }

  if (!str_cmp(type, "mob")) {
    if ((targ_ch = thing->AcquireCharacterRoom(name))) {
      if (!IS_NPC(targ_ch))
        return;
        
      targ_script = &SCRIPT(targ_ch);
    } else {
      sprintf(buf, "detach: target mob '%s' does not exist.", name);
      scripterr_log(thing, buf);
      return;
    }
  }
  else if (!str_cmp(type, "obj")) {
    if ((targ_obj = thing->AcquireObjectList(type, world[room]->contents))) {
      targ_script = &SCRIPT(targ_obj);
    } else {
      sprintf(buf, "detach: target obj '%s' does not exist.", name);
      scripterr_log(thing, buf);
      return;
    }
  }
  else if (!str_cmp(arg, "room")) {
    if ((targ_room = thing->AcquireRoom(name))) {
      targ_script = &SCRIPT(targ_room);
    } else {
      sprintf(buf, "detach: target room '%s' does not exist.", name);
      scripterr_log(thing, buf);
      return;
    }
  }

  if (targ_script && !*targ_script) {
    // sprintf(buf, "detach: Target '%s' does not have any triggers to remove.", name);
    // scripterr_log(thing, buf);
    return;
  }
  
  if (!str_cmp(trigger, "all")) {
    extract_script(*targ_script);
    *targ_script = NULL;
  } else {
    tn = atoi(trigger);
  
    trig = TRIGGERS(*targ_script);
    while (trig) {
      if (GET_TRIG_VNUM(trig) == tn)
        break;
      prev_trig = trig;
      trig = trig->next;
    }
    
    if (trig) {
      if (prev_trig) {
        prev_trig->next = trig->next;
      } else {
        TRIGGERS(*targ_script) = trig->next;
      }
      extract_trigger(trig);
    }
  
    if (!TRIGGERS(*targ_script)) {
      extract_script(*targ_script);
      *targ_script = NULL;
    }
  }
}



const struct script_command_info script_cmd_info[] =
{
  {"RESERVED", 0, 0},		/* this must be first -- for specprocs */

  {"asound", do_scriptasound, 0},
  {"echoaround", do_scriptsend, SCMD_ECHOAROUND},
  {"echo", do_scriptecho, 0},
  {"exp", do_scriptexp, 0},
  {"force", do_scriptforce, 0},
  {"load", do_scriptload, 0},
  {"purge", do_scriptpurge, 0},
  {"send", do_scriptsend, SCMD_SEND},
  {"teleport", do_scriptteleport, 0},
  {"zoneecho", do_scriptzoneecho, 0},
  {"damage", do_scriptdamage, 0},
  {"skillset", do_scriptskillset, 0},
  {"setchar", do_scriptsetchar, 0},
  {"attach", do_scriptattach, 0},
  {"detach", do_scriptdetach, 0},
  {"at", do_scriptat, 0},
  {"transform", do_scripttransform, 0},
  {"\n", 0, 0}			/* this must be last */
};


int genscript_command_interpreter(game_data * thing, char *argument, int cmd = 0)
{
  int room, mode = cmd;		// Mode preserves cmd for checking for -1,
  char *line, arg[MAX_INPUT_LENGTH];	//  which means that this is called from
					//  other interpreters
  room = thing->InRoom();

  if (room <= NOWHERE) {
    sprintf(buf, "Command '%s' issued in NOWHERE!", argument);
    scripterr_log(thing, buf);
    return 1;
  }

  skip_spaces(&argument);

  /* just drop to next line for hitting CR */
  if (!*argument)
    return 1;

  line = any_one_arg(argument, arg);

  if (cmd <= 0) {
    /* find the command */
    for (cmd = 0; *script_cmd_info[cmd].command != '\n'; ++cmd)
      if (!strcmp(script_cmd_info[cmd].command, arg))
	break;
  }

  if (*script_cmd_info[cmd].command == '\n') {
    if (mode != -1) {
      sprintf(buf2, "Unknown script cmd: '%s'", argument);
      scripterr_log(thing, buf2);
    }
    return 0;
  }
  else
    ((*script_cmd_info[cmd].command_pointer)
     (thing, line, cmd, script_cmd_info[cmd].subcmd, room));
     
  return 1;
}


void char_data::AtCommand(int target, char * p)
{
  int original = IN_ROOM(this);
  char_data *fighting = FIGHTING(this);
  
  char_from_room(this);
  char_to_room(this, target);

  Command(p);

  if (!PURGED(this)) {
    char_from_room(this);
    char_to_room(this, original);
    FIGHTING(this) = fighting;
  }
}


void room_data::AtCommand(int target, char * p)
{
  room_data *room;
  
  if (real_room(target)) {
    room = world[target];
    room->Command(p);
  }
}

// Look out!  This is highly experimental!
void obj_data::AtCommand(int target, char * p)
{
  int original = in_room;
  obj_data *insideobj = NULL;
  char_data *wearing = NULL, *carrying = NULL;
  sh_int wornwear;
  
  if (target == original) {
    Command(p);
    return;
  }
  
  if (worn_by != NULL) {
    wearing = worn_by;
    wornwear = worn_on;
    if (unequip_char(worn_by, worn_on) != this)
      log("SYSERR: Inconsistent worn_by and worn_on pointers!!");
  }

  if (original != NOWHERE) {
    obj_from_room(this);
  } else if (carried_by) {
    carrying = carried_by;
    obj_from_char(this);
  } else if (in_obj) {
    insideobj = in_obj;
    obj_from_obj(this);
  }
  
  obj_to_room(this, target);

  Command(p);

  if (PURGED(this) || worn_by || carried_by || in_obj)
    return;

  obj_from_room(this);

  if (wearing) {
    equip_char(wearing, this, wornwear);
  } else if (original != NOWHERE) {
    obj_to_room(this, original);
  } else if (carrying) {
    obj_to_char(this, carrying);
  } else if (in_obj) {
    obj_to_obj(this, insideobj);
  }
}
