/**************************************************************************
*  File: scripts.c                                                        *
*  Usage: contains general functions for using scripts.                   *
*                                                                         *
*                                                                         *
*  $Author: tlo $
*  $Date: 1999/03/06 18:27:07 $
*  $Revision: 6.36 $
**************************************************************************/

#include "conf.h"
#include "sysdep.h"

 
#include "structs.h"
#include "scripts.h"
#include "index.h"
#include "memory.h"
#include "utils.h"
#include "comm.h"
#include "interpreter.h"
#include "find.h"
#include "handler.h"
#include "events.h"
#include "db.h"
#include "screen.h"
#include "constants.h"
#include "sblock.h"
#include "spells.h"

#define PROTO_TRIG(num) ((trig_data *) trig_index[(num)]->proto)
void find_uid_name(char *uid, char *name);

const char * foreach_condition_strings[] = {
  "@objs",
  "@mobs",
  "@rooms",
  "zone",
  "room",
  "contents",
  "group",
  "fighting",
  "riding",
  "\n"
};


/* external vars from db.c */
extern struct script_data *purged_scripts;

/* external vars from triggers.c */
extern const char *trig_types[];
extern const char *otrig_types[];
extern const char *wtrig_types[];

/* function protos from this file */
struct cmdlist_element *find_done(struct cmdlist_element *cl);
struct cmdlist_element *\
  find_case(struct trig_data *trig, struct cmdlist_element *cl, \
          void *go, struct script_data *sc, int type, char *cond);

/* external functions */
sh_int find_target_room(char_data * ch, char *rawroomstr);
void free_varlist(struct trig_var_data *vd);
int obj_room(obj_data * obj);

void extract_trigger(trig_data * trig);
void extract_script(struct script_data *sc);
int eval_lhs_op_rhs(char *expr, char *result, void *go, struct script_data *sc,
		    trig_data * trig, int type);

struct trig_var_data * search_for_variable(struct script_data *sc, trig_data * trig, char *var, 
                       int varspace = 7);
struct trig_var_data * extract_variable(struct script_data *sc, trig_data * trig, char *var, 
                       int varspace = 7);

extern void obj_command_interpreter(obj_data * obj, char *argument, int cmd = 0);
extern void wld_command_interpreter(struct room_data *room, char *argument, int cmd = 0);
extern int genscript_command_interpreter(game_data * thing, char *argument, int cmd = 0);

/* function protos from this file */
int script_driver(void *go, trig_data * trig, int type, int mode);
int trgvar_in_room(int vnum);
void script_log(char *msg);

inline script_command_switches lookup_script_line (char *p);

/* local structures */
struct wait_event_data {
  trig_data *trigger;
  void *go;
  int type;
};


// A duplication of the shared string code; the purpose behind this is to 
// assure that scripts, which use this kind of thing, can be edited via OLC
// safely without screwing up the existing scripts out there.

#define MAX_SSCOUNT   INT_MAX

/* create a new shared command */
scommand *scommand_create(char *str)
{
  scommand *p;

  CREATE(p, scommand, 1);

  if (str) {
    p->command = new cmdlist_element(str);
    p->command->next = NULL;
  } else {
    p->command = NULL;
  }

  p->count = 1;

  return (p);
}


/* create another occurance of a shared command */
scommand *scommand_share(scommand * ss)
{
  if (ss->count < MAX_SSCOUNT) {
    ++ss->count;

    return ss;
  } else
    return scommand_create(ss->command->cmd);
}


/* free a shared command. can handle nonexistant ss */
void scommand_free(scommand * ss)
{
  struct cmdlist_element *cmd_list = NULL, *cmd_next;

  if (ss && --ss->count != 0)
    return;

  // Hopefully this puts an end to the memory leak. -DH
  cmd_list = ss->command;

  while (cmd_list) {
      cmd_next = cmd_list->next;
      if (cmd_list->cmd)
        FREE(cmd_list->cmd);
      FREE(cmd_list);
      cmd_list = cmd_next;
  }
  
  ss->command = NULL;

  FREE(ss);
  ss = NULL;
}


int trgvar_in_room(int vnum)
{
    int i = 0;
    char_data *ch;
    room_data *rm;

    if (NOWHERE == real_room(vnum)) {
	script_log("people.vnum: world[vnum] does not exist");
	return (-1);
    }
    rm = world[vnum];

    for (ch = rm->people; ch != NULL; ch = ch->next_in_room)
      ++i;

    return i;
}

/* checks every PULSE_SCRIPT for random triggers */
void script_trigger_check(void)
{
  char_data *ch;
  obj_data *obj;
  struct room_data *room = NULL;
  int nr;
  struct script_data *sc;

  // CHANGEPOINT:  These are loops through the entire lists of everything.  This could probably be
  // integrated into other similar loops.

  for (ch = character_list; ch; ch = ch->next) {
    if (SCRIPT(ch)) {
      sc = SCRIPT(ch);

      if (IS_SET(SCRIPT_TYPES(sc), WTRIG_RANDOM) &&
	  (!is_empty(world[IN_ROOM(ch)]->zone) ||
	   IS_SET(SCRIPT_TYPES(sc), WTRIG_GLOBAL)))
	random_mtrigger(ch);
    }
  }
  
  for (obj = object_list; obj; obj = obj->next) {
    if (SCRIPT(obj)) {
      sc = SCRIPT(obj);

      if (IS_SET(SCRIPT_TYPES(sc), OTRIG_RANDOM))
	random_otrigger(obj);
    }
  }

  for (nr = 0; nr <= top_of_world; nr++) {
    if (world[nr] && SCRIPT(world[nr])) {
      room = world[nr];
      sc = SCRIPT(room);
      
      if (IS_SET(SCRIPT_TYPES(sc), WTRIG_RANDOM) &&
	  (!is_empty(room->zone) ||
	   IS_SET(SCRIPT_TYPES(sc), WTRIG_GLOBAL)))
	random_wtrigger(room);
    }
  }
}


EVENTFUNC(trig_wait_event)
{
  struct wait_event_data *wait_event_obj = (struct wait_event_data *) event_obj;
  trig_data *trig;
  void *go;
  int type;

  trig = wait_event_obj->trigger;
  go = wait_event_obj->go;
  type = wait_event_obj->type;

  FREE(wait_event_obj);  
  GET_TRIG_WAIT(trig) = NULL;

  script_driver(go, trig, type, TRIG_RESTART);
}


void do_stat_trigger(struct char_data *ch, trig_data * trig)
{
  struct cmdlist_element *cmd_list;
  struct trig_var_data *tv = NULL;
  int i = 0;
  char name[MAX_INPUT_LENGTH];

  if (!trig) {
    log("SYSERR: NULL trigger passed to do_stat_trigger.");
    return;
  }

  sprintf(buf, "Name: '&cY%s&c0',  VNum: [&cG%5d&c0], RNum: [%5d]\r\n",
            GET_TRIG_NAME(trig), GET_TRIG_VNUM(trig), GET_TRIG_RNUM(trig));

  sprintbit(GET_TRIG_TYPE_MOB(trig), trig_types, buf2);
  sprintbit(GET_TRIG_TYPE_OBJ(trig), otrig_types, buf3);
  sprintbit(GET_TRIG_TYPE_WLD(trig), wtrig_types, buf4);
  
  sprintf(buf + strlen(buf), "Mob type: %s, Obj type: %s, Wld type: %s\r\n"
                             "Numeric Arg: %d, Arg list: %s\r\n",
            buf2, buf3, buf4, GET_TRIG_NARG(trig),
            ((GET_TRIG_ARG(trig) && *GET_TRIG_ARG(trig))
             ? GET_TRIG_ARG(trig) : "None"));

  for (tv = PROTO_TRIG(GET_TRIG_VNUM(trig))->var_list; tv; tv = tv->next) {
    sprintf(buf3, "%s:%ld", tv->name, tv->context);
    if (*(tv->value) == UID_CHAR) {
      find_uid_name(tv->value, name);
      sprintf(buf2, "  %-15s:  %s\r\n", tv->context ? buf3 : tv->name, name);
    } else 
      sprintf(buf2, "  %-15s:  %s\r\n", tv->context ? buf3 : tv->name, tv->value);
    strcat(buf, buf2);
  }

  strcat(buf, "&cWCommands:&c0\r\n");

  cmd_list = trig->cmdlist->command;

  while (cmd_list) {
    if (cmd_list->cmd) {
      strcat(buf, cmd_list->cmd);
      strcat(buf, "\r\n");
    }
    cmd_list = cmd_list->next;
  }

  page_string(ch->desc, buf, 1);
}


/* find the name of what the uid points to */
void find_uid_name(char *uid, char *name)
{
  char_data *ch;
  obj_data *obj;

  if ((ch = get_char(uid)))
    strcpy(name, ss_data(ch->player.name));
  else if ((obj = get_obj(uid)))
    strcpy(name, ss_data(obj->name));
  else
    sprintf(name, "uid = %s, (not found)", uid + 1);
}


/* general function to display stats on script sc */
void script_stat(char_data * ch, struct script_data *sc)
{
  struct trig_var_data *tv;
  trig_data *t;
  char name[MAX_INPUT_LENGTH], namebuf[MAX_INPUT_LENGTH];

  switch (sc->attach_type) {
  case MOB_TRIGGER:
    sprintbit(sc->types, trig_types, buf2);
    break;
  case OBJ_TRIGGER:
    sprintbit(sc->types, otrig_types, buf2);
    break;
  case WLD_TRIGGER:
    sprintbit(sc->types, wtrig_types, buf2);
    break;
  default:
    strcpy(buf2, "&cRError!  Wrong attach type!");
    break;
  }
  
  sprintf(buf, "Global variables: %s\r\n"
               "Global context: %ld\r\n"
               "Attach type: %d, trigger type %s\r\n", 
               sc->global_vars ? "" : "None", sc->context,
               sc->attach_type, buf2);
  send_to_char(buf, ch);

  for (tv = sc->global_vars; tv; tv = tv->next) {
    sprintf(namebuf, "%s:%ld", tv->name, tv->context);
    if (*(tv->value) == UID_CHAR) {
      find_uid_name(tv->value, name);
      sprintf(buf, "  %-15s:  %s\r\n", tv->context ? namebuf : tv->name, name);
    } else 
      sprintf(buf, "  %-15s:  %s\r\n", tv->context ? namebuf : tv->name, tv->value);
    send_to_char(buf, ch);
  }

  for (t = TRIGGERS(sc); t; t = t->next) {
    sprintf(buf, "\r\n  Trigger: &cY%s&c0, VNum: [&cG%5d&c0]\r\n",
	    GET_TRIG_NAME(t), GET_TRIG_VNUM(t));
    send_to_char(buf, ch);

    sprintf(buf, "  Numeric Arg: %d, Arg list: %s\r\n", 
	    GET_TRIG_NARG(t), 
	    ((GET_TRIG_ARG(t) && *GET_TRIG_ARG(t)) ? GET_TRIG_ARG(t) :
	     "None"));
    send_to_char(buf, ch);

    if (GET_TRIG_WAIT(t)) {
      sprintf(buf, "    Wait: %ld, Current line: %s\r\n",
	      time_to_event(GET_TRIG_WAIT(t)), t->curr_state->cmd);
      send_to_char(buf, ch);

      sprintf(buf, "  Variables: %s\r\n", GET_TRIG_VARS(t) ? "" : "None");
      send_to_char(buf, ch);

      for (tv = GET_TRIG_VARS(t); tv; tv = tv->next) {
	if (*(tv->value) == UID_CHAR) {
	  find_uid_name(tv->value, name);
	  sprintf(buf, "    %15s:  %s\r\n", tv->name, name);
	} else 
	  sprintf(buf, "    %15s:  %s\r\n", tv->name, tv->value);
	send_to_char(buf, ch);
      }
      for (tv = PROTO_TRIG(GET_TRIG_VNUM(t))->var_list; tv; tv = tv->next) {
	if (*(tv->value) == UID_CHAR) {
	  find_uid_name(tv->value, name);
	  sprintf(buf, "    %15s:  %s (shared)\r\n", tv->name, name);
	} else 
	  sprintf(buf, "    %15s:  %s (shared)\r\n", tv->name, tv->value);
	send_to_char(buf, ch);
      }
    }
  }  
}


void do_sstat_room(char_data * ch)
{
  struct room_data *rm = world[ch->in_room];

  if (!SCRIPT(rm)) {
    act("This room does not have a script.", FALSE, ch, NULL, NULL, TO_CHAR);
    return;
  }

  sprintf(buf, "Room name: %s%s%s\r\n", CCCYN(ch, C_NRM), rm->name,
          CCNRM(ch, C_NRM));
  send_to_char(buf, ch);

  sprintf(buf, "Zone: [%3d], VNum: [%s%5d%s]\r\n",
          rm->zone, CCGRN(ch, C_NRM), rm->number, CCNRM(ch, C_NRM));
  send_to_char(buf, ch);

  script_stat(ch, SCRIPT(rm));
}


void do_sstat_object(char_data * ch, obj_data * j)
{
  if (!SCRIPT(j)) {
    act("$p does not have a script.", FALSE, ch, j, NULL, TO_CHAR);
    return;
  }

  sprintf(buf, "Name: '%s%s%s', Aliases: %s\r\n", CCYEL(ch, C_NRM),
          ((ss_data(j->short_description)) ? ss_data(j->short_description) : "<None>"),
          CCNRM(ch, C_NRM), ss_data(j->name));
  send_to_char(buf, ch);

  sprinttype(GET_OBJ_TYPE(j), item_types, buf1);
  sprintf(buf, "VNum: [%s%5d%s], Type: %s\r\n",
	  CCGRN(ch, C_NRM), GET_OBJ_NUM(j), CCNRM(ch, C_NRM), buf1);
  send_to_char(buf, ch);

  script_stat(ch, SCRIPT(j));
}


void do_sstat_character(char_data * ch, char_data * k)
{
  if (!SCRIPT(k)) {
    act("$N does not have a script.", FALSE, ch, NULL, k, TO_CHAR);
    return;
  }

  sprintf(buf, "Name: '%s%s%s', Alias: %s\r\n",
	  CCYEL(ch, C_NRM), GET_SHORT(k), CCNRM(ch, C_NRM), GET_NAME(k));
  send_to_char(buf, ch);
  
  sprintf(buf, "VNum: [&cG%5d&c0], In room [&cG%5d&c0]\r\n",
	  GET_MOB_NUM(k),  world[k->in_room]->number);
  send_to_char(buf, ch);
  
  script_stat(ch, SCRIPT(k));
}


/*
 * adds the trigger t to script sc in in location loc.  loc = -1 means
 * add to the end, loc = 0 means add before all other triggers.
 */
void add_trigger(struct script_data *sc, trig_data * t, int loc)
{
  trig_data *i;
  int n;

  for (n = loc, i = TRIGGERS(sc); i && i->next && (n != 0); n--, i = i->next);

  if (!loc) {
    t->next = TRIGGERS(sc);
    TRIGGERS(sc) = t;
  } else if (!i)
    TRIGGERS(sc) = t;
  else {
    t->next = i->next;
    i->next = t;
  }

  switch (sc->attach_type) {
  case MOB_TRIGGER:
    SCRIPT_TYPES(sc) |= GET_TRIG_TYPE_MOB(t);
    break;
  case OBJ_TRIGGER:
    SCRIPT_TYPES(sc) |= GET_TRIG_TYPE_OBJ(t);
    break;
  case WLD_TRIGGER:
    SCRIPT_TYPES(sc) |= GET_TRIG_TYPE_WLD(t);
    break;
  default:
    sprintf(buf, "SCRIPTERR:  Invalid attach type %d in add_trigger", sc->attach_type);
    log(buf);
    break;
  }
}


ACMD(do_attach) 
{
  char_data *victim;
  obj_data *object;
  trig_data *trig;
  char targ_name[MAX_INPUT_LENGTH], trig_name[MAX_INPUT_LENGTH];
  char loc_name[MAX_INPUT_LENGTH];
  int loc, room, tn, rn;


  argument = two_arguments(argument, arg, trig_name);
  two_arguments(argument, targ_name, loc_name);

  if (!*arg || !*targ_name || !*trig_name) {
    send_to_char("Usage: attach { mtr | otr | wtr } { trigger } { name } [ location ]\r\n", ch);
    return;
  }
  tn = atoi(trig_name);
  loc = (*loc_name) ? atoi(loc_name) : -1;
  
  if (is_abbrev(arg, "mtr")) {
    if ((victim = get_char_vis(ch, targ_name))) {
      if (IS_NPC(victim))  {
	
	/* have a valid mob, now get trigger */
	rn = trig_index[tn] ? tn : -1;
	if ((rn >= 0) && (trig = read_trigger(rn))) {

	  if (!SCRIPT(victim))
	    // CREATE(SCRIPT(victim), struct script_data, 1);
	    SCRIPT(victim) = new script_data;

          SCRIPT(victim)->attach_type = MOB_TRIGGER;
	  add_trigger(SCRIPT(victim), trig, loc);
	  
	  sprintf(buf, "Trigger %d (%s) attached to %s.\r\n",
		  tn, GET_TRIG_NAME(trig), GET_SHORT(victim));
	  send_to_char(buf, ch);
	} else
	  send_to_char("That trigger does not exist.\r\n", ch);
      } else
	send_to_char("Players can't have scripts.\r\n", ch);
    } else
      send_to_char("That mob does not exist.\r\n", ch);
  } else if (is_abbrev(arg, "otr")) {
    if ((object = get_obj_vis(ch, targ_name))) {
	
      /* have a valid obj, now get trigger */
      rn = trig_index[tn] ? tn : -1;
      if ((rn >= 0) && (trig = read_trigger(rn))) {
	
	  if (!SCRIPT(object))
	    // CREATE(SCRIPT(object), struct script_data, 1);
	    SCRIPT(object) = new script_data;
            
          SCRIPT(object)->attach_type = OBJ_TRIGGER;
	  add_trigger(SCRIPT(object), trig, loc);
	  
	  sprintf(buf, "Trigger %d (%s) attached to %s.\r\n",
		  tn, GET_TRIG_NAME(trig), 
		  ((ss_data(object->short_description)) ?
		   ss_data(object->short_description) : ss_data(object->name)));
	  send_to_char(buf, ch);
      } else
	send_to_char("That trigger does not exist.\r\n", ch);
    } else
      send_to_char("That object does not exist.\r\n", ch); 
  } else if (is_abbrev(arg, "wtr")) {
    if (isdigit(*targ_name) && !strchr(targ_name, '.')) {
      if ((room = find_target_room(ch, targ_name)) != NOWHERE) {
	
	/* have a valid room, now get trigger */
	rn = trig_index[tn] ? tn : -1;
	if ((rn >= 0) && (trig = read_trigger(rn))) {

	  if (!(world[room]->script))
	    // CREATE(world[room]->script, struct script_data, 1);
	    world[room]->script = new script_data;

          world[room]->script->attach_type = WLD_TRIGGER;
	  add_trigger(world[room]->script, trig, loc);
	  
	  sprintf(buf, "Trigger %d (%s) attached to room %d.\r\n",
		  tn, GET_TRIG_NAME(trig), world[room]->number);
	  send_to_char(buf, ch);
	} else
	  send_to_char("That trigger does not exist.\r\n", ch);
      }
    } else
      send_to_char("You need to supply a room number.\r\n", ch);
  } else
    send_to_char("Please specify 'mtr', otr', or 'wtr'.\r\n", ch);
}


/*
 *  removes the trigger specified by name, and the script of o if
 *  it removes the last trigger.  name can either be a number, or
 *  a 'silly' name for the trigger, including things like 2.beggar-death.
 *  returns 0 if did not find the trigger, otherwise 1.  If it matters,
 *  you might need to check to see if all the triggers were removed after
 *  this function returns, in order to remove the script.
 */
int remove_trigger(struct script_data *sc, char *name)
{
  trig_data *i, *j;
  int num = 0, string = FALSE, n;
  char *cname;

  
  if (!sc)
    return 0;

  if ((cname = strstr(name, ".")) || (!isdigit(*name))) {
    string = TRUE;
    if (cname) {
      *cname = '\0';
      num = atoi(name);
      name = ++cname;
    }
  } else
    num = atoi(name);
  
  for (n = 0, j = NULL, i = TRIGGERS(sc); i; j = i, i = i->next) {
    if (string) {
      if (silly_isname(name, GET_TRIG_NAME(i)))
	if (++n >= num)
	  break;
    } else if (++n >= num)
      break;
  }

  if (i) {
    if (j) {
      j->next = i->next;
      extract_trigger(i);
    } 

    /* this was the first trigger */
    else {
      TRIGGERS(sc) = i->next;
      extract_trigger(i);
    }

    /* update the script type bitvector */
    SCRIPT_TYPES(sc) = 0;
    for (i = TRIGGERS(sc); i; i = i->next)
      switch (sc->attach_type) {
      case MOB_TRIGGER:
        SCRIPT_TYPES(sc) |= GET_TRIG_TYPE_MOB(i);
        break;
      case OBJ_TRIGGER:
        SCRIPT_TYPES(sc) |= GET_TRIG_TYPE_OBJ(i);
        break;
      case WLD_TRIGGER:
        SCRIPT_TYPES(sc) |= GET_TRIG_TYPE_WLD(i);
        break;
      default:
        log("SCRIPTERR:  Invalid attach type in remove_trigger");
        break;
      }

    return 1;
  } else
    return 0;
}


ACMD(do_detach) 
{
  char_data *victim = NULL;
  obj_data *object = NULL;
  struct room_data *room;
  char arg1[MAX_INPUT_LENGTH], arg2[MAX_INPUT_LENGTH], arg3[MAX_INPUT_LENGTH];
  char *trigger = 0;
  int tmp;


  argument = two_arguments(argument, arg1, arg2);
  one_argument(argument, arg3);

  if (!*arg1 || !*arg2) {
    send_to_char("Usage: detach [ mob | object ] { target } { trigger | 'all' }\r\n", ch);
    return;
  }

  if (!str_cmp(arg1, "room")) {
    room = world[IN_ROOM(ch)];
    if (!SCRIPT(room)) 
      send_to_char("This room does not have any triggers.\r\n", ch);
    else if (!str_cmp(arg2, "all")) {
      extract_script(SCRIPT(room));
      SCRIPT(room) = NULL;
      send_to_char("All triggers removed from room.\r\n", ch);
    } else if (TRIGGERS(SCRIPT(room)) && remove_trigger(SCRIPT(room), arg2)) {
      send_to_char("Trigger removed.\r\n", ch);
      if (!TRIGGERS(SCRIPT(room))) {
	extract_script(SCRIPT(room));
	SCRIPT(room) = NULL;
      }
    } else
      send_to_char("That trigger was not found.\r\n", ch);
  } else {
    if (is_abbrev(arg1, "mob")) {
      if (!(victim = get_char_vis(ch, arg2)))
	send_to_char("No such mobile around.\r\n", ch);
      else if (!*arg3)
	send_to_char("You must specify a trigger to remove.\r\n", ch);
      else
	trigger = arg3;
    } else if (is_abbrev(arg1, "object")) {
      if (!(object = get_obj_vis(ch, arg2)))
	send_to_char("No such object around.\r\n", ch);
      else if (!*arg3)
	send_to_char("You must specify a trigger to remove.\r\n", ch);
      else
	trigger = arg3;
    } else {
      if ((object = get_object_in_equip_vis(ch, arg1, ch->equipment, &tmp)));
      else if ((object = get_obj_in_list_vis(ch, arg1, ch->carrying)));
      else if ((victim = get_char_room_vis(ch, arg1)));
      else if ((object = get_obj_in_list_vis(ch, arg1, world[IN_ROOM(ch)]->contents)));
      else if ((victim = get_char_vis(ch, arg1)));
      else if ((object = get_obj_vis(ch, arg1)));
      else
	send_to_char("Nothing around by that name.\r\n", ch);

      trigger = arg2;
    }
    
    if (victim) {
      if (!IS_NPC(victim))
	send_to_char("Players don't have triggers.\r\n", ch);
      
      else if (!SCRIPT(victim))
	send_to_char("That mob doesn't have any triggers.\r\n", ch);	
      else if (!str_cmp(arg2, "all")) {
	extract_script(SCRIPT(victim));
	SCRIPT(victim) = NULL;
	sprintf(buf, "All triggers removed from %s.\r\n", GET_SHORT(victim));
	send_to_char(buf, ch);
      } else if (remove_trigger(SCRIPT(victim), trigger)) {
	send_to_char("Trigger removed.\r\n", ch);
	if (!TRIGGERS(SCRIPT(victim))) {
	  extract_script(SCRIPT(victim));
	  SCRIPT(victim) = NULL;
	}
      } else
	send_to_char("That trigger was not found.\r\n", ch);
    } else if (object) {
      if (!SCRIPT(object))
	send_to_char("That object doesn't have any triggers.\r\n", ch);
      
      else if (!str_cmp(arg2, "all")) {
	extract_script(SCRIPT(object));
	SCRIPT(object) = NULL;
	sprintf(buf, "All triggers removed from %s.\r\n",
		ss_data(object->short_description) ? ss_data(object->short_description) : 
		ss_data(object->name));
	send_to_char(buf, ch);
      } else if (remove_trigger(SCRIPT(object), trigger)) {
	send_to_char("Trigger removed.\r\n", ch);
	if (!TRIGGERS(SCRIPT(object))) {
	  extract_script(SCRIPT(object));
	  SCRIPT(object) = NULL;
	}
      } else
	send_to_char("That trigger was not found.\r\n", ch);
    }
  }
}


/* adds a variable with given name and value to trigger */
void add_var(struct trig_var_data **var_list, char *name, char *value, long context)
{
  struct trig_var_data *vd;

  for (vd = *var_list; vd && str_cmp(vd->name, name); vd = vd->next);

  if (vd) {
    FREE(vd->value);
    CREATE(vd->value, char, strlen(value) + 1);

    strcpy(vd->value, value);
  } else {
    CREATE(vd, struct trig_var_data, 1);
    
    CREATE(vd->name, char, strlen(name) + 1);

    strcpy(vd->name, name);
    
    CREATE(vd->value, char, strlen(value) + 1);

    strcpy(vd->value, value);
  
    vd->next = *var_list;
    *var_list = vd;
  }
  vd->context = context;
}


/* frees memory associated with var */
void free_var_el(struct trig_var_data *var)
{
  FREE(var->name);
  FREE(var->value);
  FREE(var);
}


/*
 * remove var name from var_list
 * returns 1 if found, else 0
 */
int remove_var(struct trig_var_data **var_list, char *name)
{
  struct trig_var_data *i, *j;

  for (j = NULL, i = *var_list; i && str_cmp(name, i->name);
       j = i, i = i->next);

  if (i) {
    if (j) {
      j->next = i->next;
      free_var_el(i);
    } else {
      *var_list = i->next;
      free_var_el(i);
    }

    return 1;      
  }
  return 0;
}


/*  
 *  Logs any errors caused by scripts to the system log.
 *  Will eventually allow on-line view of script errors.
 */
void script_log(char *msg)
{
  strcpy(buf, "SCRIPT ERR: ");
  strcat(buf, msg);
  mudlog(buf, NRM, TRUST_CREATOR, TRUE);
  // log("SCRIPT ERR: %s", msg);
}


#ifdef GOT_RID_OF_IT
// Support routine for scripts checking skills.
sh_int script_skill(trig_data * trig, char *argument)
{
  char *s, *t;
  
  /* get: blank, spell name, target name */
  s = strtok(argument, "'");

  if (s == NULL) {
    sprintf(buf2, "Trigger: %s, VNum %d. bad skillcheck: '%s'",
    	    GET_TRIG_NAME(trig), GET_TRIG_VNUM(trig), argument);
    script_log(buf2);
    return (-1);
  }
  s = strtok(NULL, "'");
  if (s == NULL) {
    sprintf(buf2, "Trigger: %s, VNum %d. bad skillcheck: '%s'",
    	    GET_TRIG_NAME(trig), GET_TRIG_VNUM(trig), argument);
    script_log(buf2);
    return (-1);
  }
  t = strtok(NULL, "\0");

  /* spellnum = search_block(s, spells, 0); */
  return (find_skill_num(s));

}
#endif


int text_processed(char *field, struct trig_var_data *vd, char *str, char *subfield)
{
  char *p, *p2;

  if (!str_cmp(field, "strlen")) {                     /* strlen    */
    sprintf(str, "%d", strlen(vd->value));
    return TRUE;
  } else if (!str_cmp(field, "trim")) {                /* trim      */
    /* trim whitespace from ends */
    p = vd->value;
    p2 = vd->value + strlen(vd->value) - 1;
    while (*p && isspace(*p))
      p++;
    while ((p >= p2) && isspace(*p2))
      p2--;
    if (p > p2) {		/* nothing left */
      *str = '\0';
      return TRUE;
    }
    while (p <= p2)
      *str++ = *p++;
    *str = '\0';
    return TRUE;
  } else if (!str_cmp(field, "head")) {                 /* car       */
    char *car = vd->value;
    while (*car && !isspace(*car))
      *str++ = *car++;
    *str = '\0';
    return TRUE;
  } else if (!str_cmp(field, "tail")) {                 /* cdr       */
    char *cdr = vd->value;

    while (*cdr && !isspace(*cdr))
      cdr++;			/* skip 1st field */
    while (*cdr && isspace(*cdr))
      cdr++;			/* skip to next */
    while (*cdr)
      *str++ = *cdr++;
    *str = '\0';
    return TRUE;
  } else if (!str_cmp(field, "word")) {
    int numit = atoi(subfield);

      if (numit > 0) {
        char *cdx = vd->value;

      for (int i = 0; i < numit; i++) {
	while (*cdx && !isspace(*cdx))
	  cdx++;		/* skip 1st field */
	while (*cdx && isspace(*cdx))
	  cdx++;		/* skips to xth field */
        }
        while (*cdx && !isspace(*cdx))
          *str++ = *cdx++;
        *str = '\0';
      }
        return TRUE;
  }

  return FALSE;
}


// Daniel's new routine with shared variable support.
struct trig_var_data * search_for_variable(struct script_data *sc, trig_data * trig, char *var, 
                       int varspace = 7)
{
  trig_var_data *vd = NULL;
  trig_data *tmp_trig = NULL;

  if (IS_SET(varspace, (1 << SPACE_LOCAL))) {
    for (vd = GET_TRIG_VARS(trig); vd; vd = vd->next)
      if ((!str_cmp(vd->name, var)) && (vd->context == 0 || vd->context == sc->context))
        break;
  }
  
  if ((IS_SET(varspace, (1 << SPACE_GLOBAL))) && (!vd)) {
    for (vd = sc->global_vars; vd; vd = vd->next)
      if ((!str_cmp(vd->name, var)) && (vd->context == 0 || vd->context == sc->context))
        break;
  }

  if ((IS_SET(varspace, (1 << SPACE_SHARED))) && (!vd)) {
    // Daniel's new shared variable support.  Shared variables are checked by
    // all mobiles with this trigger.
    for (tmp_trig = TRIGGERS(sc); (!vd) && tmp_trig; tmp_trig = tmp_trig->next) {
      for (vd = PROTO_TRIG(GET_TRIG_VNUM(tmp_trig))->var_list; vd; vd = vd->next)
        if ((!str_cmp(vd->name, var)) && (vd->context == 0 || vd->context == sc->context))
          break;
    }
  }
  
  return (vd);
}


// Daniel's new routine with shared variable support.
struct trig_var_data * extract_variable(struct script_data *sc, trig_data * trig, char *var,
                       int varspace = 7)
{
  trig_var_data *vd = NULL, *prevvd = NULL;
  trig_data *tmp_trig = NULL;

  if (IS_SET(varspace, (1 << SPACE_LOCAL))) {
    for (vd = GET_TRIG_VARS(trig); vd; prevvd = vd, vd = vd->next)
      if ((!str_cmp(vd->name, var)) && (vd->context == 0 || vd->context == sc->context))
  	break;
 
    if (vd) {
      if (prevvd) {
        prevvd->next = vd->next;
      } else {
        GET_TRIG_VARS(trig) = vd->next;
      }
      vd->next = NULL;
      return vd;
    }
  }

  if (IS_SET(varspace, (1 << SPACE_GLOBAL))) {
    for (vd = sc->global_vars, prevvd = NULL; vd; prevvd = vd, vd = vd->next)
      if ((vd->context == 0 || vd->context == sc->context) && (!str_cmp(vd->name, var)))
  	break;

    if (vd) {
      if (prevvd) {
        prevvd->next = vd->next;
      } else {
        sc->global_vars = vd->next;
      }
      vd->next = NULL;
      return vd;
    }
  }

  if (IS_SET(varspace, (1 << SPACE_SHARED))) {
    for (tmp_trig = TRIGGERS(sc); tmp_trig; tmp_trig = tmp_trig->next) {
      for (vd = PROTO_TRIG(GET_TRIG_VNUM(tmp_trig))->var_list,
           prevvd = NULL; vd; prevvd = vd, vd = vd->next) {
        if ((vd->context == 0 || vd->context == sc->context) && (!str_cmp(vd->name, var)))
  	  break;
      }
      if (vd)
        break;
    }

    if (vd) {
      if (prevvd) {
        prevvd->next = vd->next;
      } else {
        PROTO_TRIG(GET_TRIG_VNUM(tmp_trig))->var_list = vd->next;
      }
      vd->next = NULL;
      return vd;
    }
  }

  return NULL;  
}


/* sets str to be the value of var.field */
void find_replacement(void *go, struct script_data *sc, trig_data * trig,
                      int type, char *var, char *field, char *subfield, char *str)
{
  struct trig_var_data *vd;
  char_data *ch = NULL, *c = NULL, *rndm;
  obj_data *obj = NULL, *o = NULL;
  struct room_data *room, *r = NULL;
  char *name;
  int num, count;
  sh_int skillnum;
  bool unknown = FALSE;		// When true, this signifies an unknown field

  int find_eq_pos(struct char_data *ch, struct obj_data *obj, char *arg,
                byte bottomslot = 0, byte topslot = NUM_WEARS);

  vd = search_for_variable(sc, trig, var);

  if (!*field) {
    if (vd)
      strcpy(str, vd->value);
    else {
      if (!str_cmp(var, "self"))
	sprintf(str, "%c%d", UID_CHAR, ((game_data *) go)->Id());
      else
	*str = '\0';
    }

    return;
  } else {
    if (vd) {
      name = vd->value;

      switch (type) {
      case MOB_TRIGGER:
	ch = (char_data *) go;

	if ((o = get_object_in_equip(ch, name)));
	else if ((o = get_obj_in_list(name, ch->carrying)));
	else if ((c = get_char_room(name, IN_ROOM(ch))));
	else if ((o = get_obj_in_list(name, world[IN_ROOM(ch)]->contents)));
	else if ((c = get_char(name)));
	else if ((o = get_obj(name)));
	else if ((r = get_room(name))) {
	}
	break;
      case OBJ_TRIGGER:
	obj = (obj_data *) go;

	if ((c = get_char_by_obj(obj, name)));
	else if ((o = get_obj_by_obj(obj, name)));
	else if ((r = get_room(name))) {
	}
	break;
      case WLD_TRIGGER:
	room = (struct room_data *) go;

	if ((c = get_char_by_room(room, name)));
	else if ((o = get_obj_by_room(room, name)));
	else if ((r = get_room(name))) {
        }
	break;
      }
    } else {
      if (!str_cmp(var, "self")) {
	switch (type) {
	case MOB_TRIGGER:
	  c = (char_data *) go;
	  break;
	case OBJ_TRIGGER:
	  o = (obj_data *) go;
	  break;
	case WLD_TRIGGER:
	  r = (struct room_data *) go;
	  break;
	}
      } else if (!str_cmp(var, "people")) {
	sprintf(str, "%d", ((num = atoi(field)) > 0) ? trgvar_in_room(num) : 0);
	return;
      } else if (!str_cmp(var, "random")) {
	if (!str_cmp(field, "char")) {
	  rndm = NULL;
	  count = 0;

	  if (type == MOB_TRIGGER) {
	    ch = (char_data *) go;
	    for (c = world[IN_ROOM(ch)]->people; c; c = c->next_in_room)
	      if (!PRF_FLAGGED(c, PRF_NOHASSLE) && (c != ch) &&
		  CAN_SEE(ch, c)) {
		if (!number(0, count))
		  rndm = c;
		count++;
	      }
	  } else if (type == OBJ_TRIGGER) {
	    for (c = world[((obj_data *) go)->InRoom()]->people; c;
		 c = c->next_in_room)
	      if (!PRF_FLAGGED(c, PRF_NOHASSLE) && !GET_INVIS_LEV(c)) {
		if (!number(0, count))
		  rndm = c;
		count++;
	      }
	  } else if (type == WLD_TRIGGER) {
	    for (c = ((struct room_data *) go)->people; c;
		 c = c->next_in_room)
	      if (!PRF_FLAGGED(c, PRF_NOHASSLE) && !GET_INVIS_LEV(c)) {
		if (!number(0, count))
		  rndm = c;
		count++;
	      }
	  }
	  if (rndm)
	    sprintf(str, "%c%ld", UID_CHAR, GET_ID(rndm));
	  else
	    *str = '\0';
	} else
	  sprintf(str, "%d", ((num = atoi(field)) > 0) ? number(1, num) : 0);
	
	return;
      }
    }
    
    if (c) {
      if (text_processed(field, vd, str, subfield))
	return;

      switch (*field) {
      case 'a':
	if (!str_cmp(field, "alias"))
	  strcpy(str, GET_NAME(c));

	else if (!str_cmp(field, "align"))
	  sprintf(str, "%d", GET_ALIGNMENT(c));

	else if (!str_cmp(field, "aiming")) {
	  if (AIMING(c) == NOBODY)
	    strcpy(str, "0");
        else
          sprintf(str, "%c%ld", UID_CHAR, AIMING(c));
	} 

	else
	  unknown = TRUE;
	break;

      case 'c':
	if (!str_cmp(field, "cha"))
	  sprintf(str, "%d", GET_CHA(c));

	else if (!str_cmp(field, "con"))
	  sprintf(str, "%d", GET_CON(c));

	else if (!str_cmp(field, "class"))
	  sprinttype(GET_CLASS(c), class_types, str);

	else if (!str_cmp(field, "canbeseen")) {
	  if ((type == MOB_TRIGGER) && !CAN_SEE((char_data *) go, c))
	    strcpy(str, "0");
	  else
	    strcpy(str, "1");
	} 
	
	else if (!str_cmp(field, "carrying")) {
	  if (c->carrying == NULL)
            strcpy(str, "0");
	  else
            sprintf(str, "%c%ld", UID_CHAR, GET_ID(c->carrying));
	}

	else
	  unknown = TRUE;
	break;

      case 'd':
	if (!str_cmp(field, "dex"))
	  sprintf(str, "%d", GET_DEX(c));

	else
	  unknown = TRUE;
	break;

      case 'e':
	if (!str_cmp(field, "encumbrance"))
	  sprintf(str, "%d", GET_ENCUMBRANCE(c, IS_CARRYING_W(c)));

#ifdef GOT_RID_OF_IT
	else if (!str_cmp(field, "eq")) {
	  int pos = find_eq_pos(c, NULL, subfield);

	  if (pos == -1 && isdigit(*subfield))
	    pos = atoi(subfield);
	  if (!subfield || !*subfield || pos < 0 || pos > NUM_WEARS)
	    strcpy(str, "");
	  else
	    sprintf(str, "%c%ld", UID_CHAR, GET_ID(GET_EQ(c, pos)));
	}
#endif

	else
	  unknown = TRUE;
	break;

      case 'f':
	if (!str_cmp(field, "fighting")) {
	  if (FIGHTING(c) == NULL)
	    strcpy(str, "0");
	  else
	    sprintf(str, "%c%ld", UID_CHAR, GET_ID(FIGHTING(c)));
	} 
	
	else if (!str_cmp(field, "following")) {
	  if (c->master == NULL)
	    strcpy(str, "0");
	  else
	    sprintf(str, "%c%ld", UID_CHAR, GET_ID(c->master));
	} 
	
	else
	  unknown = TRUE;
	break;

      case 'g':
	if (!str_cmp(field, "gold"))
	  sprintf(str, "%d", GET_GOLD(c));

	else
	  unknown = TRUE;
	break;

      case 'h':
	if (!str_cmp(field, "hit"))
	  sprintf(str, "%d", GET_HIT(c));

	else if (!str_cmp(field, "height"))
	  sprintf(str, "%d", GET_HEIGHT(c));

      else if (!str_cmp(field, "heshe"))
	strcpy(str, HSSH(c));

	else if (!str_cmp(field, "hisher"))
	  strcpy(str, HSHR(c));

      else if (!str_cmp(field, "himher"))
	strcpy(str, HMHR(c));

	else if (!str_cmp(field, "hishers"))
	  strcpy(str, HSHRS(c));

	else if (!str_cmp(field, "hunting")) {
	  if (HUNTING(c) == NOBODY)
	    strcpy(str, "0");
	  else
	    sprintf(str, "%c%ld", UID_CHAR, HUNTING(c));
	} 

	else if (!str_cmp(field, "hitpercent"))
	  sprintf(str, "%d", (int) (100 * GET_HIT(c)) / MAX(GET_MAX_HIT(c), 1));

	else
	  unknown = TRUE;
	break;

      case 'i':
	if (!str_cmp(field, "id"))
	  sprintf(str, "%ld", GET_ID(c));

	else if (!str_cmp(field, "int"))
          sprintf(str, "%d", GET_INT(c));

	else if (!str_cmp(field, "isnpc"))
          sprintf(str, "%d", IS_NPC(c) ? 1 : 0);

        else if (!str_cmp(field, "isgood"))
          sprintf(str, "%d", IS_GOOD(c) ? 1 : 0);

        else if (!str_cmp(field, "isevil"))
          sprintf(str, "%d", IS_EVIL(c) ? 1 : 0);

        else if (!str_cmp(field, "isneutral"))
          sprintf(str, "%d", IS_NEUTRAL(c) ? 1 : 0);

	else
	  unknown = TRUE;
	break;

      case 'k':
	if (!str_cmp(field, "keywords"))
          strcpy(str, GET_NAME(c));

	else
	  unknown = TRUE;
	break;

      case 'l':
	if (!str_cmp(field, "lawful"))
	  sprintf(str, "%d", GET_LAWFULNESS(c));

	else if (!str_cmp(field, "level"))
          sprintf(str, "%ld", GET_LEVEL(c));

	else if (!str_cmp(field, "leader")) {
	  if ((c->master == NULL) || (!AFF_FLAGGED(c, AFF_GROUP)))
	    sprintf(str, "%c%ld", UID_CHAR, GET_ID(c));
	  else
	    sprintf(str, "%c%ld", UID_CHAR, GET_ID(c->master));
	} 

	else
	  unknown = TRUE;
	break;

      case 'm':
	if (!str_cmp(field, "mana"))
	  sprintf(str, "%d", GET_MANA(c));

	else if (!str_cmp(field, "move"))
	  sprintf(str, "%d", GET_MOVE(c));

	else if (!str_cmp(field, "master")) {
	  if ((c->master == NULL) || (!AFF_FLAGGED(ch, AFF_CHARM)))
	    strcpy(str, "0");
	  else
	    sprintf(str, "%c%ld", UID_CHAR, GET_ID(c->master));
	} 

        else if (!str_cmp(field, "material"))
          sprintf(str, "%s", material_types[GET_CHAR_MATERIAL(c)]);

        else if (!str_cmp(field, "maxhit"))
          sprintf(str, "%d", GET_MAX_HIT(c));

        else if (!str_cmp(field, "maxmana"))
          sprintf(str, "%d", GET_MAX_MANA(c));

        else if (!str_cmp(field, "maxmove"))
          sprintf(str, "%d", GET_MAX_MOVE(c));

	else
	  unknown = TRUE;
	break;

      case 'n':
	if (!str_cmp(field, "name"))
	  strcpy(str, GET_SHORT(c));

	else if (!str_cmp(field, "nextinroom")) {
	  if (c->next_in_room == NULL)
	    strcpy(str, "0");
	  else
	    sprintf(str, "%c%ld", UID_CHAR, GET_ID(c->next_in_room));
	} 

	else
	  unknown = TRUE;
	break;

      case 'p':
	if (!str_cmp(field, "position"))
	strcpy(str, position_types[(int) GET_POS(c)]);

	else
	  unknown = TRUE;
	break;

      case 'r':
	if (!str_cmp(field, "race"))
	  sprinttype(GET_RACE(c), race_types, str);

	else if (!str_cmp(field, "room"))
	  sprintf(str, "%d", IN_ROOM(c));

	else if (!str_cmp(field, "ridden_by")) {
	if (RIDER(c) == NULL)
	  strcpy(str, "0");
	else
          sprintf(str, "%c%ld", UID_CHAR, GET_ID(RIDER(c)));
	} 
	
	else if (!str_cmp(field, "riding")) {
	if (RIDING(c) == NULL)
	  strcpy(str, "0");
	else
          sprintf(str, "%c%ld", UID_CHAR, GET_ID(RIDING(c)));
	} 
	
	else if (!str_cmp(field, "roomseeking")) {
          if (ROOMSEEKING(c) == NOWHERE)
	  strcpy(str, "");
	else
            sprintf(str, "%d", ROOMSEEKING(c));
	} 
	
	else
	  unknown = TRUE;
	break;

      case 's':
	if (!str_cmp(field, "sex"))
	  strcpy(str, genders[(int) GET_SEX(c)]);

	else if (!str_cmp(field, "str"))
	  sprintf(str, "%d", GET_STR(c));

	else if (!str_cmp(field, "seeking")) {

	if (SEEKING(c) == NOBODY)
	  strcpy(str, "0");
	else
          sprintf(str, "%c%ld", UID_CHAR, SEEKING(c));
	} 
	
	else if (!str_cmp(field, "stalking")) {

	if (STALKING(c) == NOBODY)
	  strcpy(str, "0");
	else
          sprintf(str, "%c%ld", UID_CHAR, STALKING(c));
	} 

	else if (!str_cmp(field, "skillpts")) {
          skillnum = find_skill_num(subfield, FIRST_STAT, LAST_STAT);
          if (skillnum < 0)
            skillnum = find_skill_num(subfield);
	  sprintf(str, "%d", CHAR_SKILL(c).GetSkillPts(skillnum, NULL));
        }

	else if (!str_cmp(field, "skillroll")) {
          skillnum = find_skill_num(subfield, FIRST_STAT, LAST_STAT);
          if (skillnum < 0)
            skillnum = find_skill_num(subfield);
	  sprintf(str, "%d", SKILLROLL(c, skillnum));
        }

	else if (!str_cmp(field, "skillchance")) {
          skillnum = find_skill_num(subfield, FIRST_STAT, LAST_STAT);
          if (skillnum < 0)
            skillnum = find_skill_num(subfield);
	  sprintf(str, "%d", SKILLCHANCE(c, skillnum, NULL));
        }

	else if (!str_cmp(field, "skilltheory")) {
          skillnum = find_skill_num(subfield, FIRST_STAT, LAST_STAT);
          if (skillnum < 0)
            skillnum = find_skill_num(subfield);
	  sprintf(str, "%d", CHAR_SKILL(c).GetSkillTheory(skillnum, NULL));
        }

	else
	  unknown = TRUE;
	break;

      case 't':
	if (!str_cmp(field, "trust"))
	  sprintf(str, "%d", GET_TRUST(c));

	else
	  unknown = TRUE;
	break;

      case 'u':
	if (!str_cmp(field, "uid"))
          sprintf(str, "%c%ld", UID_CHAR, GET_ID(c));
	else
	  unknown = TRUE;
	break;

      case 'v':
	if (!str_cmp(field, "vnum"))
	  sprintf(str, "%d", GET_MOB_NUM(c));

	else
	  unknown = TRUE;
	break;

      case 'w':
	if (!str_cmp(field, "wis"))
	  sprintf(str, "%d", GET_WIS(c));

	else if (!str_cmp(field, "weight"))
	  sprintf(str, "%d", GET_WEIGHT(c));

	else if (!str_cmp(field, "weightcarried"))
	  sprintf(str, "%d", IS_CARRYING_W(c));

	else
	  unknown = TRUE;
	break;

      case 'z':
	if (!str_cmp(field, "zone")) {
	  if (!PURGED(c) && (IN_ROOM(c) != NOWHERE))
            sprintf(str, "%d", world[IN_ROOM(ch)]->zone);
	  else
            *str = '\0';
	} 
	
	else
	  unknown = TRUE;
	break;

      default:
	unknown = TRUE;
      }

      if (unknown) {
	*str = '\0';
	sprintf(buf2, 
		"Trigger: %s, VNum %d. unknown char field: '%s'",
		GET_TRIG_NAME(trig), GET_TRIG_VNUM(trig), field);
	script_log(buf2);
      }

    } else if (o) {

      if (text_processed(field, vd, str, subfield))
	return;

      switch (*field) {
      case 'a':
	if (!str_cmp(field, "ana"))
	  sprintf(str, "%s", ANA(o));

	else
	  unknown = TRUE;
	break;
      case 'c':
	if (!str_cmp(field, "contains")) {
	  if (o->contains == NULL)
            strcpy(str, "0");
	  else
            sprintf(str, "%c%ld", UID_CHAR, GET_ID(o->contains));
	}
      case 'i':
	if (!str_cmp(field, "id"))
          sprintf(str, "%ld", GET_ID(o));
	else
	  unknown = TRUE;
	break;

      case 'k':
	if (!str_cmp(field, "keywords"))
          strcpy(str, OBJ_NAME(o));

	else
	  unknown = TRUE;
	break;

      case 'm':
	if (!str_cmp(field, "material"))
          sprintf(str, "%s", material_types[GET_OBJ_MATERIAL(o)]);
  
	else
	  unknown = TRUE;
	break;

      case 'n':
	if (!str_cmp(field, "name"))
	  strcpy(str, ss_data(o->name));

	else if (!str_cmp(field, "nextinroom")) {
	  if (o->next_content == NULL)
	  strcpy(str, "0");
	  else
            sprintf(str, "%c%ld", UID_CHAR, GET_ID(o->next_content));
	}
	
	else
	  unknown = TRUE;
	break;

      case 'o':
	if (!str_cmp(field, "owner")) {
          if (OBJ_CARRIED_BY(o)) {
            sprintf(str, "%c%ld", UID_CHAR, GET_ID(OBJ_CARRIED_BY(o)));
          } else {
            strcpy(str, "0");
          }
	} 

	else
	  unknown = TRUE;
	break;

      case 'r':
	if (!str_cmp(field, "room"))
	  sprintf(str, "%d", IN_ROOM(o));

      case 's':
	if (!str_cmp(field, "sana"))
	  sprintf(str, "%s", SANA(o));
	else if (!str_cmp(field, "shortdesc"))
	  strcpy(str, ss_data(o->short_description));
      
	else
	  unknown = TRUE;
	break;

      case 't':
	if (!str_cmp(field, "type"))
	  sprinttype(GET_OBJ_TYPE(o), item_types, str);
      else if (!str_cmp(field, "timer"))
        sprintf(str, "%f", (GET_OBJ_TIMER(o) ? 
	(float) time_to_event(GET_OBJ_TIMER(o)) / (60 RL_SEC) : 0.0));

              
	else
	  unknown = TRUE;
	break;

      case 'u':
	if (!str_cmp(field, "uid"))
          sprintf(str, "%c%ld", UID_CHAR, GET_ID(o));
	else
	  unknown = TRUE;
	break;

      case 'v':
	if (!str_cmp(field, "vnum"))
	  sprintf(str, "%d", GET_OBJ_NUM(o));

      else if (!str_cmp(field, "val0"))
	sprintf(str, "%d", GET_OBJ_VAL(o, 0));
      
      else if (!str_cmp(field, "val1"))
	sprintf(str, "%d", GET_OBJ_VAL(o, 1));
      
      else if (!str_cmp(field, "val2"))
	sprintf(str, "%d", GET_OBJ_VAL(o, 2));
      
      else if (!str_cmp(field, "val3"))
	sprintf(str, "%d", GET_OBJ_VAL(o, 3));
      
      else if (!str_cmp(field, "val4"))
	sprintf(str, "%d", GET_OBJ_VAL(o, 4));
      
      else if (!str_cmp(field, "val5"))
	sprintf(str, "%d", GET_OBJ_VAL(o, 5));
      
      else if (!str_cmp(field, "val6"))
	sprintf(str, "%d", GET_OBJ_VAL(o, 6));
      
      else if (!str_cmp(field, "val7"))
	sprintf(str, "%d", GET_OBJ_VAL(o, 7));
      
	else
	  unknown = TRUE;
	break;
      case 'w':
	if (!str_cmp(field, "wearer")) {
	  if (OBJ_WORN_BY(o)) {
	    sprintf(str, "%c%ld", UID_CHAR, GET_ID(OBJ_WORN_BY(o)));
	  } else {
	    strcpy(str, "0");
	  }
	} 
	
	else if (!str_cmp(field, "weight"))
	  sprintf(str, "%d", GET_OBJ_WEIGHT(o));

	else
	  unknown = TRUE;
	break;

      default:
	unknown = TRUE;
      }

      if (unknown) {
	*str = '\0';
	sprintf(buf2,
		"Trigger: %s, VNum %d, type: %d. unknown object field: '%s'",
		GET_TRIG_NAME(trig), GET_TRIG_VNUM(trig), type, field);
	script_log(buf2);
      }
    } else if (r) {
      if (text_processed(field, vd, str, subfield))
	return;

      switch (*field) {
      case 'd':
	if (!str_cmp(field, "down")) {
	  if (r->dir_option[DOWN])
	    sprintbit(r->dir_option[DOWN]->exit_info, exit_bits, str);
	  else
	    *str = '\0';
	} 
	
	else
	  unknown = TRUE;
	break;
      case 'e':
	if (!str_cmp(field, "east")) {
	  if (r->dir_option[EAST])
	    sprintbit(r->dir_option[EAST]->exit_info, exit_bits, str);
	  else
	    *str = '\0';
	} 
	
	else
	  unknown = TRUE;
	break;
      case 'n':
      if (!str_cmp(field, "name"))
	strcpy(str, r->name);

      else if (!str_cmp(field, "north")) {
	if (r->dir_option[NORTH])
	  sprintbit(r->dir_option[NORTH]->exit_info, exit_bits, str);
	else
	  *str = '\0';
      }
	
	else
	  unknown = TRUE;
	break;
      case 'o':
	if (!str_cmp(field, "objects")) {
	  if (r->contents == NULL)
	    strcpy(str, "0");
	  else
	    sprintf(str, "%c%ld", UID_CHAR, GET_ID(r->contents));
      }
	
	else
	  unknown = TRUE;
	break;
      case 'p':
	if (!str_cmp(field, "people")) {
	  if (r->people == NULL)
	    strcpy(str, "0");
	  else
	    sprintf(str, "%c%ld", UID_CHAR, GET_ID(r->people));
	} 
	
	else
	  unknown = TRUE;
	break;
      case 's':
	if (!str_cmp(field, "south")) {
	if (r->dir_option[SOUTH])
	  sprintbit(r->dir_option[SOUTH]->exit_info, exit_bits, str);
	else
	  *str = '\0';
      }
	
	else
	  unknown = TRUE;
	break;
      case 'u':
	if (!str_cmp(field, "up")) {
	if (r->dir_option[UP])
	  sprintbit(r->dir_option[UP]->exit_info, exit_bits, str);
	else
	  *str = '\0';
      }
	
	else
	  unknown = TRUE;
	break;
      case 'v':
	if (!str_cmp(field, "vnum"))
	  sprintf(str, "%d", r->number);

	else
	  unknown = TRUE;
	break;

      case 'w':
	if (!str_cmp(field, "west")) {
	  if (r->dir_option[WEST])
	    sprintbit(r->dir_option[WEST]->exit_info, exit_bits, str);
          else
	    *str = '\0';
	} 
	
	else
	  unknown = TRUE;
	break;

      case 'z':
	if (!str_cmp(field, "zone")) {
	  sprintf(str, "%d", r->zone);
	} 
	
	else
	  unknown = TRUE;
	break;

      default:
	unknown = TRUE;
      }

      if (unknown) {
	*str = '\0';
	sprintf(buf2,
		"Trigger: %s, VNum %d, type: %d. unknown object field: '%s'",
		GET_TRIG_NAME(trig), GET_TRIG_VNUM(trig), type, field);
	script_log(buf2);
      }
    } 
    
    else
      *str = '\0';
  }
}


/* substitutes any variables into line and returns it as buf */
void var_subst(void *go, struct script_data *sc, trig_data * trig,
	       int type, char *line, char *buf)
{
  char tmp[MAX_INPUT_LENGTH], repl_str[MAX_INPUT_LENGTH], *var, *field, *p;
  char *subfield_p, subfield[MAX_INPUT_LENGTH];
  int left, len;
  int paren_count = 0;

  if (!strchr(line, '%')) {
    strcpy(buf, line);
    return;
  }
  p = strcpy(tmp, line);
  subfield_p = subfield;
  
  left = MAX_INPUT_LENGTH - 1;
  
  while (*p && (left > 0)) {

    while (*p && (*p != '%') && (left > 0)) {
      *(buf++) = *(p++);
      left--;
    }
    
    *buf = '\0';
    
    /* double % */
    if (*p && (*(++p) == '%') && (left > 0)) {
      *(buf++) = *(p++);
      *buf = '\0';
      left--;
      continue;
    } else if (*p && (left > 0)) {
      
      for (var = p; *p && (*p != '%') && (*p != '.'); p++);

      field = p;
      if (*p == '.') {
	*(p++) = '\0';
	for (field = p; *p && ((*p != '%') || (paren_count)); p++) {
	  if (*p == '(') {
            *p = '\0';
            paren_count++;
	  } else if (*p == ')') {
            *p = '\0';
            paren_count--;
	  } else if (paren_count)
	    *subfield_p++ = *p;
        }
      }
      *(p++) = '\0';
      *subfield_p = '\0';

      find_replacement(go, sc, trig, type, var, field, subfield, repl_str);
      
      strncat(buf, repl_str, left);
      len = strlen(repl_str);
      buf += len;
      left -= len;
    }
  }  
}


/* returns 1 if string is all digits, else 0 */
int is_num(char *num)
{
  while (*num && (isdigit(*num) || *num == '-'))
    num++;

  if (!*num || isspace(*num))
    return 1;
  else
    return 0;
}


/* evaluates 'lhs op rhs', and copies to result */
void eval_op(char *op, char *lhs, char *rhs, char *result, void *go,
	     struct script_data *sc, trig_data * trig)
{
  char *p;
  int n;

  /* strip off extra spaces at begin and end */
  while (*lhs && isspace(*lhs)) 
    lhs++;
  while (*rhs && isspace(*rhs))
    rhs++;
  
  for (p = lhs; *p; p++);
  for (--p; isspace(*p) && (p > lhs); *p-- = '\0');
  for (p = rhs; *p; p++);
  for (--p; isspace(*p) && (p > rhs); *p-- = '\0');  


  /* find the op, and figure out the value */
  if (!strcmp("||", op)) {
    if ((!*lhs || (*lhs == '0')) && (!*rhs || (*rhs == '0')))
      strcpy(result, "0");
    else
      strcpy(result, "1");
  } else if (!strcmp("&&", op)) {
    if (!*lhs || (*lhs == '0') || !*rhs || (*rhs == '0'))
      strcpy(result, "0");
    else
      strcpy(result, "1");
  } else if (!strcmp("==", op)) {
    if (is_num(lhs) && is_num(rhs))
      sprintf(result, "%d", atoi(lhs) == atoi(rhs));
    else
      sprintf(result, "%d", !str_cmp(lhs, rhs));
  } else if (!strcmp("!=", op)) {
    if (is_num(lhs) && is_num(rhs))
      sprintf(result, "%d", atoi(lhs) != atoi(rhs));
    else
      sprintf(result, "%d", str_cmp(lhs, rhs));
  } else if (!strcmp("<=", op)) {
    if (is_num(lhs) && is_num(rhs))
      sprintf(result, "%d", atoi(lhs) <= atoi(rhs));
    else
      sprintf(result, "%d", str_cmp(lhs, rhs) <= 0);
  } else if (!strcmp(">=", op)) {
    if (is_num(lhs) && is_num(rhs))
      sprintf(result, "%d", atoi(lhs) >= atoi(rhs));
    else
      sprintf(result, "%d", str_cmp(lhs, rhs) <= 0);
  } else if (!strcmp("<", op)) {
    if (is_num(lhs) && is_num(rhs))
      sprintf(result, "%d", atoi(lhs) < atoi(rhs));
    else
      sprintf(result, "%d", str_cmp(lhs, rhs) < 0);
  } else if (!strcmp(">", op)) {
    if (is_num(lhs) && is_num(rhs))
      sprintf(result, "%d", atoi(lhs) > atoi(rhs));
    else
      sprintf(result, "%d", str_cmp(lhs, rhs) > 0);
  } else if (!strcmp("/=", op))
    sprintf(result, "%c", str_str(lhs, rhs) ? '1' : '0');

  else if (!strcmp("*", op))
    sprintf(result, "%d", atoi(lhs) * atoi(rhs));
  
  else if (!strcmp("/", op))
    sprintf(result, "%d", (n = atoi(rhs)) ? (atoi(lhs) / n) : 0);

  else if (!strcmp("+", op)) 
    sprintf(result, "%d", atoi(lhs) + atoi(rhs));

  else if (!strcmp("-", op))
    sprintf(result, "%d", atoi(lhs) - atoi(rhs));

  else if (!strcmp("!", op)) {
    if (is_num(rhs))
      sprintf(result, "%d", !atoi(rhs));
    else
      sprintf(result, "%d", !*rhs);
  }
}


/*
 * p points to the first quote, returns the matching
 * end quote, or the last non-null char in p.
 */
char *matching_quote(char *p)
{
  for (p++; *p && (*p != '"'); p++) {
    if (*p == '\\')
      p++;
  }

  if (!*p)
    p--;

  return p;
}

/*
 * p points to the first paren.  returns a pointer to the
 * matching closing paren, or the last non-null char in p.
 */
char *matching_paren(char *p)
{
  int i;

  for (p++, i = 1; *p && i; ++p) {
    if (*p == '(')
      i++;
    else if (*p == ')')
      i--;
    else if (*p == '"')
      p = matching_quote(p);
  }

  return --p;
}


/* evaluates line, and returns answer in result */
void eval_expr(char *line, char *result, void *go, struct script_data *sc,
	       trig_data * trig, int type)
{
  char expr[MAX_INPUT_LENGTH], *p;

  while (*line && isspace(*line))
    line++;
  
  if (eval_lhs_op_rhs(line, result, go, sc, trig, type));

  else if (*line == '(') {
    p = strcpy(expr, line);
    p = matching_paren(expr);
    *p = '\0';
    eval_expr(expr + 1, result, go, sc, trig, type);
  } else
    var_subst(go, sc, trig, type, line, result);
}


/*
 * evaluates expr if it is in the form lhs op rhs, and copies
 * answer in result.  returns 1 if expr is evaluated, else 0
 */
int eval_lhs_op_rhs(char *expr, char *result, void *go, struct script_data *sc,
		    trig_data * trig, int type)
{
  char *p, *tokens[MAX_INPUT_LENGTH];
  char line[MAX_INPUT_LENGTH], lhr[MAX_INPUT_LENGTH], rhr[MAX_INPUT_LENGTH];
  int i, j;
  
  /*
   * valid operands, in order of priority
   * each must also be defined in eval_op()
   */
  static char *ops[] =
  {
    "||",
    "&&",
    "==",
    "!=",
    "<=",
    ">=",
    "<",
    ">",
    "/=",
    "-",
    "+",
    "/",
    "*",
    "!",
    "\n"
  };

  p = strcpy(line, expr);

  /*
   * initialize tokens, an array of pointers to locations
   * in line where the ops could possibly occur.
   */
  for (j = 0; *p; j++) {
    tokens[j] = p;
    if (*p == '(')
      p = matching_paren(p) + 1;
    else if (*p == '"')
      p = matching_quote(p) + 1;
    else if (isalnum(*p))
      for (p++; *p && (isalnum(*p) || isspace(*p)); p++);
    else
      p++;
  }
  tokens[j] = NULL;

  for (i = 0; *ops[i] != '\n'; i++)
    for (j = 0; tokens[j]; j++)
      if (!strn_cmp(ops[i], tokens[j], strlen(ops[i]))) {
	*tokens[j] = '\0';
	p = tokens[j] + strlen(ops[i]);

	eval_expr(line, lhr, go, sc, trig, type);
	eval_expr(p, rhr, go, sc, trig, type);
	eval_op(ops[i], lhr, rhr, result, go, sc, trig);

	return 1;
      }

  return 0;
}



/* returns 1 if cond is true, else 0 */
int process_if(char *cond, void *go, struct script_data *sc,
	       trig_data * trig, int type)
{
  char result[MAX_INPUT_LENGTH], *p;

  eval_expr(cond, result, go, sc, trig, type);
  
  p = result;
  skip_spaces(&p);

  if (!*p || *p == '0')
    return 0;
  else
    return 1;
}


/* returns 1 if cond is true, else 0 */
int process_foreach(char *cond, void *go, struct script_data *sc,
	       trig_data * trig, int &type, unsigned long &loops)
{
  char variable[MAX_INPUT_LENGTH],
       args[MAX_INPUT_LENGTH],
       expr[MAX_INPUT_LENGTH],
       *line, *p;
  int tmp;

  struct trig_var_data *vd = NULL;
  char_data *c = NULL;
  obj_data *o = NULL;
  room_data *r = NULL;
  long flags = 0;

  *variable = '\0';
  *args = '\0';
  *expr = '\0';

  line = cond;

  if (!strn_cmp(line, "mob ", 4)) {
    SET_BIT(flags, (1 << FOREACH_MOBS));
  } else if (!strn_cmp(line, "obj ", 4)) {
    SET_BIT(flags, (1 << FOREACH_OBJS));
  } else if (!strn_cmp(line, "room ", 5)) {
    SET_BIT(flags, (1 << FOREACH_ROOMS));
  } else {
    sprintf(buf2, "Trigger: %s, VNum: %d:  foreach without type obj|mob|room",
            GET_TRIG_NAME(trig), GET_TRIG_VNUM(trig));
    script_log(buf2);
    return 0;
  }

  // Grab the name of the variable being set
  // We need the most raw, direct way to do this to keep speed up.
  while (*line && (*line != ' '))
    ++line;
  while (*line && (*line == ' '))
    ++line;
  
  p = variable;
  while (*line && (*line != ' ')) {
    *p++ = *line++;
  }
  *p = '\0';
  
  if (!*variable) {
    sprintf(buf2, "Trigger: %s, VNum: %d:  foreach without variable specified",
            GET_TRIG_NAME(trig), GET_TRIG_VNUM(trig));
    script_log(buf2);
    return 0;
  }

  while (*line && (*line == ' '))
    ++line;

  while (*line && (*line != '(') && (*line != '%')) {

    p = args;
    while (*line && (*line != ' ')) {
      *p++ = *line++;
    }
    *p = '\0';

    tmp = search_block(args, foreach_condition_strings, FALSE);
    if (tmp >= 0)
      SET_BIT(flags, (1 << tmp));
  }

  while (*line && (*line == ' '))
    ++line;

  // We only have the expression left.  If we're already in the loop, this is unnecessary;
  // otherwise, it is the head of the list to loop through.

  if (!loops) {		// If this is the first loop, get the ID from the expression
    // Evaluate the expression, grabbing the ID of the head of the list to loop through
    eval_expr(line, expr, go, sc, trig, type);
      
    if ((!*expr) || (!strcmp(expr, "0"))) {
      return 0;
    }

    if (!str_cmp(expr, "self")) {
      switch (type) {
      case MOB_TRIGGER:
        c = (char_data *) go;
        break;
      case OBJ_TRIGGER:
        o = (obj_data *) go;
        break;
      case WLD_TRIGGER:
        r = (struct room_data *) go;
        break;
      }
    } else {
      if (IS_SET((flags), (1 << FOREACH_MOBS))) {
        c = get_char(expr);
      }
      else if (IS_SET((flags), (1 << FOREACH_OBJS))) {
        o = get_obj(expr);
      }
      else if (IS_SET((flags), (1 << FOREACH_ROOMS))) {
        r = get_room(expr);
      }
    }

    if (c) {
      sprintf(expr, "%c%ld", UID_CHAR, GET_ID(c));
    } else if (o) {
      sprintf(expr, "%c%ld", UID_CHAR, GET_ID(o));
    } else if (r) {
      sprintf(expr, "%c%ld", UID_CHAR, GET_ID(r));
    } else 
      return 0;

  } else {	// This is not the first loop.  Get the variable from the iterator.
    vd = search_for_variable(sc, trig, variable, (1 << SPACE_LOCAL));
    
    if (!vd) {
      return 0;
    }
    
    if (IS_SET((flags), (1 << FOREACH_MOBS))) {
      c = get_char(vd->value);
      if (c) {
        c = c->next_in_room;
        if (c) {
          sprintf(expr, "%c%ld", UID_CHAR, GET_ID(c));
        } else
          return 0;
      }
    }
    else if (IS_SET((flags), (1 << FOREACH_OBJS))) {
      o = get_obj(vd->value);
      if (o) {
        o = o->next_content;

        if (o) {
          sprintf(expr, "%c%ld", UID_CHAR, GET_ID(o));
        } else
          return 0;
      }
    }
    else if (IS_SET((flags), (1 << FOREACH_ROOMS))) {
      r = get_room(vd->value);
      int roomnum = r->number;
      r = NULL;
      while (!r) {
        if (++roomnum >= top_of_world)
          break;
	if (!world[roomnum])
	  continue;
        else
          r = world[roomnum];
      }
      if (r) {
        sprintf(expr, "%c%ld", UID_CHAR, GET_ID(r));
      } else {
        return 0;
      }
    } else {
      sprintf(buf2, "Trigger: %s, VNum: %d:  foreach target '%s' not found",
        GET_TRIG_NAME(trig), GET_TRIG_VNUM(trig), args);
      script_log(buf2);
      return 0;
    }
  }

  if (!*expr) {
    vd = extract_variable(sc, trig, variable, (1 << SPACE_LOCAL));
    if (vd) {
      free_var_el(vd);
    }
    return 0;
  } else {
    add_var(&GET_TRIG_VARS(trig), variable, expr, sc->context);
  }

  return 1;

}


/*
 * scans for end of if-block.
 * returns the line containg 'end', or the last
 * line of the trigger if not found.
 */
struct cmdlist_element *find_end(struct cmdlist_element *cl)
{
  struct cmdlist_element *c;
  char *p;

  if (!(cl->next))
    return cl;

  for (c = cl->next; c && c->next; c = c->next) {
    for (p = c->cmd; *p && isspace(*p); p++);

    if (!strn_cmp("if ", p, 3))
      c = find_end(c);
    else if (!strn_cmp("end", p, 3))
      return c;
  }
  
  return c;
}


/*
 * searches for valid elseif, else, or end to continue execution at.
 * returns line of elseif, else, or end if found, or last line of trigger.
 */
struct cmdlist_element *find_else_end(trig_data * trig,
				      struct cmdlist_element *cl, void *go,
				      struct script_data *sc, int type)
{
  struct cmdlist_element *c;
  char *p;

  if (!(cl->next))
    return cl;

  for (c = cl->next; c->next; c = c->next) {
    for (p = c->cmd; *p && isspace(*p); p++);

    if (!strn_cmp("if ", p, 3))
      c = find_end(c);

    else if (!strn_cmp("elseif ", p, 7)) {
      if (process_if(p + 7, go, sc, trig, type)) {
	GET_TRIG_DEPTH(trig)++;
	return c;
      }
    } else if (!strn_cmp("else", p, 4)) {
      GET_TRIG_DEPTH(trig)++;
      return c;
    } else if (!strn_cmp("end", p, 3))
      return c;
  }

  return c;
}


/* processes any 'wait' commands in a trigger */
void process_wait(void *go, trig_data * trig, int type, char *cmd,
		  struct cmdlist_element *cl, long flags)
{
  char buf[MAX_INPUT_LENGTH], *arg;
  struct wait_event_data *wait_event_obj;
  long time, hr, min, ntime;
  char c;

  extern struct time_info_data time_info;
  extern long global_pulse;


  arg = any_one_arg(cmd, buf);
  skip_spaces(&arg);
  
  if (!*arg) {
    sprintf(buf2, "Trigger: %s, VNum %d. wait w/o an arg: '%s'",
	    GET_TRIG_NAME(trig), GET_TRIG_VNUM(trig), cl->cmd);
    script_log(buf2);
  } else if (!strn_cmp(arg, "until ", 6)) {

    /* valid forms of time are 14:30 and 1430 */
    if (sscanf(arg, "until %ld:%ld", &hr, &min) == 2)
      min += (hr * 60);
    else
      min = (hr % 100) + ((hr / 100) * 60);

    /* calculate the pulse of the day of "until" time */
    ntime = (min * SECS_PER_MUD_HOUR * PASSES_PER_SEC) / 60;

    /* calculate pulse of day of current time */
    time = (global_pulse % (SECS_PER_MUD_HOUR * PASSES_PER_SEC)) +
      (time_info.hours * SECS_PER_MUD_HOUR * PASSES_PER_SEC);
    
    if (time >= ntime) /* adjust for next day */
      time = (SECS_PER_MUD_DAY * PASSES_PER_SEC) - time + ntime;
    else
      time = ntime - time;
  } else {
    if (sscanf(arg, "%ld %c", &time, &c) == 2) {
      if (c == 't')
	time *= PULSES_PER_MUD_HOUR;
      else if (c == 's')
	time *= PASSES_PER_SEC;
    }
  }

  CREATE(wait_event_obj, struct wait_event_data, 1);
  wait_event_obj->trigger = trig;
  wait_event_obj->go = go;
  wait_event_obj->type = type;

  GET_TRIG_WAIT(trig) = create_event(trig_wait_event, wait_event_obj, time, flags);
  trig->curr_state = cl->next;
}


/* processes a script set command */
void process_set(struct script_data *sc, trig_data * trig, char *cmd, byte varspace = SPACE_LOCAL)
{
  char arg[MAX_INPUT_LENGTH], name[MAX_INPUT_LENGTH], *value;
  struct trig_var_data *vd = NULL;
  struct trig_var_data **var_list;

  switch (varspace) {
  case SPACE_GLOBAL: var_list = &(sc->global_vars); break;
  case SPACE_SHARED: var_list = &(PROTO_TRIG(GET_TRIG_VNUM(trig))->var_list); break;
  default: var_list = &GET_TRIG_VARS(trig); break;
  }
  
  value = two_arguments(cmd, arg, name);

  skip_spaces(&value);

  if (!*name) {
    sprintf(buf2, "Trigger: %s, VNum %d. set w/o an arg: '%s'",
	    GET_TRIG_NAME(trig), GET_TRIG_VNUM(trig), cmd);
    script_log(buf2);
    return;
  }

  vd = search_for_variable(sc, trig, name);
  
  if (vd) {
    FREE(vd->value);
    CREATE(vd->value, char, strlen(value) + 1);
    strcpy(vd->value, value);
  } else {
    add_var(var_list, name, value, sc->context);
  }

}

/* processes a script eval command */
void process_eval(void *go, struct script_data *sc, trig_data * trig,
		 int type, char *cmd, byte varspace = SPACE_LOCAL)
{
  char arg[MAX_INPUT_LENGTH], name[MAX_INPUT_LENGTH];
  char result[MAX_INPUT_LENGTH], *expr;
  struct trig_var_data *vd = NULL;
  struct trig_var_data **var_list;

  switch (varspace) {
  case SPACE_GLOBAL: var_list = &(sc->global_vars); break;
  case SPACE_SHARED: var_list = &(PROTO_TRIG(GET_TRIG_VNUM(trig))->var_list); break;
  default: var_list = &GET_TRIG_VARS(trig); break;
  }
  
  expr = two_arguments(cmd, arg, name);

  skip_spaces(&expr);

  if (!*name) {
    sprintf(buf2, "Trigger: %s, VNum %d. eval w/o an arg: '%s'",
	    GET_TRIG_NAME(trig), GET_TRIG_VNUM(trig), cmd);
    script_log(buf2);
    return;
  }

  eval_expr(expr, result, go, sc, trig, type);

  vd = search_for_variable(sc, trig, name);

  if (vd) {
    FREE(vd->value);
    CREATE(vd->value, char, strlen(result) + 1);
    strcpy(vd->value, result);
  } else {
    add_var(var_list, name, result, sc->context);
  }

}


// Usage:  pushfront variable value
void process_pushfront(struct script_data *sc, trig_data * trig, char *cmd)
{
  char arg[MAX_INPUT_LENGTH], name[MAX_INPUT_LENGTH], newval[MAX_INPUT_LENGTH], *value;
  struct trig_var_data *vd = NULL;

  value = two_arguments(cmd, arg, name);

  skip_spaces(&value);

  if (!*name) {
    sprintf(buf2, "Trigger: %s, VNum %d. set w/o an arg: '%s'",
	    GET_TRIG_NAME(trig), GET_TRIG_VNUM(trig), cmd);
    script_log(buf2);
    return;
  }

  vd = search_for_variable(sc, trig, name);
  
  if (vd) {
    strcpy(newval, value);
    if (*vd->value)
      strcat(newval, " ");
    strcat(newval, vd->value);
    FREE(vd->value);
    CREATE(vd->value, char, strlen(newval) + 1);
    strcpy(vd->value, newval);
  } else {
    add_var(&GET_TRIG_VARS(trig), name, value, sc->context);
  }

}


// Usage:  popfront variable variable
void process_popfront(struct script_data *sc, trig_data * trig, char *cmd)
{
  char arg[MAX_INPUT_LENGTH], name[MAX_INPUT_LENGTH], newval[MAX_INPUT_LENGTH],
       popped[MAX_INPUT_LENGTH], *popvar, *p;
  struct trig_var_data *vd = NULL;

  popvar = two_arguments(cmd, arg, name);

  skip_spaces(&popvar);

  if (!*name) {
    sprintf(buf2, "Trigger: %s, VNum %d. set w/o an arg: '%s'",
	    GET_TRIG_NAME(trig), GET_TRIG_VNUM(trig), cmd);
    script_log(buf2);
    return;
  }

  vd = search_for_variable(sc, trig, name);
  
  if (vd) {
    half_chop(vd->value, popped, newval);
    FREE(vd->value);
    CREATE(vd->value, char, strlen(newval) + 1);
    strcpy(vd->value, newval);
    
    if (popvar && *popvar) {
      add_var(&GET_TRIG_VARS(trig), popvar, popped ? popped : "", sc->context);
    }
  }
}


// Usage:  pushback variable value
void process_pushback(struct script_data *sc, trig_data * trig, char *cmd)
{
  char arg[MAX_INPUT_LENGTH], name[MAX_INPUT_LENGTH], newval[MAX_INPUT_LENGTH], *value;
  struct trig_var_data *vd = NULL;

  value = two_arguments(cmd, arg, name);

  skip_spaces(&value);

  if (!*name) {
    sprintf(buf2, "Trigger: %s, VNum %d. set w/o an arg: '%s'",
	    GET_TRIG_NAME(trig), GET_TRIG_VNUM(trig), cmd);
    script_log(buf2);
    return;
  }

  vd = search_for_variable(sc, trig, name);
  
  if (vd) {
    strcpy(newval, vd->value);
    if (*newval)
      strcat(newval, " ");
    strcat(newval, value);
    FREE(vd->value);
    CREATE(vd->value, char, strlen(newval) + 1);
    strcpy(vd->value, newval);
  } else {
    add_var(&GET_TRIG_VARS(trig), name, value, sc->context);
  }

}


// Usage:  pushback variable value
void process_popback(struct script_data *sc, trig_data * trig, char *cmd)
{
  char arg[MAX_INPUT_LENGTH], name[MAX_INPUT_LENGTH], newval[MAX_INPUT_LENGTH], 
       *p, *popvar, *popped = NULL;
  struct trig_var_data *vd = NULL;
  int place, len;

  popvar = two_arguments(cmd, arg, name);

  skip_spaces(&popvar);

  if (!*name) {
    sprintf(buf2, "Trigger: %s, VNum %d. set w/o an arg: '%s'",
	    GET_TRIG_NAME(trig), GET_TRIG_VNUM(trig), cmd);
    script_log(buf2);
    return;
  }

  vd = search_for_variable(sc, trig, name);
  
  if (vd) {
    strcpy(newval, vd->value);
    p = newval;
    place = strlen(p) - 1;
    p += place;

    while (place) {
      if (*p && (*p != ' '))
        break;
      --p;
      --place;
    }

    while (place) {
      if (*p && (*p == ' '))
        break;
      --p;
      --place;
    }

    if (place) {
      popped = p + 1;
      *p = '\0';
      p = newval;
      skip_spaces(&p);

      FREE(vd->value);
      CREATE(vd->value, char, strlen(p) + 1);
      strcpy(vd->value, p);
    } else {
      FREE(vd->value);
      CREATE(vd->value, char, 1);
      *vd->value = '\0';
    }

    if (popvar && *popvar) {
      add_var(&GET_TRIG_VARS(trig), popvar, popped ? popped : "", sc->context);
    }
  }
}


/* create a UID variable from the id number */
void makeuid_var(void *go, script_data * sc, trig_data * trig,
		 int type, char *cmd)
{
  char arg[MAX_INPUT_LENGTH], varname[MAX_INPUT_LENGTH];
  char result[MAX_INPUT_LENGTH], *uid_p;
  char uid[MAX_INPUT_LENGTH];
  
  uid_p = two_arguments(cmd, arg, varname);
  skip_spaces(&uid_p);

  if (!*varname) {
    sprintf(buf2, "Trigger: %s, VNum %d. makeuid w/o an arg: '%s'",
	    GET_TRIG_NAME(trig), GET_TRIG_VNUM(trig), cmd);
    script_log(buf2);
    return;
  }
  if (!uid_p || !*uid_p || atoi(uid_p) == 0) {
    sprintf(buf2, "Trigger: %s, VNum %d. makeuid invalid id arg: '%s'",
	    GET_TRIG_NAME(trig), GET_TRIG_VNUM(trig), cmd);
    script_log(buf2);
    return;
  }
  eval_expr(uid_p, result, go, sc, trig, type);
  sprintf(uid, "%c%s", UID_CHAR, result);
  add_var(&GET_TRIG_VARS(trig), varname, uid, sc->context);
}

/*
 * processes a script return command.
 * returns the new value for the script to return.
 */
int process_return(trig_data * trig, char *cmd)
{
  char arg1[MAX_INPUT_LENGTH], arg2[MAX_INPUT_LENGTH];
  
  two_arguments(cmd, arg1, arg2);
  
  if (!*arg2) {
    sprintf(buf2, "Trigger: %s, VNum %d. return w/o an arg: '%s'",
	    GET_TRIG_NAME(trig), GET_TRIG_VNUM(trig), cmd);
    script_log(buf2);
    return 1;
  }

  return atoi(arg2);
}


/*
 * removes a variable from the global vars of sc,
 * or the local vars of trig if not found in global list.
 */
void process_unset(struct script_data *sc, trig_data * trig, char *cmd)
{
  char arg[MAX_INPUT_LENGTH], *var;
  struct trig_var_data *vd = NULL;

  half_chop(cmd, arg, cmd);

  if (!*arg) {
    sprintf(buf2, "Trigger: %s, VNum %d. unset w/o an arg: '%s'",
	    GET_TRIG_NAME(trig), GET_TRIG_VNUM(trig), cmd);
    script_log(buf2);
    return;
  }

  while (*arg) {
    vd = extract_variable(sc, trig, arg);
    if (vd) {
      free_var_el(vd);
    }
    half_chop(cmd, arg, cmd);
  }

#ifdef GOT_RID_OF_IT
  if (!remove_var(&(sc->global_vars), var))
    if (!remove_var(&GET_TRIG_VARS(trig), var))
      for (tmp_trig = TRIGGERS(sc); tmp_trig; tmp_trig = tmp_trig->next) {
        remove_var(&(PROTO_TRIG(GET_TRIG_VNUM(tmp_trig))->var_list), var);
      }
#endif
}


/*
 * copy a locally owned variable to the globals of another script
 *     'remote <variable_name> <uid>'
 */
void process_remote(struct script_data *sc, trig_data * trig, char *cmd)
{
  struct trig_var_data *vd;
  struct script_data *sc_remote = NULL;
  char *line, *var, *uid_p;
  char arg[MAX_INPUT_LENGTH];
  long uid;
  room_data *room;
  char_data *mob;
  obj_data *obj;

  line = any_one_arg(cmd, arg);
  two_arguments(line, buf, buf2);
  var = buf;
  uid_p = buf2;
  skip_spaces(&var);
  skip_spaces(&uid_p);
  

  if (!*buf || !*buf2) {
    sprintf(buf2, "Trigger: %s, VNum %d. remote: invalid arguments '%s'",
	    GET_TRIG_NAME(trig), GET_TRIG_VNUM(trig), cmd);
    script_log(buf2);
    return;
  }

  vd = search_for_variable(sc, trig, var);

  if (!vd) {
    sprintf(buf2, "Trigger: %s, VNum %d. variable '%s' not found in remote call",
	    GET_TRIG_NAME(trig), GET_TRIG_VNUM(trig), buf);
    script_log(buf2);
    return;
  }    
  /* find the target script from the uid number */
  uid = atoi(buf2);
  if (uid <= 0) {
    sprintf(buf, "Trigger: %s, VNum %d. remote: illegal uid '%s'",
            GET_TRIG_NAME(trig), GET_TRIG_VNUM(trig), buf2);
    script_log(buf2);
    return;
  } else if (uid < ROOM_ID_BASE) {
    sprintf(buf, "Trigger: %s, VNum %d. remote: uid '%ld' is a PC?",
            GET_TRIG_NAME(trig), GET_TRIG_VNUM(trig), uid);
    script_log(buf2);
    return;
  }

  if ((room = find_room(uid))) {
    sc_remote = SCRIPT(room);
  } else if ((mob = find_char(uid))) {
    sc_remote = SCRIPT(mob);
  } else if ((obj = find_obj(uid))) {
    sc_remote = SCRIPT(obj);
  } else {
    sprintf(buf, "Trigger: %s, VNum %d. remote: uid '%ld' invalid",
            GET_TRIG_NAME(trig), GET_TRIG_VNUM(trig), uid);
    script_log(buf2);
    return;
  }

  if (sc_remote == NULL)
    return;			/* no script to assign */

  add_var(&(sc_remote->global_vars), vd->name, vd->value, vd->context);
}


/*
 * makes a local variable into a global variable
 */
void process_global(struct script_data *sc, trig_data * trig, char *cmd, long id)
{
  struct trig_var_data *vd = NULL;
  char arg[MAX_INPUT_LENGTH], *var;

  var = any_one_arg(cmd, arg);

  skip_spaces(&var);

  if (!*var) {
    sprintf(buf2, "Trigger: %s, VNum %d. global w/o an arg: '%s'",
	    GET_TRIG_NAME(trig), GET_TRIG_VNUM(trig), cmd);
    script_log(buf2);
    return;
  }

  vd = extract_variable(sc, trig, var, 5);

  if (vd) {
    add_var(&(sc->global_vars), vd->name, vd->value, id);
    return;
  } else if (search_for_variable(sc, trig, var, 2)) {
    return;
  }
  
  sprintf(buf2, "Trigger: %s, VNum %d. variable '%s' not found in global call",
  	  GET_TRIG_NAME(trig), GET_TRIG_VNUM(trig), var);
  script_log(buf2);

}


/*
 * makes a local variable into a shared variable (attached to prototype)
 */
void process_shared(struct script_data *sc, trig_data * trig, char *cmd, long id)
{
  struct trig_var_data *vd;
  char arg[MAX_INPUT_LENGTH], *var;

  var = any_one_arg(cmd, arg);

  skip_spaces(&var);

  if (!*var) {
    sprintf(buf2, "Trigger: %s, VNum %d. global w/o an arg: '%s'",
	    GET_TRIG_NAME(trig), GET_TRIG_VNUM(trig), cmd);
    script_log(buf2);
    return;
  }

  vd = extract_variable(sc, trig, var, 3);

  if (vd) {
    add_var(&(PROTO_TRIG(GET_TRIG_VNUM(trig))->var_list), vd->name, vd->value, id);
    return;
  } else if (search_for_variable(sc, trig, var, 4)) {
    return;
  }
  
  sprintf(buf2, "Trigger: %s, VNum %d. local var '%s' not found in shared call",
  	  GET_TRIG_NAME(trig), GET_TRIG_VNUM(trig), var);
  script_log(buf2);

}


/* set the current context for a script */
void process_context(struct script_data *sc, trig_data * trig, char *cmd)
{
  char arg[MAX_INPUT_LENGTH], *var;

  var = any_one_arg(cmd, arg);

  skip_spaces(&var);

  if (!*var) {
    sprintf(buf2, "Trigger: %s, VNum %d. context w/o an arg: '%s'",
	    GET_TRIG_NAME(trig), GET_TRIG_VNUM(trig), cmd);
    script_log(buf2);
    return;
  }
  sc->context = atol(var);
}


/* event object structure for point regen */
struct delayed_event_script {
  cmdlist_element *cl;
  trig_data *trig;
  long id;
  int type;
};


EVENTFUNC(delayed_script_command)
{
  int depth = 0, ret_val = 0;
  bool done = FALSE;
  char *p;
  char_data *c;
  obj_data *o;
  room_data *r;
  game_data *thing;

  script_data *sc = NULL;
  
  struct delayed_event_script *cmd = (struct delayed_event_script *) event_obj;
 
  if ((c = find_char(cmd->id)))
    thing = (game_data *) c;
  else if ((o = find_obj(cmd->id)))
    thing = (game_data *) o;
  else if ((r = find_room(cmd->id)))
    thing = (game_data *) r;
  else {
    FREE(cmd);
    return;
  }
 
  switch (thing->ClassType()) {
  case DATATYPE_CHAR_DATA:
    sc = SCRIPT((char_data *) thing);
    break;
  case DATATYPE_OBJ_DATA:
    sc = SCRIPT((obj_data *) thing);
    break;
  case DATATYPE_ROOM_DATA:
    sc = SCRIPT((room_data *) thing);
    break;
  default:
    FREE(cmd);
    return;
  }

  if ((!PURGED(sc)) && (!PURGED(cmd->trig))) {
    strcpy(buf, cmd->cl->cmd);

    p = two_arguments(buf, buf2, buf3);
    *buf2 = '\0';
    *buf3 = '\0';

    switch (cmd->type) {
    case SCRIPTC_MOBCMD:
      command_interpreter((char_data *) thing, p);
      break;
    case SCRIPTC_OBJCMD:
      obj_command_interpreter((obj_data *) thing, p);
      break;
    case SCRIPTC_WLDCMD:
      wld_command_interpreter((struct room_data *) thing, p);
      break;
    }
  }

  FREE(cmd);
}


/*
 * Creates a delayed event with the script.
 */
void process_delay(void *go, trig_data * trig, int type, char *cmd,
		  struct cmdlist_element *cl, long flags)
{
  char buf[MAX_INPUT_LENGTH], *arg;
  struct delayed_event_script *delayed = NULL;

  struct event_list *el = NULL;

  long time, hr, min, ntime;
  char c;

  extern struct time_info_data time_info;
  extern long global_pulse;


  arg = any_one_arg(cmd, buf);
  skip_spaces(&arg);

  if (!*arg) {
    sprintf(buf2, "Trigger: %s, VNum %d. delay w/o an arg: '%s'",
	    GET_TRIG_NAME(trig), GET_TRIG_VNUM(trig), cl->cmd);
    script_log(buf2);
  } else if (!strn_cmp(arg, "until ", 6)) {

    /* valid forms of time are 14:30 and 1430 */
    if (sscanf(arg, "until %ld:%ld", &hr, &min) == 2)
      min += (hr * 60);
    else
      min = (hr % 100) + ((hr / 100) * 60);

    /* calculate the pulse of the day of "until" time */
    ntime = (min * SECS_PER_MUD_HOUR * PASSES_PER_SEC) / 60;

    /* calculate pulse of day of current time */
    time = (global_pulse % (SECS_PER_MUD_HOUR * PASSES_PER_SEC)) +
      (time_info.hours * SECS_PER_MUD_HOUR * PASSES_PER_SEC);

    if (time >= ntime)		/* adjust for next day */
      time = (SECS_PER_MUD_DAY * PASSES_PER_SEC) - time + ntime;
    else
      time = ntime - time;
  } else {
    if (sscanf(arg, "%ld %c", &time, &c) == 2) {
      if (c == 't')
	time *= PULSES_PER_MUD_HOUR;
      else if (c == 's')
	time *= PASSES_PER_SEC;
    }
  }

  CREATE(delayed, delayed_event_script, 1);
  delayed->cl = cl;
  delayed->trig = trig;
  delayed->id = ((game_data *) go)->Id();
  delayed->type = cl->type;

  /* add event to damage event list */
  CREATE(el, struct event_list, 1);
  el->type = type;
  el->event = create_event(delayed_script_command, delayed, time, flags);
  el->next = GET_TRIG_DELAYED(trig);
  GET_TRIG_DELAYED(trig) = el;
}


void extract_value(struct script_data *sc, trig_data * trig, char *cmd)
{
  char buf[MAX_INPUT_LENGTH];
  char buf2[MAX_INPUT_LENGTH];
  char *buf3;
  char to[128];
  int num;

  buf3 = any_one_arg(cmd, buf);
  half_chop(buf3, buf2, buf);
  strcpy(to, buf2);

  num = atoi(buf);
  if (num < 1) {
    script_log("extract number < 1!");
    return;
  }

  half_chop(buf, buf3, buf2);

  while (num>0) {
    half_chop(buf2, buf, buf2);
    num--;
  }

  add_var(&GET_TRIG_VARS(trig), to, buf, sc->context);
}

/*  This is the core driver for scripts. */
int script_driver(void *go, trig_data *trig, int type, int mode)
{
  static int depth = 0;
  int ret_val = 1;
  register struct cmdlist_element *cl;
  register char *p;
  char cmd[MAX_INPUT_LENGTH], *loopp;
  struct script_data *sc = 0;
  struct cmdlist_element *temp;
  // unsigned long loops = 0;
  bool interpreted = TRUE;
  bool halted = FALSE;
  bool breakhit = FALSE;
  sh_int linetype = SCRIPTC_NONE;

  if (depth > MAX_SCRIPT_DEPTH) {
    script_log("Triggers recursed beyond maximum allowed depth.");
    return ret_val;
  }

  ++depth;

  switch (type) {
  case MOB_TRIGGER:
    sc = SCRIPT((char_data *) go);
    break;
  case OBJ_TRIGGER:
    sc = SCRIPT((obj_data *) go);
    break;
  case WLD_TRIGGER:
    sc = SCRIPT((struct room_data *) go);
    break;
  }

  if (mode == TRIG_NEW) {
    GET_TRIG_DEPTH(trig) = 1;
    GET_TRIG_LOOPS(trig) = 0;
    sc->context = 0;
    for (cl = trig->cmdlist->command; cl; cl = cl->next) {
      cl->loopcount = 0;
    }
  }

  for (cl = (mode == TRIG_NEW) ? trig->cmdlist->command : trig->curr_state;
        (!halted) && cl && GET_TRIG_DEPTH(trig); cl = cl->next) {
    if (PURGED(sc))
      return 0;
    if (PURGED(trig))
      return 0;

    linetype = cl->type;
    interpreted = TRUE;
    breakhit = FALSE;
    
    if (linetype == SCRIPTC_COMMENT)
      continue;
    
    for (p = cl->cmd; *p && isspace(*p); ++p);
    
    if (!*p)
      continue;
    
    switch (linetype) {
    case SCRIPTC_IF:
      if (process_if(p + 3, go, sc, trig, type))
        GET_TRIG_DEPTH(trig)++;
      else
        cl = find_else_end(trig, cl, go, sc, type);
      break;
    case SCRIPTC_ELSEIF:
    case SCRIPTC_ELSE:
      cl = find_end(cl);
      GET_TRIG_DEPTH(trig)--;
      break;
    case SCRIPTC_WHILE:
      temp = find_done(cl);
      if (process_if(p + 6, go, sc, trig, type)) {
        temp->original = cl;
      }
      else {
        (cl->loopcount) = 0;
        cl = temp;
      }
      break;
    case SCRIPTC_FOREACH:
      temp = find_done(cl);
      if (process_foreach(p + 8, go, sc, trig, type, cl->loopcount)) {
        temp->original = cl;
        cl->loopcount += 1;  // Foreach operates differently from while.  We need to increment this
                             // the first time because the processing of done will call foreach without
                             // incrementing the loopcount first.
      } else {
        (cl->loopcount) = 0;
        cl = temp;
      }
      break;
    case SCRIPTC_SWITCH:
      cl = find_case(trig, cl, go, sc, type, p + 7);
      break;
    case SCRIPTC_END:
      GET_TRIG_DEPTH(trig)--;
      break;
    case SCRIPTC_DONE:
      if (cl->original) {
        for (loopp = cl->original->cmd; *loopp && isspace(*loopp); ++loopp);
 
        if (((cl->original->type == SCRIPTC_WHILE) &&
           (process_if(loopp + 6, go, sc, trig, type))) ||
           ((cl->original->type == SCRIPTC_FOREACH) &&
           (process_foreach(loopp + 8, go, sc, trig, type, cl->original->loopcount)))) {
          cl = cl->original;
          cl->loopcount += 1;
          ++GET_TRIG_LOOPS(trig);
          if ((cl->loopcount % 30) == 0) {
            // cl->loopcount = 0;
            process_wait(go, trig, type, "wait 5", cl, 0);
            depth--;
            return ret_val;
          }
          if (GET_TRIG_LOOPS(trig) == 100) {
            char *buf = (char *) malloc(MAX_STRING_LENGTH);

            sprintf(buf, "SCRIPTERR: Trigger VNum %d has looped 100 times!!!",
                    GET_TRIG_VNUM(trig));
            mudlog(buf, NRM, TRUST_CREATOR, TRUE);
            free(buf);
          }
        }
      }
      break;
    case SCRIPTC_BREAK:
      cl = find_done(cl);
      if (cl->original)
        cl->original->loopcount = 0;
      breakhit = TRUE;
      break;
    case SCRIPTC_CASE:
    case SCRIPTC_DEFAULT:
      break;
    default:
      interpreted = FALSE;
      break;
    }

    if (breakhit)
      continue;

    if (interpreted == FALSE) {

      var_subst(go, sc, trig, type, p, cmd);

      if (!linetype)
        linetype = lookup_script_line(cmd);

      switch (linetype) {
      case SCRIPTC_CONTEXT:
        process_context(sc, trig, cmd);
        break;
      case SCRIPTC_DELAY:
        process_delay(go, trig, type, cmd, cl, 0);
        break;
      case SCRIPTC_EVAL:
        process_eval(go, sc, trig, type, cmd);
        break;
      case SCRIPTC_EVALGLOBAL:
        process_eval(go, sc, trig, type, cmd, SPACE_GLOBAL);
        break;
      case SCRIPTC_EVALSHARED:
        process_eval(go, sc, trig, type, cmd, SPACE_SHARED);
        break;
      case SCRIPTC_EXTRACT:
        extract_value(sc, trig, cmd);
        break;
      case SCRIPTC_GLOBAL:
        process_global(sc, trig, cmd, sc->context);
        break;
      case SCRIPTC_HALT:
        halted = TRUE;
        break;
      case SCRIPTC_MAKEUID:
        makeuid_var(go, sc, trig, type, cmd);
        break;
      case SCRIPTC_PUSHFRONT:
        process_pushfront(sc, trig, cmd);
        break;
      case SCRIPTC_POPFRONT:
        process_popfront(sc, trig, cmd);
        break;
      case SCRIPTC_PUSHBACK:
        process_pushback(sc, trig, cmd);
        break;
      case SCRIPTC_POPBACK:
        process_popback(sc, trig, cmd);
        break;
      case SCRIPTC_REMOTE:
        process_remote(sc, trig, cmd);
        break;
      case SCRIPTC_RETURN:
        ret_val = process_return(trig, cmd);
        halted = TRUE;
        break;
      case SCRIPTC_SET:
        process_set(sc, trig, cmd);
        break;
      case SCRIPTC_SETGLOBAL:
        process_set(sc, trig, cmd, SPACE_GLOBAL);
        break;
      case SCRIPTC_SETSHARED:
        process_set(sc, trig, cmd, SPACE_SHARED);
        break;
      case SCRIPTC_SHARED:
        process_shared(sc, trig, cmd, sc->context);
        break;
      case SCRIPTC_UNSET:
        process_unset(sc, trig, cmd);
        break;
      case SCRIPTC_WAIT:
        process_wait(go, trig, type, cmd, cl, 0);
        depth--;
        return ret_val;
      case SCRIPTC_SCRIPTCMD:
        genscript_command_interpreter((game_data *) go, cmd, cl->number);
        break;

      default:
        switch (type) {
        case MOB_TRIGGER:
          if (linetype && linetype != SCRIPTC_MOBCMD)
            break;
          command_interpreter((char_data *) go, cmd, cl->number);
          break;
        case OBJ_TRIGGER:
          if (linetype && linetype != SCRIPTC_OBJCMD)
            break;
          obj_command_interpreter((obj_data *) go, cmd, cl->number);
          break;
        case WLD_TRIGGER:
          if (linetype && linetype != SCRIPTC_WLDCMD)
            break;
          wld_command_interpreter((struct room_data *) go, cmd, cl->number);
          break;
        }
      }
    }
  }

  free_varlist(GET_TRIG_VARS(trig));
  GET_TRIG_VARS(trig) = NULL;
  GET_TRIG_DEPTH(trig) = 0;

  depth--;
  return ret_val;
}


/*
   * scans for a case/default instance
   * returns the line containg the correct case instance, or the last
   * line of the trigger if not found.
 */
struct cmdlist_element *
  find_case(struct trig_data *trig, struct cmdlist_element *cl,
          void *go, struct script_data *sc, int type, char *cond)
{
  struct cmdlist_element *c;
  char *p, *buf;

  if (!(cl->next))
    return cl;  
        
  for (c = cl->next; c->next; c = c->next) {
      
    if ((c->type == SCRIPTC_WHILE) || (c->type == SCRIPTC_SWITCH))
      c = find_done(c);
    else if (c->type == SCRIPTC_CASE) {
      for (p = c->cmd; *p && isspace(*p); p++);
      buf = (char *) malloc(MAX_STRING_LENGTH);
      sprintf(buf, "(%s) == (%s)", cond, p + 5);
      if (process_if(buf, go, sc, trig, type)) {
        free(buf);
        return c;
      }
      free(buf);
    } else if (c->type == SCRIPTC_DEFAULT)
      return c;
    else if (c->type == SCRIPTC_DONE)   
     return c;
  }
  return c;
}        
       
/*
   * scans for end of while/switch-blocks.   
   * returns the line containg 'end', or the last
   * line of the trigger if not found.     
 */
struct cmdlist_element *
  find_done(struct cmdlist_element *cl)
{
  struct cmdlist_element *c;
  char *p;
  
  if (!(cl->next))
    return cl;

  for (c = cl->next; c->next; c = c->next) {
    if ((c->type == SCRIPTC_WHILE) || (c->type == SCRIPTC_SWITCH))
      c = find_done(c);
    else if (c->type == SCRIPTC_DONE)
      return c;
  }
    
  return c;
}


cmdlist_element::cmdlist_element() {
  bzero(this, sizeof(cmdlist_element));
}


cmdlist_element::cmdlist_element(const char * argument) {
  int cmdnum = 0;
  char *p;

  bzero(this, sizeof(cmdlist_element));

  if (*argument)
    cmd = str_dup(argument);
  else {
    cmd = NULL;
    return;
  }

  p = (char *) argument;
  skip_spaces(&p);
  
  if (*p == '%')
    return;

  half_chop(p, buf2, buf1);

  type = lookup_script_line(p);
  if (type) {
    // sprintf(buf, "Command:  %s Type:  %d", buf2, type);
    // log (buf);
    return;
  }

  // log ("Checking room command table");
  for (cmdnum = 0; *script_cmd_info[cmdnum].command != '\n'; ++cmdnum) {
    if (!strncmp(script_cmd_info[cmdnum].command, buf2, 
        strlen(script_cmd_info[cmdnum].command))) {
      break;
    }
  }

  if (*script_cmd_info[cmdnum].command != '\n') {
    number = cmdnum;
    type = SCRIPTC_SCRIPTCMD;
    return;
  }

  // log ("Checking mobile command table");
  for (cmdnum = 0; *cmd_info[cmdnum].command != '\n'; ++cmdnum) {
    if (is_abbrev(buf2, cmd_info[cmdnum].command))
      break;
  }

  if (*cmd_info[cmdnum].command != '\n') {
    number = cmdnum;
    type = SCRIPTC_MOBCMD;
    return;
  }

  // log ("Checking object command table");
  for (cmdnum = 0; *obj_cmd_info[cmdnum].command != '\n'; ++cmdnum) {
    if (!strncmp(obj_cmd_info[cmdnum].command, buf2, 
        strlen(obj_cmd_info[cmdnum].command))) {
      break;
    }
  }

  if (*obj_cmd_info[cmdnum].command != '\n') {
    number = cmdnum;
    type = SCRIPTC_OBJCMD;
    return;
  }

  // log ("Checking room command table");
  for (cmdnum = 0; *wld_cmd_info[cmdnum].command != '\n'; ++cmdnum) {
    if (!strncmp(wld_cmd_info[cmdnum].command, buf2, 
        strlen(wld_cmd_info[cmdnum].command))) {
      break;
    }
  }

  if (*wld_cmd_info[cmdnum].command != '\n') {
    number = cmdnum;
    type = SCRIPTC_WLDCMD;
    return;
  }

  // sprintf(buf, "Command:  %s Number:  %d", buf2, number);
  // log (buf);

}


cmdlist_element::~cmdlist_element() {
  if (cmd)
    delete (cmd);
  cmd = NULL;
}


// Accepts a space-skipped line of a trigger and checks for a script command, 
// returning the value.  This is for script_driver().
inline script_command_switches 
lookup_script_line (char *p) 
{
  script_command_switches ret_val = SCRIPTC_NONE;

  switch (*p) {
  case '*':
    ret_val = SCRIPTC_COMMENT;
    break;

  case 'b':
    if (!strn_cmp(p, "break", 5))
      ret_val = SCRIPTC_BREAK;
    break;

  case 'c':
    if (!strn_cmp(p, "case", 4))
      ret_val = SCRIPTC_CASE;
    else if (!strn_cmp(p, "context ", 8))
      ret_val = SCRIPTC_CONTEXT;
    break;

  case 'd':
    if (!strn_cmp(p, "done", 4))
      ret_val = SCRIPTC_DONE;
    else if (!strn_cmp(p, "delay ", 6))
      ret_val = SCRIPTC_DELAY;
    else if (!strn_cmp(p, "default", 7))
      ret_val = SCRIPTC_DEFAULT;
    break;

  case 'e':
    if (!strn_cmp(p, "end", 3))
      ret_val = SCRIPTC_END;
    else if (!strn_cmp(p, "else", 4))
      ret_val = SCRIPTC_ELSE;
    else if (!strn_cmp(p, "elseif ", 7))
      ret_val = SCRIPTC_ELSEIF;
    else if (!strn_cmp(p, "eval ", 5))
      ret_val = SCRIPTC_EVAL;
    else if (!strn_cmp(p, "evalglobal ", 11))
      ret_val = SCRIPTC_EVALGLOBAL;
    else if (!strn_cmp(p, "evalshared ", 11))
      ret_val = SCRIPTC_EVALSHARED;
    else if (!strn_cmp(p, "extract ", 8))
      ret_val = SCRIPTC_EXTRACT;
    break;

  case 'f':
    if (!strn_cmp(p, "foreach ", 8))
      ret_val = SCRIPTC_FOREACH;
    break;

  case 'g':
    if (!strn_cmp(p, "global ", 7))
      ret_val = SCRIPTC_GLOBAL;
    break;

  case 'h':
    if (!strn_cmp(p, "halt", 4))
      ret_val = SCRIPTC_HALT;
    break;

  case 'i':
    if (!strn_cmp(p, "if ", 3))
      ret_val = SCRIPTC_IF;
    break;

  case 'm':
    if (!strn_cmp(p, "makeuid ", 8))
      ret_val = SCRIPTC_MAKEUID;
    break;

  case 'p':
    if (!strn_cmp(p, "popback ", 8))
      ret_val = SCRIPTC_POPBACK;
    else if (!strn_cmp(p, "popfront ", 9))
      ret_val = SCRIPTC_POPFRONT;
    else if (!strn_cmp(p, "pushback ", 9))
      ret_val = SCRIPTC_PUSHBACK;
    else if (!strn_cmp(p, "pushfront ", 10))
      ret_val = SCRIPTC_PUSHFRONT;
    break;

  case 'r':
    if (!strn_cmp(p, "remote ", 7))
      ret_val = SCRIPTC_REMOTE;
    else if (!strn_cmp(p, "return ", 7))
      ret_val = SCRIPTC_RETURN;
    break;

  case 's':
    if (!strn_cmp(p, "set ", 4))
      ret_val = SCRIPTC_SET;
    else if (!strn_cmp(p, "switch ", 7))
      ret_val = SCRIPTC_SWITCH;
    else if (!strn_cmp(p, "setglobal ", 10))
      ret_val = SCRIPTC_SETGLOBAL;
    else if (!strn_cmp(p, "setshared ", 10))
      ret_val = SCRIPTC_SETSHARED;
    else if (!strn_cmp(p, "shared ", 7))
      ret_val = SCRIPTC_SHARED;
    break;

  case 'u':
    if (!strn_cmp(p, "unset ", 6))
      ret_val = SCRIPTC_UNSET;
    break;

  case 'w':
    if (!strn_cmp(p, "wait ", 5))
      ret_val = SCRIPTC_WAIT;
    else if (!strn_cmp(p, "while ", 6))
      ret_val = SCRIPTC_WHILE;
    break;

  default:
    break;
  }

  return ret_val;
}
