/************************************************************************
 * OasisOLC - sedit.c						v1.5	*
 * Copyright 1996 Harvey Gilpin.					*
 ************************************************************************/

#include "conf.h"
#include "sysdep.h"

#include "structs.h"
#include "scripts.h"
#include "index.h"
#include "comm.h"
#include "spells.h"
#include "utils.h"
#include "db.h"
#include "boards.h"
#include "olc.h"
#include "constants.h"
#include "events.h"
#include "shop.h"

/*-------------------------------------------------------------------*/

/*
 * External variable declarations.
 */
extern struct shop_data *shop_index;
extern int top_shop;
// extern struct char_data *mob_proto;
// extern struct obj_data *obj_proto;
// extern struct room_data *world;
// extern struct zone_data *zone_table;
// extern struct index_data *mob_index;
// extern struct index_data *obj_index;
// extern char *trade_letters[];
// extern char *shop_bits[];
// extern char *item_types[];

/*-------------------------------------------------------------------*/

/*
 * Handy macros.
 */
#define S_NUM(i)		((i)->virt)
#define S_KEEPER(i)		((i)->keeper)
#define S_OPEN1(i)		((i)->open1)
#define S_CLOSE1(i)		((i)->close1)
#define S_OPEN2(i)		((i)->open2)
#define S_CLOSE2(i)		((i)->close2)
#define S_BANK(i)		((i)->bankAccount)
#define S_BROKE_TEMPER(i)	((i)->temper1)
#define S_BITVECTOR(i)		((i)->bitvector)
#define S_NOTRADE(i)		((i)->with_who)
#define S_SORT(i)		((i)->lastsort)
#define S_BUYPROFIT(i)		((i)->profit_buy)
#define S_SELLPROFIT(i)		((i)->profit_sell)
#define S_FUNC(i)		((i)->func)

#define S_ROOMS(i)		((i)->in_room)
#define S_PRODUCTS(i)		((i)->producing)
#define S_NAMELISTS(i)		((i)->type)
#define S_ROOM(i, num)		((i)->in_room[(num)])
#define S_PRODUCT(i, num)	((i)->producing[(num)])
#define S_BUYTYPE(i, num)	(BUY_TYPE((i)->type[(num)]))
#define S_BUYWORD(i, num)	(BUY_WORD((i)->type[(num)]))

#define S_NOITEM1(i)		((i)->no_such_item1)
#define S_NOITEM2(i)		((i)->no_such_item2)
#define S_NOCASH1(i)		((i)->missing_cash1)
#define S_NOCASH2(i)		((i)->missing_cash2)
#define S_NOBUY(i)		((i)->do_not_buy)
#define S_BUY(i)		((i)->message_buy)
#define S_SELL(i)		((i)->message_sell)
#define S_PRAC_NOT_KNOWN(i)		((i)->prac_not_known)
#define S_PRAC_MISSING_CASH(i)		((i)->prac_missing_cash)

#define PROTO_MOB(num)  ((char_data *) mob_index[(num)]->proto)
#define PROTO_OBJ(num)  ((obj_data *) obj_index[(num)]->proto)
/*-------------------------------------------------------------------*/

/*
 * Function prototypes.
 */
int real_shop(int vshop_num);
void sedit_setup_new(char_data *ch);
void sedit_setup_existing(char_data *ch, int rmob_num);
void sedit_parse(char_data *ch, char *arg);
void sedit_disp_menu(char_data *ch);
void sedit_namelist_menu(char_data *ch);
void sedit_types_menu(char_data *ch);
void sedit_products_menu(char_data *ch);
void sedit_rooms_menu(char_data *ch);
void sedit_compact_rooms_menu(char_data *ch);
void sedit_shop_flags_menu(char_data *ch);
void sedit_no_trade_menu(char_data *ch);
void sedit_save_internally(char_data *ch);
void sedit_save_to_disk(int zone);
void copy_shop(struct shop_data *tshop, struct shop_data *fshop);
void copy_list(int **tlist, int *flist);
void copy_type_list(struct shop_buy_data **tlist, struct shop_buy_data *flist);
void sedit_add_to_type_list(struct shop_buy_data **list, struct shop_buy_data *newlist);
void sedit_remove_from_type_list(struct shop_buy_data **list, int num);
void free_shop_strings(struct shop_data *shop);
void free_type_list(struct shop_buy_data **list);
void free_shop(struct shop_data *shop);
void sedit_modify_string(char **str, char *newstring);

/*
 * External functions.
 */
SPECIAL(shop_keeper);

/*-------------------------------------------------------------------*\
  utility functions 
\*-------------------------------------------------------------------*/

void sedit_setup_new(char_data *ch)
{
  struct shop_data *shop;

  /*
   * Allocate a scratch shop structure.
   */
  CREATE(shop, struct shop_data, 1);

  /*
   * Fill in some default values.
   */
  S_KEEPER(shop) = -1;
  S_CLOSE1(shop) = 28;
  S_BUYPROFIT(shop) = 1.0;
  S_SELLPROFIT(shop) = 1.0;
  /*
   * Add a spice of default strings.
   */
  S_NOITEM1(shop) = str_dup("%s Sorry, I don't stock that item.");
  S_NOITEM2(shop) = str_dup("%s You don't seem to have that.");
  S_NOCASH1(shop) = str_dup("%s I can't afford that!");
  S_NOCASH2(shop) = str_dup("%s You are too poor!");
  S_NOBUY(shop) = str_dup("%s I don't trade in such items.");
  S_BUY(shop) = str_dup("%s That'll be %d coins, thanks.");
  S_SELL(shop) = str_dup("%s I'll give you %d coins for that.");
  S_PRAC_NOT_KNOWN(shop) = str_dup("%s I can't teach you that, sorry.");
  S_PRAC_MISSING_CASH(shop) = str_dup("%s I can't teach for free!");
  /*
   * Stir the lists lightly.
   */
  CREATE(S_PRODUCTS(shop), int, 1);

  S_PRODUCT(shop, 0) = -1;
  CREATE(S_ROOMS(shop), int, 1);

  S_ROOM(shop, 0) = -1;
  CREATE(S_NAMELISTS(shop), struct shop_buy_data, 1);

  S_BUYTYPE(shop, 0) = -1;

  /*
   * Presto! A shop.
   */
  OLC_SHOP(ch) = shop;
  OLC_SAVE(ch) = OLC_SAVE_NO;
  sedit_disp_menu(ch);
}

/*-------------------------------------------------------------------*/

void sedit_setup_existing(char_data *ch, int rshop_num)
{
  /*
   * Create a scratch shop structure.
   */
  CREATE(OLC_SHOP(ch), struct shop_data, 1);

  copy_shop(OLC_SHOP(ch), shop_index + rshop_num);
  OLC_SAVE(ch) = OLC_SAVE_NO;
  sedit_disp_menu(ch);
}

/*-------------------------------------------------------------------*/

void copy_shop(struct shop_data *tshop, struct shop_data *fshop)
{
  /*
   * Copy basic information over.
   */
  S_NUM(tshop) = S_NUM(fshop);
  S_KEEPER(tshop) = S_KEEPER(fshop);
  S_OPEN1(tshop) = S_OPEN1(fshop);
  S_CLOSE1(tshop) = S_CLOSE1(fshop);
  S_OPEN2(tshop) = S_OPEN2(fshop);
  S_CLOSE2(tshop) = S_CLOSE2(fshop);
  S_BANK(tshop) = S_BANK(fshop);
  S_BROKE_TEMPER(tshop) = S_BROKE_TEMPER(fshop);
  S_BITVECTOR(tshop) = S_BITVECTOR(fshop);
  S_NOTRADE(tshop) = S_NOTRADE(fshop);
  S_SORT(tshop) = S_SORT(fshop);
  S_BUYPROFIT(tshop) = S_BUYPROFIT(fshop);
  S_SELLPROFIT(tshop) = S_SELLPROFIT(fshop);
  S_FUNC(tshop) = S_FUNC(fshop);

  /*
   * Copy lists over.
   */
  copy_list(&(S_ROOMS(tshop)), S_ROOMS(fshop));
  copy_list(&(S_PRODUCTS(tshop)), S_PRODUCTS(fshop));
  copy_type_list(&(tshop->type), fshop->type);

  /*
   * Copy notification strings over.
   */
  free_shop_strings(tshop);
  S_NOITEM1(tshop) = str_dup(S_NOITEM1(fshop));
  S_NOITEM2(tshop) = str_dup(S_NOITEM2(fshop));
  S_NOCASH1(tshop) = str_dup(S_NOCASH1(fshop));
  S_NOCASH2(tshop) = str_dup(S_NOCASH2(fshop));
  S_NOBUY(tshop) = str_dup(S_NOBUY(fshop));
  S_BUY(tshop) = str_dup(S_BUY(fshop));
  S_SELL(tshop) = str_dup(S_SELL(fshop));
  S_PRAC_NOT_KNOWN(tshop) = str_dup(S_PRAC_NOT_KNOWN(fshop));
  S_PRAC_MISSING_CASH(tshop) = str_dup(S_PRAC_MISSING_CASH(fshop));

}

/*-------------------------------------------------------------------*/

/*
 * Copy a -1 terminated integer array list.
 */
void copy_list(int **tlist, int *flist)
{
  int num_items, i;

  if (*tlist)
    free(*tlist);

  /*
   * Count number of entries.
   */
  for (i = 0; flist[i] != -1; i++);
  num_items = i + 1;

  /*
   * Make space for entries.
   */
  CREATE(*tlist, int, num_items);

  /*
   * Copy entries over.
   */
  i = 0;
  do {
    (*tlist)[i] = flist[i];
  } while (++i < num_items);
}

/*-------------------------------------------------------------------*/

/*
 * Copy a -1 terminated (in the type field) shop_buy_data 
 * array list.
 */
void copy_type_list(struct shop_buy_data **tlist, struct shop_buy_data *flist)
{
  int num_items, i;

  if (*tlist)
    free_type_list(tlist);

  /*
   * Count number of entries.
   */
  for (i = 0; BUY_TYPE(flist[i]) != -1; i++);
  num_items = i + 1;

  /*
   * Make space for entries.
   */
  CREATE(*tlist, struct shop_buy_data, num_items);

  /*
   * Copy entries over.
   */
  i = 0;
  do {
    (*tlist)[i].type = flist[i].type;
    if (BUY_WORD(flist[i]))
      BUY_WORD((*tlist)[i]) = str_dup(BUY_WORD(flist[i]));
  } while (++i < num_items);
}

/*-------------------------------------------------------------------*/

void sedit_remove_from_type_list(struct shop_buy_data **list, int num)
{
  int i, num_items;
  struct shop_buy_data *nlist;

  /*
   * Count number of entries.
   */
  for (i = 0; (*list)[i].type != -1; i++);

  if (num >= i || num < 0)
    return;
  num_items = i;

  CREATE(nlist, struct shop_buy_data, num_items);

  for (i = 0; i < num_items; i++)
    nlist[i] = (i < num) ? (*list)[i] : (*list)[i + 1];

  free(BUY_WORD((*list)[num]));
  free(*list);
  *list = nlist;
}

/*-------------------------------------------------------------------*/

void sedit_add_to_type_list(struct shop_buy_data **list, struct shop_buy_data *newlist)
{
  int i, num_items;
  struct shop_buy_data *nlist;

  /*
   * Count number of entries.
   */
  for (i = 0; (*list)[i].type != -1; i++);
  num_items = i;

  /*
   * Make a new list and slot in the new entry.
   */
  CREATE(nlist, struct shop_buy_data, num_items + 2);

  for (i = 0; i < num_items; i++)
    nlist[i] = (*list)[i];
  nlist[num_items] = *newlist;
  nlist[num_items + 1].type = -1;

  /*
   * Out with the old, in with the new.
   */
  free(*list);
  *list = nlist;
}

/*-------------------------------------------------------------------*/

void sedit_add_to_int_list(int **list, int newnum)
{
  int i, num_items, *nlist;

  /*
   * Count number of entries.
   */
  for (i = 0; (*list)[i] != -1; i++);
  num_items = i;

  /*
   * Make a new list and slot in the new entry.
   */
  CREATE(nlist, int, num_items + 2);

  for (i = 0; i < num_items; i++)
    nlist[i] = (*list)[i];
  nlist[num_items] = newnum;
  nlist[num_items + 1] = -1;

  /*
   * Out with the old, in with the new.
   */
  free(*list);
  *list = nlist;
}

/*-------------------------------------------------------------------*/

void sedit_remove_from_int_list(int **list, int num)
{
  int i, num_items, *nlist;

  /*
   * Count number of entries.
   */
  for (i = 0; (*list)[i] != -1; i++);

  if (num >= i || num < 0)
    return;
  num_items = i;

  CREATE(nlist, int, num_items);

  for (i = 0; i < num_items; i++)
    nlist[i] = (i < num) ? (*list)[i] : (*list)[i + 1];

  free(*list);
  *list = nlist;
}

/*-------------------------------------------------------------------*/

/*
 * Free all the notice character strings in a shop structure.
 */
void free_shop_strings(struct shop_data *shop)
{
  if (S_NOITEM1(shop)) {
    free(S_NOITEM1(shop));
    S_NOITEM1(shop) = NULL;
  }
  if (S_NOITEM2(shop)) {
    free(S_NOITEM2(shop));
    S_NOITEM2(shop) = NULL;
  }
  if (S_NOCASH1(shop)) {
    free(S_NOCASH1(shop));
    S_NOCASH1(shop) = NULL;
  }
  if (S_NOCASH2(shop)) {
    free(S_NOCASH2(shop));
    S_NOCASH2(shop) = NULL;
  }
  if (S_NOBUY(shop)) {
    free(S_NOBUY(shop));
    S_NOBUY(shop) = NULL;
  }
  if (S_BUY(shop)) {
    free(S_BUY(shop));
    S_BUY(shop) = NULL;
  }
  if (S_SELL(shop)) {
    free(S_SELL(shop));
    S_SELL(shop) = NULL;
  }
  if (S_PRAC_NOT_KNOWN(shop)) {
    free(S_PRAC_NOT_KNOWN(shop));
    S_PRAC_NOT_KNOWN(shop) = NULL;
  }
  if (S_PRAC_MISSING_CASH(shop)) {
    free(S_PRAC_MISSING_CASH(shop));
    S_PRAC_MISSING_CASH(shop) = NULL;
  }
}

/*-------------------------------------------------------------------*/

/*
 * Free a type list and all the strings it contains.
 */
void free_type_list(struct shop_buy_data **list)
{
  int i;

  for (i = 0; (*list)[i].type != -1; i++)
    if (BUY_WORD((*list)[i]))
      free(BUY_WORD((*list)[i]));
  free(*list);
  *list = NULL;
}

/*-------------------------------------------------------------------*/

/*
 * Free up the whole shop structure and it's content.
 */
void free_shop(struct shop_data *shop)
{
  free_shop_strings(shop);
  free_type_list(&(S_NAMELISTS(shop)));
  free(S_ROOMS(shop));
  free(S_PRODUCTS(shop));
  free(shop);
}

/*-------------------------------------------------------------------*/

int real_shop(int vshop_num)
{
  int rshop_num;

  for (rshop_num = 0; rshop_num < top_shop; rshop_num++)
    if (SHOP_NUM(rshop_num) == vshop_num)
      return rshop_num;

  return -1;
}

/*-------------------------------------------------------------------*/

/*
 * Generic string modifyer for shop keeper messages.
 */
void sedit_modify_string(char **str, char *newstring)
{
  char *pointer;

  /*
   * Check the '%s' is present, if not, add it.
   */
  if (*newstring != '%') {
    strcpy(buf, "%s ");
    strcat(buf, newstring);
    pointer = buf;
  } else
    pointer = newstring;

  if (*str)
    free(*str);
  *str = str_dup(pointer);
}

/*-------------------------------------------------------------------*/

void sedit_save_internally(char_data *ch)
{
  int rshop, found = 0;
  struct shop_data *shop;
  struct shop_data *new_index;

  rshop = real_shop(OLC_NUM(ch));
  shop = OLC_SHOP(ch);
  S_NUM(shop) = OLC_NUM(ch);

  if (rshop > -1) {	/* The shop already exists, just update it. */
    copy_shop((shop_index + rshop), shop);
  } else {		/* Doesn't exist - have to insert it. */
    CREATE(new_index, struct shop_data, top_shop + 1);

    for (rshop = 0; rshop < top_shop; rshop++) {
      if (!found) {	/* Is this the place? */
	if (SHOP_NUM(rshop) > OLC_NUM(ch)) {	/* Yep, stick it in here. */
	  found = 1;
	  copy_shop(&(new_index[rshop]), shop);
	  /*
	   * Move the entry that used to go here up a place.
	   */
	  new_index[rshop + 1] = shop_index[rshop];
	} else	/* This isn't the place, copy over info. */
	  new_index[rshop] = shop_index[rshop];
      } else {	/* Shop's already inserted, copy rest over. */
	new_index[rshop + 1] = shop_index[rshop];
      }
    }
    if (!found)
      copy_shop(&(new_index[rshop]), shop);

    /*
     * Switch the new index in.
     */
    free(shop_index);
    shop_index = new_index;
    top_shop++;
  }
  olc_add_to_save_list(zone_table[OLC_ZNUM(ch)].number, OLC_SEDIT);
}

/**************************************************************************
 Menu functions 
 **************************************************************************/

void sedit_products_menu(char_data *ch)
{
  struct shop_data *shop;
  int i;

  shop = OLC_SHOP(ch);

#if defined(CLEAR_SCREEN)
  send_to_char("[H[J", ch);
#endif
  send_to_char("##     VNUM     Product\r\n", ch);
  for (i = 0; S_PRODUCT(shop, i) != -1; i++) {
    sprintf(buf, "%2d - [&cC%5d&c0] - &cY%s&c0\r\n", i,
	    S_PRODUCT(shop, i), 
	    (obj_index[S_PRODUCT(shop, i)] == NULL) ?
	    "*Undefined*" : 
	    ss_data(PROTO_OBJ(S_PRODUCT(shop, i))->short_description)
	    
	    );
    send_to_char(buf, ch);
  }
  sprintf(buf, "\r\n"
	  "&cGA&c0) Add a new product.\r\n"
	  "&cGD&c0) Delete a product.\r\n"
	  "&cGQ&c0) Quit\r\n"
	  "Enter choice : ");
  send_to_char(buf, ch);

  OLC_MENU(ch) = SEDIT_PRODUCTS_MENU;
}

/*-------------------------------------------------------------------*/

void sedit_compact_rooms_menu(char_data *ch)
{
  struct shop_data *shop;
  int i, count = 0;

  shop = OLC_SHOP(ch);

#if defined(CLEAR_SCREEN)
  send_to_char("[H[J", ch);
#endif
  for (i = 0; S_ROOM(shop, i) != -1; i++) {
    sprintf(buf, "%2d - [&cC%5d&c0]  | %s", i, S_ROOM(shop, i), 
			!(++count % 5) ? "\r\n" : "");
    send_to_char(buf, ch);
  }
  sprintf(buf, "\r\n"
	  "&cGA&c0) Add a new room.\r\n"
	  "&cGD&c0) Delete a room.\r\n"
	  "&cGL&c0) Long display.\r\n"
	  "&cGQ&c0) Quit\r\n"
	  "Enter choice : ");
  send_to_char(buf, ch);

  OLC_MENU(ch) = SEDIT_ROOMS_MENU;
}

/*-------------------------------------------------------------------*/

void sedit_rooms_menu(char_data *ch)
{
  struct shop_data *shop;
  int i;

  shop = OLC_SHOP(ch);

#if defined(CLEAR_SCREEN)
  send_to_char("[H[J", ch);
#endif
  send_to_char("##     VNUM     Room\r\n\r\n", ch);
  for (i = 0; S_ROOM(shop, i) != -1; i++) {
    sprintf(buf, "%2d - [&cC%5d&c0] - &cY%s&c0\r\n", i, S_ROOM(shop, i), 
	    world[S_ROOM(shop, i)]->name);
    send_to_char(buf, ch);
  }
  sprintf(buf, "\r\n"
	  "&cGA&c0) Add a new room.\r\n"
	  "&cGD&c0) Delete a room.\r\n"
	  "&cGC&c0) Compact Display.\r\n"
	  "&cGQ&c0) Quit\r\n"
	  "Enter choice : ");
  send_to_char(buf, ch);

  OLC_MENU(ch) = SEDIT_ROOMS_MENU;
}

/*-------------------------------------------------------------------*/

void sedit_namelist_menu(char_data *ch)
{
  struct shop_data *shop;
  int i;

  shop = OLC_SHOP(ch);

#if defined(CLEAR_SCREEN)
  send_to_char("[H[J", ch);
#endif
  send_to_char("##              Type   Namelist\r\n\r\n", ch);
  for (i = 0; S_BUYTYPE(shop, i) != -1; i++) {
    sprintf(buf, "%2d - &cC%15s&c0 - &cY%s&c0\r\n", i, 
		item_types[S_BUYTYPE(shop, i)], 
		S_BUYWORD(shop, i) ? S_BUYWORD(shop, i) : "<None>");
    send_to_char(buf, ch);
  }
  sprintf(buf, "\r\n"
	  "&cGA&c0) Add a new entry.\r\n"
	  "&cGD&c0) Delete an entry.\r\n"
	  "&cGQ&c0) Quit\r\n"
	  "Enter choice : ");
  send_to_char(buf, ch);
  OLC_MENU(ch) = SEDIT_NAMELIST_MENU;
}

/*-------------------------------------------------------------------*/

void sedit_shop_flags_menu(char_data *ch)
{
  int i, count = 0;

#if defined(CLEAR_SCREEN)
  send_to_char("[H[J", ch);
#endif
#ifdef GOT_RID_OF_IT
  for (i = 0; i < NUM_SHOP_FLAGS; i++) {
    sprintf(buf, "&cG%2d&c0) %-20.20s   %s", i + 1, shop_bits[i],
		!(++count % 2) ? "\r\n" : "");
    send_to_char(buf, ch);
  }
#endif
  oasis_gen_edit(ch, "", SEDIT_SHOP_FLAGS, "");
  OLC_MENU(ch) = SEDIT_SHOP_FLAGS;
}

/*-------------------------------------------------------------------*/

void sedit_no_trade_menu(char_data *ch)
{
  int i, count = 0;

#if defined(CLEAR_SCREEN)
  send_to_char("[H[J", ch);
#endif
#ifdef GOT_RID_OF_IT
  for (i = 0; i < NUM_TRADERS; i++) {
    sprintf(buf, "&cG%2d&c0) %-20.20s   %s", i + 1, trade_letters[i],
		!(++count % 2) ? "\r\n" : "");
    send_to_char(buf, ch);
  }
#endif
  oasis_gen_edit(ch, "", SEDIT_NOTRADE, "");
  OLC_MENU(ch) = SEDIT_NOTRADE;
}

/*-------------------------------------------------------------------*/

void sedit_types_menu(char_data *ch)
{
  struct shop_data *shop;
  int i, count = 0;

  shop = OLC_SHOP(ch);

#if defined(CLEAR_SCREEN)
  send_to_char("[H[J", ch);
#endif
  for (i = 0; i < NUM_ITEM_TYPES; i++) {
    sprintf(buf, "&cG%2d&c0) &cC%-20s&c0  %s", i, item_types[i],
		!(++count % 3) ? "\r\n" : "");
    send_to_char(buf, ch);
  }
  sprintf(buf, "&c0Enter choice : ");
  send_to_char(buf, ch);
  OLC_MENU(ch) = SEDIT_TYPE_MENU;
}

/*-------------------------------------------------------------------*/

/*
 * Display main menu.
 */
void sedit_disp_menu(char_data *ch)
{
  struct shop_data *shop;

  shop = OLC_SHOP(ch);

  sprintbit(S_NOTRADE(shop), trade_letters, buf1);
  sprintbit(S_BITVECTOR(shop), shop_bits, buf2);
  sprintf(buf,
#if defined(CLEAR_SCREEN)
	  "[H[J"
#endif
	  "-- Shop Number : [&cC%d&c0]\r\n"
	  "&cG0&c0) Keeper      : [&cC%d&c0] &cY%s\r\n"
          "&cG1&c0) Open 1      : &cC%4d&c0          &cG2&c0) Close 1     : &cC%4d\r\n"
          "&cG3&c0) Open 2      : &cC%4d&c0          &cG4&c0) Close 2     : &cC%4d\r\n"
	  "&cG5&c0) Sell rate   : &cC%1.2f&c0          &cG6&c0) Buy rate    : &cC%1.2f\r\n"
	  "&cG7&c0) Keeper no item : &cY%s\r\n"
	  "&cG8&c0) Player no item : &cY%s\r\n"
	  "&cG9&c0) Keeper no cash : &cY%s\r\n"
	  "&cGA&c0) Player no cash : &cY%s\r\n"
	  "&cGB&c0) Keeper no buy  : &cY%s\r\n"
	  "&cGC&c0) Buy success    : &cY%s\r\n"
	  "&cGD&c0) Sell success   : &cY%s\r\n"
	  "&cGE&c0) Skill unknown  : &cY%s\r\n"
	  "&cGF&c0) Prac w/o cash  : &cY%s\r\n"
	  "&cGG&c0) No Trade With  : &cY%s\r\n"
	  "&cGH&c0) Shop flags     : &cY%s\r\n"
	  "&cGR&c0) Rooms Menu\r\n"
	  "&cGP&c0) Products Menu\r\n"
	  "&cGT&c0) Accept Types Menu\r\n"
	  "&cGQ&c0) Quit\r\n"
	  "Enter Choice : ",

	  OLC_NUM(ch),
	  S_KEEPER(shop) == -1 ?
	  -1 : S_KEEPER(shop), 
	  S_KEEPER(shop) == -1 ?
	  "None" : ss_data(PROTO_MOB(S_KEEPER(shop))->player.short_descr),
	  S_OPEN1(shop), 
	  S_CLOSE1(shop),
	  S_OPEN2(shop), 
	  S_CLOSE2(shop),
	  S_BUYPROFIT(shop),
	  S_SELLPROFIT(shop),
	  S_NOITEM1(shop),
	  S_NOITEM2(shop),
	  S_NOCASH1(shop),
	  S_NOCASH2(shop),
	  S_NOBUY(shop),
	  S_BUY(shop),
	  S_SELL(shop),
	  S_PRAC_NOT_KNOWN(shop),
	  S_PRAC_MISSING_CASH(shop),
	  buf1,
	  buf2);
  send_to_char(buf, ch);

  OLC_MENU(ch) = SEDIT_MAIN_MENU;
}

/**************************************************************************
  The GARGANTUAN event handler
 **************************************************************************/

void sedit_parse(char_data *ch, char *arg)
{
  int i;

  if (OLC_MENU(ch) > SEDIT_NUMERICAL_RESPONSE) {
    if (!isdigit(arg[0]) && ((*arg == '-') && (!isdigit(arg[1])))) {
      send_to_char("Field must be numerical, try again : ", ch);
      return;
    }
  }
  switch (OLC_MENU(ch)) {
/*-------------------------------------------------------------------*/
  case SEDIT_CONFIRM_SAVESTRING:
    switch (*arg) {
    case 'y':
    case 'Y':
      send_to_char("Saving shop to memory.\r\n", ch);
      sedit_save_internally(ch);
      sprintf(buf, "OLC: %s edits shop %d", GET_NAME(ch),
	      OLC_NUM(ch));
      mudlog(buf, CMP, MAX(TRUST_CREATOR, GET_INVIS_LEV(ch)), TRUE);
      cleanup_olc(ch, CLEANUP_STRUCTS);
      return;
    case 'n':
    case 'N':
      cleanup_olc(ch, CLEANUP_ALL);
      return;
    default:
      send_to_char("Invalid choice!\r\nDo you wish to save the shop? : ", ch);
      return;
    }
    break;

/*-------------------------------------------------------------------*/
  case SEDIT_MAIN_MENU:
    i = 0;
    switch (*arg) {
    case 'q':
    case 'Q':
      if (OLC_SAVE(ch)) {		/* Anything been changed? */
	send_to_char("Do you wish to save the changes to the shop? (y/n) : ", ch);
	OLC_MENU(ch) = SEDIT_CONFIRM_SAVESTRING;
      } else
	cleanup_olc(ch, CLEANUP_ALL);
      return;
    case '0':
      OLC_MENU(ch) = SEDIT_KEEPER;
      send_to_char("Enter virtual number of shop keeper : ", ch);
      return;
    case '1':
      OLC_MENU(ch) = SEDIT_OPEN1;
      i++;
      break;
    case '2':
      OLC_MENU(ch) = SEDIT_CLOSE1;
      i++;
      break;
    case '3':
      OLC_MENU(ch) = SEDIT_OPEN2;
      i++;
      break;
    case '4':
      OLC_MENU(ch) = SEDIT_CLOSE2;
      i++;
      break;
    case '5':
      OLC_MENU(ch) = SEDIT_BUY_PROFIT;
      i++;
      break;
    case '6':
      OLC_MENU(ch) = SEDIT_SELL_PROFIT;
      i++;
      break;
    case '7':
      OLC_MENU(ch) = SEDIT_NOITEM1;
      i--;
      break;
    case '8':
      OLC_MENU(ch) = SEDIT_NOITEM2;
      i--;
      break;
    case '9':
      OLC_MENU(ch) = SEDIT_NOCASH1;
      i--;
      break;
    case 'a':
    case 'A':
      OLC_MENU(ch) = SEDIT_NOCASH2;
      i--;
      break;
    case 'b':
    case 'B':
      OLC_MENU(ch) = SEDIT_NOBUY;
      i--;
      break;
    case 'c':
    case 'C':
      OLC_MENU(ch) = SEDIT_BUY;
      i--;
      break;
    case 'd':
    case 'D':
      OLC_MENU(ch) = SEDIT_SELL;
      i--;
      break;
    case 'e':
    case 'E':
      OLC_MENU(ch) = SEDIT_PRAC_NOT_KNOWN;
      return;
    case 'f':
    case 'F':
      OLC_MENU(ch) = SEDIT_PRAC_MISSING_CASH;
      return;
    case 'g':
    case 'G':
      sedit_no_trade_menu(ch);
      return;
    case 'h':
    case 'H':
      sedit_shop_flags_menu(ch);
      return;
    case 'r':
    case 'R':
      sedit_rooms_menu(ch);
      return;
    case 'p':
    case 'P':
      sedit_products_menu(ch);
      return;
    case 't':
    case 'T':
      sedit_namelist_menu(ch);
      return;
    default:
      sedit_disp_menu(ch);
      return;
    }

    if (i != 0) {
      send_to_char(i == 1 ? "\r\nEnter new value : " : (i == -1 ?
		"\r\nEnter new text :\r\n] " : "Oops...\r\n"), ch);
      return;
    }
    break;
/*-------------------------------------------------------------------*/
  case SEDIT_NAMELIST_MENU:
    switch (*arg) {
    case 'a':
    case 'A':
      sedit_types_menu(ch);
      return;
    case 'd':
    case 'D':
      send_to_char("\r\nDelete which entry? : ", ch);
      OLC_MENU(ch) = SEDIT_DELETE_TYPE;
      return;
    case 'q':
    case 'Q':
      break;
    }
    break;
/*-------------------------------------------------------------------*/
  case SEDIT_PRODUCTS_MENU:
    switch (*arg) {
    case 'a':
    case 'A':
      send_to_char("\r\nEnter new product virtual number : ", ch);
      OLC_MENU(ch) = SEDIT_NEW_PRODUCT;
      return;
    case 'd':
    case 'D':
      send_to_char("\r\nDelete which product? : ", ch);
      OLC_MENU(ch) = SEDIT_DELETE_PRODUCT;
      return;
    case 'q':
    case 'Q':
      break;
    }
    break;
/*-------------------------------------------------------------------*/
  case SEDIT_ROOMS_MENU:
    switch (*arg) {
    case 'a':
    case 'A':
      send_to_char("\r\nEnter new room virtual number : ", ch);
      OLC_MENU(ch) = SEDIT_NEW_ROOM;
      return;
    case 'c':
    case 'C':
      sedit_compact_rooms_menu(ch);
      return;
    case 'l':
    case 'L':
      sedit_rooms_menu(ch);
      return;
    case 'd':
    case 'D':
      send_to_char("\r\nDelete which room? : ", ch);
      OLC_MENU(ch) = SEDIT_DELETE_ROOM;
      return;
    case 'q':
    case 'Q':
      break;
    }
    break;
/*-------------------------------------------------------------------*/
    /*
     * String edits.
     */
  case SEDIT_NOITEM1:
    sedit_modify_string(&S_NOITEM1(OLC_SHOP(ch)), arg);
    break;
  case SEDIT_NOITEM2:
    sedit_modify_string(&S_NOITEM2(OLC_SHOP(ch)), arg);
    break;
  case SEDIT_NOCASH1:
    sedit_modify_string(&S_NOCASH1(OLC_SHOP(ch)), arg);
    break;
  case SEDIT_NOCASH2:
    sedit_modify_string(&S_NOCASH2(OLC_SHOP(ch)), arg);
    break;
  case SEDIT_NOBUY:
    sedit_modify_string(&S_NOBUY(OLC_SHOP(ch)), arg);
    break;
  case SEDIT_BUY:
    sedit_modify_string(&S_BUY(OLC_SHOP(ch)), arg);
    break;
  case SEDIT_SELL:
    sedit_modify_string(&S_SELL(OLC_SHOP(ch)), arg);
    break;
  case SEDIT_PRAC_NOT_KNOWN:
    sedit_modify_string(&S_PRAC_NOT_KNOWN(OLC_SHOP(ch)), arg);
    break;
  case SEDIT_PRAC_MISSING_CASH:
    sedit_modify_string(&S_PRAC_MISSING_CASH(OLC_SHOP(ch)), arg);
    break;
  case SEDIT_NAMELIST:
    {
      struct shop_buy_data new_entry;

      BUY_TYPE(new_entry) = OLC_VAL(ch);
      BUY_WORD(new_entry) = (arg && *arg) ? str_dup(arg) : 0;
      sedit_add_to_type_list(&(S_NAMELISTS(OLC_SHOP(ch))), &new_entry);
    }
    sedit_namelist_menu(ch);
    return;

/*-------------------------------------------------------------------*/
    /*
     * Numerical responses.
     */
  case SEDIT_KEEPER:
    i = atoi(arg);
    if ((i = atoi(arg)) != -1)
      if (!(mob_index[i])) {
	send_to_char("That mobile does not exist, try again : ", ch);
	return;
      }
    S_KEEPER(OLC_SHOP(ch)) = i;
    if (i == -1)
      break;
    /*
     * Fiddle with special procs.
     */
    S_FUNC(OLC_SHOP(ch)) = mob_index[i]->func;
    mob_index[i]->func = shop_keeper;
    break;
  case SEDIT_OPEN1:
    S_OPEN1(OLC_SHOP(ch)) = MAX(0, MIN(28, atoi(arg)));
    break;
  case SEDIT_OPEN2:
    S_OPEN2(OLC_SHOP(ch)) = MAX(0, MIN(28, atoi(arg)));
    break;
  case SEDIT_CLOSE1:
    S_CLOSE1(OLC_SHOP(ch)) = MAX(0, MIN(28, atoi(arg)));
    break;
  case SEDIT_CLOSE2:
    S_CLOSE2(OLC_SHOP(ch)) = MAX(0, MIN(28, atoi(arg)));
    break;
  case SEDIT_BUY_PROFIT:
    sscanf(arg, "%f", &S_BUYPROFIT(OLC_SHOP(ch)));
    break;
  case SEDIT_SELL_PROFIT:
    sscanf(arg, "%f", &S_SELLPROFIT(OLC_SHOP(ch)));
    break;
  case SEDIT_TYPE_MENU:
    OLC_VAL(ch) = MAX(0, MIN(NUM_ITEM_TYPES - 1, atoi(arg)));
    send_to_char("Enter namelist (return for none) :-\r\n] ", ch);
    OLC_MENU(ch) = SEDIT_NAMELIST;
    return;
  case SEDIT_DELETE_TYPE:
    sedit_remove_from_type_list(&(S_NAMELISTS(OLC_SHOP(ch))), atoi(arg));
    sedit_namelist_menu(ch);
    return;
  case SEDIT_NEW_PRODUCT:
    if ((i = atoi(arg)) != -1)
      if (!(obj_index[i])) {
	send_to_char("That object does not exist, try again : ", ch);
	return;
      }
    if (i > 0)
      sedit_add_to_int_list(&(S_PRODUCTS(OLC_SHOP(ch))), i);
    sedit_products_menu(ch);
    return;
  case SEDIT_DELETE_PRODUCT:
    sedit_remove_from_int_list(&(S_PRODUCTS(OLC_SHOP(ch))), atoi(arg));
    sedit_products_menu(ch);
    return;
  case SEDIT_NEW_ROOM:
    if ((i = atoi(arg)) != -1)
      if ((i = real_room(i)) < 0) {
	send_to_char("That room does not exist, try again : ", ch);
	return;
      }
    if (i >= 0)
      sedit_add_to_int_list(&(S_ROOMS(OLC_SHOP(ch))), atoi(arg));
    sedit_rooms_menu(ch);
    return;
  case SEDIT_DELETE_ROOM:
    sedit_remove_from_int_list(&(S_ROOMS(OLC_SHOP(ch))), atoi(arg));
    sedit_rooms_menu(ch);
    return;
  case SEDIT_SHOP_FLAGS:
    switch (*arg && oasis_gen_edit(ch, "", SEDIT_SHOP_FLAGS, arg)) {
      case 0: sedit_disp_menu(ch); return;
      default: 
        if (!OLC_SAVE(ch))
	  OLC_SAVE(ch) = OLC_SAVE_YES;
      sedit_shop_flags_menu(ch);
      return;
    }
  case SEDIT_NOTRADE:
    switch (*arg && oasis_gen_edit(ch, "", SEDIT_NOTRADE, arg)) {
      case 0: sedit_disp_menu(ch); return;
      default: 
        if (!OLC_SAVE(ch))
	  OLC_SAVE(ch) = OLC_SAVE_YES;
      sedit_no_trade_menu(ch);
      return;
    }

/*-------------------------------------------------------------------*/
  default:
    /*
     * We should never get here.
     */
    cleanup_olc(ch, CLEANUP_ALL);
    mudlog("SYSERR: OLC: sedit_parse(): Reached default case!", BRF, TRUST_CREATOR, TRUE);
    send_to_char("Oops...\r\n", ch);
    break;
  }

/*-------------------------------------------------------------------*/

/*
 * END OF CASE 
 * If we get here, we have probably changed something, and now want to
 * return to main menu.  Use OLC_VAL as a 'has changed' flag.
 */
  OLC_SAVE(ch) = OLC_SAVE_YES;
  sedit_disp_menu(ch);
}


/* prints one room to file */
void fprint_shop(FILE *stream, shop_data *shop)
{
  int i, j;
  const char *name;
  sh_int keeper_num = 1;
  
  fprintf(stream, "#%d~\n", shop->virt);

  /*
   * Save the products.
   */
  for (j = 0; S_PRODUCT(shop, j) != -1; j++)
    if (obj_index[S_PRODUCT(shop, j)])
      fprintf(stream, "%d\n", S_PRODUCT(shop, j));

  /*
   * Save the rates.
   */
  fprintf(stream, "-1\n%1.2f\n%1.2f\n", S_BUYPROFIT(shop), S_SELLPROFIT(shop));

  /*
   * Save the buy types and namelists.
   */
  j = -1;
  do {
    j++;
    fprintf(stream, "%d%s\n", S_BUYTYPE(shop, j),
	    S_BUYWORD(shop, j) ? S_BUYWORD(shop, j) : "");
  } while (S_BUYTYPE(shop, j) != -1);

  // Added just in case the shopkeeper has been deleted. -DH
  
  if (mob_index[S_KEEPER(shop)])
    keeper_num = GET_MOB_NUM((char_data *) mob_index[S_KEEPER(shop)]->proto);

  /*
   * Save messages'n'stuff.
   * Added some small'n'silly defaults as sanity checks.
   */
  fprintf(stream,
	  "%s~\n%s~\n%s~\n%s~\n%s~\n%s~\n%s~\n%s~\n%s~\n"
	  "%d\n%d\n%d\n%d\n",
	  S_NOITEM1(shop) ? S_NOITEM1(shop) : "%s Ke?!",
	  S_NOITEM2(shop) ? S_NOITEM2(shop) : "%s Ke?!",
	  S_NOBUY(shop) ? S_NOBUY(shop) : "%s Ke?!",
	  S_NOCASH1(shop) ? S_NOCASH1(shop) : "%s Ke?!",
	  S_NOCASH2(shop) ? S_NOCASH2(shop) : "%s Ke?!",
	  S_BUY(shop) ? S_BUY(shop) : "%s Ke?! %d?",
	  S_SELL(shop) ? S_SELL(shop) : "%s Ke?! %d?",
	  S_PRAC_NOT_KNOWN(shop) ? S_PRAC_NOT_KNOWN(shop) : "%s Ke?! %d?",
	  S_PRAC_MISSING_CASH(shop) ? S_PRAC_MISSING_CASH(shop) : "%s Ke?! %d?",
	  S_BROKE_TEMPER(shop),
	  S_BITVECTOR(shop),
	  keeper_num,
	  S_NOTRADE(shop)
	  );

  /*
   * Save the rooms.
   */
  j = -1;
  do {
    j++;
    fprintf(stream, "%d\n", S_ROOM(shop, j));
  } while (S_ROOM(shop, j) != -1);

  /*
   * Save open/closing times 
   */
  fprintf(stream, "%d\n%d\n%d\n%d\n", S_OPEN1(shop), S_CLOSE1(shop),
	    S_OPEN2(shop), S_CLOSE2(shop));
  
}

