/* ************************************************************************
*   File: structs.h                                     Part of CircleMUD *
*  Usage: header file for central structures and constants                *
*                                                                         *
*  All rights reserved.  See license.doc for complete information.        *
*                                                                         *
*  Copyright (C) 1993, 94 by the Trustees of the Johns Hopkins University *
*  CircleMUD is based on DikuMUD, Copyright (C) 1990, 1991.               *
*                                                                         *
*  $Author: tlo $
*  $Date: 1999/03/06 18:27:09 $
*  $Revision: 6.19 $
************************************************************************ */

#define RANGED_WEAPONS
#define OASIS_OLC

#define FIRST_SKILLO			1
#define TOP_SKILLO_DEFINE		700

#define JD_MATTACH

#include "find.h"

extern int pulsevio, pulsemob, pulsezon;
extern int base_level_exp;

extern void command_interpreter(struct char_data *ch, char *argument, int cmd = 0);
extern void obj_command_interpreter(obj_data * obj, char *argument, int cmd = 0);
extern void wld_command_interpreter(struct room_data *room, char *argument, int cmd = 0);
extern void act(char *str, int hide_invisible, char_data *ch, obj_data *obj, void *vict_obj, int type);

/* preamble *************************************************************/

#define NOWHERE    -1    /* nil reference for room-database	*/
#define NOTHING	   -1    /* nil reference for objects or mobiles */
#define NOBODY	   -1    /* nil reference for mobiles		*/

#define SPECIAL(name) \
   int (name)(char_data *ch, void *me, int cmd, char *argument, char *farg)

/* room-related defines *************************************************/


/* The cardinal directions: used as index to room_data.dir_option[] */
#define NORTH          0
#define EAST           1
#define SOUTH          2
#define WEST           3
#define UP             4
#define DOWN           5

#ifndef TO_ROOM
#define TO_ROOM		1
#define TO_VICT		2
#define TO_NOTVICT	3
#define TO_CHAR		4
#define TO_NOTVICTROOM	5
#define TO_SLEEP	128	/* to char, even if sleeping */
#define NO_TRIG		256	/* to char, even if sleeping */
#endif

#ifndef NULL
#define NULL (void *)0
#endif

#if !defined(FALSE)
#define FALSE 0
#endif

#if !defined(TRUE)
#define TRUE  (!FALSE)
#endif

#define MAX_ROOM_LENGTH 10000
#define MAX_ROOM_WIDTH  10000
#define MAX_ROOM_HEIGHT 100

#define DEFAULT_ROOM_LENGTH 4
#define DEFAULT_ROOM_WIDTH  4
#define DEFAULT_ROOM_HEIGHT 3


enum datatypes {
  DATATYPE_UNDEFINED,
  DATATYPE_SBYTE,
  DATATYPE_UBYTE,
  DATATYPE_SH_INT,
  DATATYPE_USH_INT,
  DATATYPE_INT,
  DATATYPE_UINT,
  DATATYPE_LONG,
  DATATYPE_ULONG,
  DATATYPE_CSTRING,  
  DATATYPE_ALL_DATA,
  DATATYPE_GAME_DATA,
  DATATYPE_THING,
  DATATYPE_TRIG_DATA,
  DATATYPE_OBJ_DATA,
  DATATYPE_ROOM_DATA,
  DATATYPE_CHAR_DATA
};


/* Room flags: used in room_data.room_flags */
/* WARNING: In the world files, NEVER set the bits marked "R" ("Reserved") */
#define ROOM_DARK		(1 << 0)   /* Dark			*/
#define ROOM_DEATH		(1 << 1)   /* Death trap		*/
#define ROOM_NOMOB		(1 << 2)   /* MOBs not allowed		*/
#define ROOM_INDOORS		(1 << 3)   /* Indoors			*/
#define ROOM_PEACEFUL		(1 << 4)   /* Violence not allowed	*/
#define ROOM_SOUNDPROOF		(1 << 5)   /* Shouts, gossip blocked	*/
#define ROOM_NOTRACK		(1 << 6)   /* Track won't go through	*/
#define ROOM_NOMAGIC		(1 << 7)   /* Magic not allowed		*/
#define ROOM_TUNNEL		(1 << 8)   /* room for only 1 pers	*/
#define ROOM_PRIVATE		(1 << 9)   /* Can't teleport in		*/
#define ROOM_GODROOM		(1 << 10)  /* LVL_GOD+ only allowed	*/
#define ROOM_HOUSE		(1 << 11)  /* (R) Room is a house	*/
#define ROOM_HOUSE_CRASH	(1 << 12)  /* (R) House needs saving	*/
#define ROOM_ATRIUM		(1 << 13)  /* (R) The door to a house	*/
#define ROOM_OLC		(1 << 14)  /* (R) Modifyable/!compress	*/
#define ROOM_BFS_MARK		(1 << 15)  /* (R) breath-first srch mrk	*/
#define ROOM_MAP		(1 << 16)  /* Set to make a room a map room */

/* Room affections */
#define RAFF_FOG		(1 << 0)   /* The Wall of Fog Spell	*/

/* Exit info: used in room_data.dir_option.exit_info */
#define EX_ISDOOR		(1 << 0)   /* Exit is a door		*/
#define EX_PICKPROOF		(1 << 1)   /* Lock can't be picked	*/
#define EX_CLOSED		(1 << 2)   /* The door is closed	*/
#define EX_LOCKED		(1 << 3)   /* The door is locked	*/
#define EX_SECRET               (1 << 4)   /* The door is secret        */
#define EX_VISTA                (1 << 5)   /* Exit is actually a window */
#define EX_SEETHRU              (1 << 6)   /* Can look thru if closed   */


/* Sector types: used in room_data.sector_type */
#define SECT_INSIDE          0		   /* Indoors			*/
#define SECT_CITY            1		   /* In a city			*/
#define SECT_FIELD           2		   /* In a field		*/
#define SECT_FOREST          3		   /* In a forest		*/
#define SECT_HILLS           4		   /* In the hills		*/
#define SECT_MOUNTAIN        5		   /* On a mountain		*/
#define SECT_WATER_SWIM      6		   /* Swimmable water		*/
#define SECT_WATER_NOSWIM    7		   /* Water - need a boat	*/
#define SECT_UNDERWATER	     8		   /* Underwater		*/
#define SECT_FLYING	     9		   /* Wheee!			*/
#define SECT_ROAD	     10		   /* For map rooms only	*/
#define SECT_DESERT	     11		   /* 	*/
#define SECT_JUNGLE	     12		   /* 	*/
#define SECT_SWAMP	     13		   /* 	*/
#define SECT_ARCTIC	     14		   /* 	*/
#define SECT_HIGH_MOUNTAIN   15		   /* 	*/
#define SECT_UNDERGROUND     16		   /* 	*/
#define SECT_VACUUM          17
#define NUM_SECTOR_TYPES     18 

/* char and mob-related defines *****************************************/

/* PC races */
#define RACE_ANIMAL       0
/* Base PC Races */
#define RACE_HYUMANN		1
#define RACE_ELF          2
#define RACE_DWOMAX		3
#define RACE_SHASTICK		4
#define RACE_MURRASH		5
#define RACE_LAORI		6
#define RACE_THADYRI		7
#define RACE_TERRAN		8
#define RACE_TRULOR		9
/* Advanced PC Races */
#define RACE_DRAGON		10
#define RACE_DEVA		11
#define RACE_SOLAR		12

#define NUM_RACES         13  /* This, of course, must be the # of races */

// Used for racial advantages
extern const long race_advantages[NUM_RACES];
extern const long race_disadvantages[NUM_RACES];


/* PC classes */
#define CLASS_WRONGRACE   -2
#define CLASS_UNDEFINED	  -1
#define CLASS_MAGICIAN    0
#define CLASS_CLERIC	  1
#define CLASS_THIEF       2
#define CLASS_WARRIOR     3
#define CLASS_RANGER	  4
#define CLASS_ASSASSIN	  5
#define CLASS_ALCHEMIST	  6
#define CLASS_VAMPIRE	  7
#define CLASS_PSIONICIST  8
#define CLASS_MOB         9
#define CLASS_UNDEAD      10
#define CLASS_INSECT	  11
#define CLASS_MAMMAL      12
#define CLASS_AVIAN       13
#define CLASS_FISH	  14
#define CLASS_REPTILE     15

#define NUM_CLASSES	  16  /* This must be the number of classes!! */


#define WEAR_MSG_SLIDEONTO	0
#define WEAR_MSG_WEARAROUND	1
#define WEAR_MSG_WEARABOUT	2
#define WEAR_MSG_WEARON		3
#define WEAR_MSG_WEAROVER	4
#define WEAR_MSG_WEARFROM	5
#define WEAR_MSG_PUTON		6
#define WEAR_MSG_ENGULFED	7
#define WEAR_MSG_HELDIN		8
#define WEAR_MSG_LIGHT		9
#define WEAR_MSG_ATTACH		10
#define MAX_WEAR_MSG		11
#define WEAR_MSG_WIELD		11
#define WEAR_MSG_STRAPTO	12
#define MAX_HANDWEAR_MSG	13

#define WEAR_ANYTHING_ELSE		0
#define WEARING_SOMETHING_AROUND	1
#define WEARING_SOMETHING_OVER		2
#define WEARING_SOMETHING_ON		3
#define WEARING_SOMETHING_ONBOTH	4
#define WEARING_SOMETHING_ONALL		5
#define WEARING_SOMETHING_ABOUT		6
#define HAVE_SOMETHING_ON		7
#define HAVE_SOMETHING_AROUND		8
#define HOLDING_SOMETHING_IN		9
#define HOLDING_SOMETHING_INBOTH	10
#define MAX_ALREADY_WEARING		11

#define BODY_TYPE_HUMANOID	0
#define BODY_TYPE_INSECTOID	1
#define BODY_TYPE_LIZARD	2
#define BODY_TYPE_DRAGON	3
#define BODY_TYPE_AVIAN		4
#define BODY_TYPE_AMORPHOUS	5
#define BODY_TYPE_QUADHOOVED	6
#define BODY_TYPE_QUADPAW	7
#define BODY_TYPE_SHASTICK	8
#define BODY_TYPE_TERRAN_ARMOR	9
#define BODY_TYPE_TRULOR_ARMOR	10
#define BODY_TYPE_BLANK		11
#define MAX_BODY_TYPE		12

// Damage types, as per GURPS - DH
#define DAMTYPE_BASIC		0
#define DAMTYPE_SLASHING	1
#define DAMTYPE_PIERCING	2
#define DAMTYPE_FIRE		3
#define DAMTYPE_ELECTRICITY	4
#define DAMTYPE_ACID		5
#define DAMTYPE_FREEZING	6
#define DAMTYPE_POISON		7
#define DAMTYPE_PSIONIC		8
#define DAMTYPE_VIM		9
#define MAX_DAMTYPE		10

// Material types, imported from NetHack 3.1.3 - thanks NH dev team! :)  -DH
#define MATERIAL_TYPE_UNDEFINED		0	
#define MATERIAL_TYPE_LIQUID		1	/* currently only for venom */
#define MATERIAL_TYPE_WAX		2
#define MATERIAL_TYPE_VEGGY		3	/* foodstuffs */
#define MATERIAL_TYPE_FLESH		4	/*   ditto    */
#define MATERIAL_TYPE_PAPER		5
#define MATERIAL_TYPE_CLOTH		6
#define MATERIAL_TYPE_LEATHER		7
#define MATERIAL_TYPE_WOOD		8
#define MATERIAL_TYPE_BONE		9
#define MATERIAL_TYPE_DRAGON_HIDE	10	/* not leather! */
#define MATERIAL_TYPE_IRON		11	/* Fe - includes steel */
#define MATERIAL_TYPE_METAL		12	/* Sn, &c. */
#define MATERIAL_TYPE_COPPER		13	/* Cu - includes brass */
#define MATERIAL_TYPE_SILVER		14	/* Ag */
#define MATERIAL_TYPE_GOLD		15	/* Au */
#define MATERIAL_TYPE_PLATINUM		16	/* Pt */
#define MATERIAL_TYPE_MITHRIL		17
#define MATERIAL_TYPE_TITANIUM		18
#define MATERIAL_TYPE_PLASTIC		19
#define MATERIAL_TYPE_GLASS		20
#define MATERIAL_TYPE_GEMSTONE		21
#define MATERIAL_TYPE_MINERAL		22
#define MATERIAL_TYPE_SHADOW		23
#define MATERIAL_TYPE_ETHER		24
#define MATERIAL_TYPE_STEEL		25
#define MATERIAL_TYPE_ADAMANT		26
#define MATERIAL_TYPE_DIAMOND		27
#define MATERIAL_TYPE_PORCELAIN         28
#define MATERIAL_TYPE_CERAMIC		29
#define MATERIAL_TYPE_JELLY		30
#define MATERIAL_TYPE_VAPOR		31
#define MATERIAL_TYPE_DARK_STEEL	32


/* Sex */
#define SEX_NEUTRAL   0
#define SEX_MALE      1
#define SEX_FEMALE    2


/* Positions */
#define POS_DEAD       0	/* dead			*/
#define POS_MORTALLYW  1	/* mortally wounded	*/
#define POS_INCAP      2	/* incapacitated	*/
#define POS_STUNNED    3	/* stunned		*/
#define POS_SLEEPING   4	/* sleeping		*/
#define POS_RESTING    5	/* resting		*/
#define POS_SITTING    6	/* sitting		*/
#define POS_FIGHTING   7	/* fighting		*/
#define POS_STANDING   8	/* standing		*/
#define POS_RIDING     9      /* mounted - DO NOT USE	    */
#define POS_AUDIBLE    10	/* note silenced - DO NOT USE */

#define HIT_INCAP      -3     /* The hit level for incapacitation   */
#define HIT_MORTALLYW  -6     /* The hit level for mortally wound   */
#define HIT_DEAD       -11	/* The point you never want to get to */

/* mount check types */
#define MNT_MOUNT      0      /* Mount attempt  	 */
#define MNT_DISMOUNT   1      /* Dismount attempt	 */
#define MNT_COMBAT     2      /* combat check to stay on */
#define MNT_TOROOM     3	/* char in different room  */
#define MNT_MOVE       4	/* trying to move on mount */
#define MNT_EXTRACT    5	/* char or mount extracted */
#define MNT_FLEE       6	/* mount fled		   */


/* Player flags: used by char_data.char_specials.act */
#define PLR_KILLER	(1 << 0)   /* Player is a player-killer		*/
#define PLR_THIEF	(1 << 1)   /* Player is a player-thief		*/
#define PLR_FROZEN	(1 << 2)   /* Player is frozen			*/
#define PLR_DONTSET     (1 << 3)   /* Don't EVER set (ISNPC bit)	*/
#define PLR_WRITING	(1 << 4)   /* Player writing (board/mail/olc)	*/
#define PLR_MAILING	(1 << 5)   /* Player is writing mail		*/
#define PLR_CRASH	(1 << 6)   /* Player needs to be crash-saved	*/
#define PLR_SITEOK	(1 << 7)   /* Player has been site-cleared	*/
#define PLR_NOSHOUT	(1 << 8)   /* Player not allowed to shout/goss	*/
#define PLR_UNJUST	(1 << 9)   /* Player has been unjust            */
#define PLR_DELETED	(1 << 10)  /* Player deleted - space reusable	*/
#define PLR_LOADROOM	(1 << 11)  /* Player uses nonstandard loadroom	*/
#define PLR_NOWIZLIST	(1 << 12)  /* Player shouldn't be on wizlist	*/
#define PLR_NODELETE	(1 << 13)  /* Player shouldn't be deleted	*/
#define PLR_INVSTART	(1 << 14)  /* Player should enter game wizinvis	*/
#define PLR_CRYO	(1 << 15)  /* Player is cryo-saved (purge prog)	*/
#define PLR_AFK         (1 << 16)  /* Player is away from keyboard      */
#define PLR_NOTITLE	(1 << 17)  /* Player not allowed to set title	*/
#define PLR_ENGAGED	(1 << 18)  /* Gonna get MARRIED!!!!!!		*/
#define PLR_MARRIED	(1 << 19)  /* Are NOW Man and Wife (hopefully!) */
#define PLR_IC		(1 << 20)  /* In character for roleplay         */
#define PLR_PK		(1 << 21)  /* A designated player killer        */
#define PLR_TELEPATHY   (1 << 22)  /* Player is telepathic  		*/
#define PLR_WATCHED	(1 << 23)  /* Player needs to be watched	*/


/* Mobile flags: used by char_data.char_specials.act */
#define MOB_SPEC         (1 << 0)  /* Mob has a callable spec-proc	*/
#define MOB_SENTINEL     (1 << 1)  /* Mob should not move		*/
#define MOB_SCAVENGER    (1 << 2)  /* Mob picks up stuff on the ground	*/
#define MOB_ISNPC        (1 << 3)  /* (R) Automatically set on all Mobs	*/
#define MOB_AWARE	 (1 << 4)  /* Mob can't be backstabbed		*/
#define MOB_AGGRESSIVE   (1 << 5)  /* Mob hits players in the room	*/
#define MOB_STAY_ZONE    (1 << 6)  /* Mob shouldn't wander out of zone	*/
#define MOB_WIMPY        (1 << 7)  /* Mob flees if severely injured	*/
#define MOB_AGGR_GOOD    (1 << 8)  /* auto attack good PC's		*/
#define MOB_AGGR_NEUTRAL (1 << 9)  /* auto attack neutral PC's		*/
#define MOB_AGGR_EVIL    (1 << 10) /* auto attack good PC's		*/
#define MOB_MEMORY	 (1 << 11) /* remember attackers if attacked	*/
#define MOB_HELPER	 (1 << 12) /* attack PCs fighting other NPCs	*/
#define MOB_NOCHARM	 (1 << 13) /* Mob can't be charmed		*/
#define MOB_NOSUMMON	 (1 << 14) /* Mob can't be summoned		*/
#define MOB_NOSLEEP	 (1 << 15) /* Mob can't be slept		*/
#define MOB_NOBASH	 (1 << 16) /* Mob can't be bashed (e.g. trees)	*/
#define MOB_NOBLIND	 (1 << 17) /* Mob can't be blinded		*/
#define MOB_NOSHOOT	 (1 << 18) /* Mob can't be shot at		*/
#define MOB_SEEKER	 (1 << 19) /* Mob hunts/seeks across zones	*/
#define MOB_HELPER_RACE	 (1 << 20) /* assists NPCs of same race         */
#define MOB_HELPER_ALIGN (1 << 21) /* assists NPCs of same race         */
#define MOB_HUNTER	 (1 << 22) /* Mob hunts hated characters	*/


/* Preference flags: used by char_data.player_specials.pref */
#define PRF_BRIEF       (1 << 0)  /* Room descs won't normally be shown	*/
#define PRF_COMPACT     (1 << 1)  /* No extra CRLF pair before prompts	*/
#define PRF_DEAF	(1 << 2)  /* Can't hear shouts			*/
#define PRF_NOTELL	(1 << 3)  /* Can't receive tells		*/
#define PRF_DISPHP	(1 << 4)  /* Display hit points in prompt	*/
#define PRF_DISPMANA	(1 << 5)  /* Display mana points in prompt	*/
#define PRF_DISPMOVE	(1 << 6)  /* Display move points in prompt	*/
#define PRF_AUTOEXIT	(1 << 7)  /* Display exits in a room		*/
#define PRF_NOHASSLE	(1 << 8)  /* Aggr mobs won't attack		*/
#define PRF_QUEST	(1 << 9)  /* On quest				*/
#define PRF_SUMMONABLE	(1 << 10) /* Can be summoned			*/
#define PRF_NOREPEAT	(1 << 11) /* No repetition of comm commands	*/
#define PRF_HOLYLIGHT	(1 << 12) /* Can see in dark			*/
#define PRF_COLOR_1	(1 << 13) /* Color (low bit)			*/
#define PRF_COLOR_2	(1 << 14) /* Color (high bit)			*/
#define PRF_NOWIZ	(1 << 15) /* Can't hear wizline			*/
#define PRF_LOG1	(1 << 16) /* On-line System Log (low bit)	*/
#define PRF_LOG2	(1 << 17) /* On-line System Log (high bit)	*/
#define PRF_NOAUCT	(1 << 18) /* Can't hear auction channel		*/
#define PRF_NOGOSS	(1 << 19) /* Can't hear gossip channel		*/
#define PRF_NOGRATZ	(1 << 20) /* Can't hear grats channel		*/
#define PRF_ROOMFLAGS	(1 << 21) /* Can see room flags (ROOM_x)	*/
#define PRF_INVULN      (1 << 22) /* Invulnerable to normal attacks     */
#define PRF_AUTOLOOT	(1 << 23) /* Auto Loot a Corpse of Eq, etc..	*/
#define PRF_AUTOSPLIT	(1 << 24) /* Auto Split Gold of a Corpse....	*/
#define PRF_BUILDING	(1 << 25) /* I am BUILDING Leave me ALONE!	*/
#define PRF_DISPTANK	(1 << 26) /* I want to see the Percentages!	*/
#define PRF_DISPATTK	(1 << 27) /* I also want the above tooo!!!!	*/
#define PRF_WWWADD	(1 << 28) /* I want my HomePage SHOWN on HTML   */
#define PRF_CLANTALK    (1 << 29) /* Can hear clan talk                 */
#define PRF_AUTOASSIST	(1 << 30) /* auto assist thy friends		*/
#define PRF_SOUND	(1 << 31) /* MUD Sounds enabled                 */

/* Affect bits: used in char_data.char_specials.saved.affected_by */
/* WARNING: In the world files, NEVER set the bits marked "R" ("Reserved") */
#define AFF_BLIND             (1 << 0)	   /* (R) Char is blind		*/
#define AFF_INVISIBLE         (1 << 1)	   /* Char is invisible		*/
#define AFF_CURSE	      (1 << 2)	   /* Char is cursed		*/
#define AFF_DETECT_INVIS      (1 << 3)	   /* Char can see invis chars  */
#define AFF_DETECT_MAGIC      (1 << 4)	   /* Char is sensitive to magic*/
#define AFF_SENSE_LIFE        (1 << 5)	   /* Char can sense hidden life*/
#define AFF_FLY               (1 << 6)	   /* Char is flying	        */
#define AFF_SANCTUARY	      (1 << 7)	   /* Char protected by sanct.	*/
#define AFF_GROUP             (1 << 8)	   /* (R) Char is grouped	*/
#define AFF_MAGE_SHIELD       (1 << 9)	   /* Char prot by en armor sp. */
#define AFF_INFRAVISION       (1 << 10)	   /* Char can see in dark	*/
#define AFF_POISON            (1 << 11)	   /* (R) Char is poisoned	*/
#define AFF_PROTECT_EVIL      (1 << 12)	   /* Char protected from evil  */
#define AFF_DETECT_ALIGN      (1 << 13)	   /* Char sees alignments	*/
#define AFF_SLEEP             (1 << 14)	   /* (R) Char magically asleep	*/
#define AFF_NOTRACK	      (1 << 15)	   /* Char can't be tracked	*/
#define AFF_LIGHT   	      (1 << 16)	   /* Room is lighted by this  	*/
#define AFF_REGENERATION      (1 << 17)	   /* Character regens 2x hp	*/
#define AFF_SNEAK             (1 << 18)	   /* Char can move quietly	*/
#define AFF_HIDE              (1 << 19)	   /* Char is hidden		*/
#define AFF_BERSERK	      (1 << 20)	   /* Char is berserk   	*/
#define AFF_CHARM             (1 << 21)	   /* Char is charmed		*/
#define AFF_SILENCE           (1 << 22)    /* Char can not speak        */
#define AFF_HASTE	      (1 << 23)	   /* Char is hasted         	*/
#define AFF_BLADE_BARRIER     (1 << 24)	   /* Char is protected		*/
#define AFF_FIRE_SHIELD       (1 << 25)	   /* Char is protected		*/
#define AFF_VACUUM_OK         (1 << 26)	   /* Char can survive vacuum   */
#define AFF_PARALYZED	      (1 << 27)	   /* Room for future expansion	*/
#define AFF_CAMOFLAUGE	      (1 << 28)	   /* Room for future expansion	*/
#define AFF_VAMP_REGEN        (1 << 29)	   /* Vampiric Regeneration	*/

// Bits used for racial and intrinsic advantanges.
#define ADV_LUCKY		(1 << 0)
#define ADV_FAST_HP_REGEN	(1 << 1)
#define ADV_FAST_MANA_REGEN	(1 << 2)
#define ADV_FAST_MOVE_REGEN	(1 << 3)
#define ADV_SLOW_METABOLISM	(1 << 4)
#define ADV_INFRAVISION		(1 << 5)
#define ADV_FLIGHT		(1 << 6)
#define ADV_NATURE_RECHARGE	(1 << 7)
#define ADV_AMBIDEXTROUS	(1 << 8)
#define ADV_GROUP_MAGIC_BONUS	(1 << 9)
#define ADV_RUNE_AWARE		(1 << 10)
#define ADV_COMBAT_REFLEXES	(1 << 11)


// Bits used for racial and intrinsic disadvantanges.
#define DISADV_UNLUCKY		(1 << 0)
#define DISADV_SLOW_HP_REGEN	(1 << 1)
#define DISADV_SLOW_MANA_REGEN	(1 << 2)
#define DISADV_SLOW_MOVE_REGEN	(1 << 3)
#define DISADV_FAST_METABOLISM	(1 << 4)
#define DISADV_NO_MAGIC		(1 << 5)
#define DISADV_ATHEIST		(1 << 6)
#define DISADV_COLD_BLOODED	(1 << 7)
#define DISADV_NECRO_DESTROYS_FLESH	(1 << 8)

/* special affects that should not be set by objects/mob files */
#define SPEC_AFFS	      (AFF_CHARM | AFF_GROUP)


/* Immortal command flags:  Used by player_specials->saved.godcmds1 */
#define GCMD_GEN            (1 << 0)  /*  General god commands */
#define GCMD_ADVANCE        (1 << 1)
#define GCMD_AT             (1 << 2)
#define GCMD_BAN            (1 << 3)
#define GCMD_DC             (1 << 4)
#define GCMD_ECHO           (1 << 5)
#define GCMD_FORCE          (1 << 6)
#define GCMD_FREEZE         (1 << 7)
#define GCMD_HCONTROL       (1 << 8)
#define GCMD_LOAD           (1 << 9)
#define GCMD_MUTE           (1 << 10)
#define GCMD_UNUSED0        (1 << 11) /* unused */
#define GCMD_JUDGE          (1 << 12)
#define GCMD_PURGE          (1 << 13)
#define GCMD_RELOAD         (1 << 14)
#define GCMD_REROLL         (1 << 15)
#define GCMD_RESTORE        (1 << 16)
#define GCMD_SEND           (1 << 17)
#define GCMD_SET            (1 << 18)
#define GCMD_SHUTDOWN       (1 << 19)
#define GCMD_SKILLSET       (1 << 20)
#define GCMD_SLAY           (1 << 21)
#define GCMD_HOSTNAMES      (1 << 22)
#define GCMD_SNOOP          (1 << 23)
#define GCMD_SWITCH         (1 << 24)
#define GCMD_SYSTEM         (1 << 25)
#define GCMD_TRANS          (1 << 26)
#define GCMD_UNAFFECT       (1 << 27)
#define GCMD_WIZLOCK        (1 << 28)
#define GCMD_ZRESET         (1 << 29)
#define GCMD_SCRIPTS        (1 << 30)
#define GCMD_CMDSET         (1 << 31)  /* Don't change the pos. of this! */

/* Immortal command flags:  Used by player_specials->saved.godcmds2 */
#define GCMD2_OLC           (1 << 0)
#define GCMD2_INVIS         (1 << 1)
#define GCMD2_LORDGEN       (1 << 2)
#define GCMD2_INVULN        (1 << 3) /* Invulnerable to normal attacks     */
#define GCMD2_WORLD         (1 << 4)
#define GCMD2_SYSWHO        (1 << 5)
#define GCMD2_SYSPS         (1 << 6)
#define GCMD2_IMMPUN        (1 << 7)
#define GCMD2_TEDIT	    (1 << 8)
#define GCMD2_WIZNET	    (1 << 9)
#define GCMD2_TRIGEDIT      (1 << 10)
#define GCMD2_CLANGOD	    (1 << 11)


/* Values of name_services */
#define NS_HOSTNAMES        (1 << 0)	/* resolve hostnames            */
#define NS_IDENT            (1 << 1)    /* try to resolve users names   */

/* Modes of connectedness: used by descriptor_data.state */
#define CON_PLAYING		0  /* Playing - Nominal state */
#define CON_CLOSE		1  /* Disconnecting  */
#define CON_GET_NAME		2  /* By what name ..?  */
#define CON_NAME_CNFRM		3  /* Did I get that right, x? */
#define CON_PASSWORD		4  /* Password:   */
#define CON_NEWPASSWD		5  /* Give me a password for x */
#define CON_CNFPASSWD		6  /* Please retype password: */
#define CON_QSEX		7  /* Sex?    */
#define CON_QCLASS		8  /* Class?   */
#define CON_RMOTD		9  /* PRESS RETURN after MOTD */
#define CON_MENU		10  /* Your choice: (main menu) */
#define CON_EXDESC		11  /* Enter a new description: */
#define CON_CHPWD_GETOLD	12  /* Changing passwd: get old */
#define CON_CHPWD_GETNEW	13  /* Changing passwd: get new */
#define CON_CHPWD_VRFY		14  /* Verify new password  */
#define CON_DELCNF1		15  /* Delete confirmation 1 */
#define CON_DELCNF2		16  /* Delete confirmation 2 */
#define CON_QRACE		17             /* Race?    */
#define CON_NEWPLAYMESG		18             /* Reading newplayer message    */
#define CON_IDCONING		19             /* waiting for ident connection */
#define CON_IDCONED		20             /* ident connection complete    */
#define CON_IDREADING		21             /* waiting to read ident sock   */
#define CON_IDREAD		22             /* ident results read           */
#define CON_ASKNAME		23             /* Ask user for name            */
#define CON_IOC			24  /* In/OOC State - UNUSED  */
#define CON_TEXTED		25  /* Tedit editing  */
#define CON_QANSI		26  /* Ask for ANSI support  */
#define CON_CLAN_EDIT		27  /* Clan editing   */
#define CON_MEDIT		28  /*. OLC mode - mobile edit     .*/
#define CON_OEDIT		29  /*. OLC mode - object edit     .*/
#define CON_REDIT		30  /*. OLC mode - room edit       .*/
#define CON_ZEDIT		31  /*. OLC mode - zone info edit  .*/
#define CON_SEDIT		32  /*. OLC mode - shop edit       .*/
#define CON_TRIGEDIT		33  /*. OLC mode - trigger edit       .*/
#define CON_JDINFO		34  /* JD's New CharGenerator	  */
#define CON_QSTR		35  /* JD's New CharGenerator	  */
#define CON_QSAVE		36  /* JD's New CharGenerator	  */
#define CON_QINT		37  /* JD's New CharGenerator	  */
#define CON_QWIS		38  /* JD's New CharGenerator	  */
#define CON_QDEX		39  /* JD's New CharGenerator	  */
#define CON_QCON		40  /* JD's New CharGenerator	  */
#define CON_QCHA		41  /* JD's New CharGenerator	  */
#define CON_NRMOTD		42  /* JD's New CharGenerator!	  */
#define CON_QALIGN		43  /* JD's New CharGenerator	  */
#define CON_QLAWFUL		44  /* JD's New CharGenerator	  */
#define CON_QHEIGHT		45  /* JD's New CharGenerator	  */
#define CON_QWEIGHT		46  /* JD's New CharGenerator	  */
#define CON_QHAIR		47  /* JD's New CharGenerator	  */
#define CON_QEYES		48  /* JD's New CharGenerator	  */
#define CON_QADV		49  /* GURPS-style advantages	  */
#define CON_QDISADV		50  /* GURPS-style disadvantages  */

/* Character equipment positions: used as index for char_data.equipment[] */
/* NOTE: Don't confuse these constants with the ITEM_ bitvectors
   which control the valid places you can wear a piece of equipment */
#define WEAR_FINGER_R   0
#define WEAR_FINGER_L   1
#define WEAR_FINGER_3	2
#define WEAR_FINGER_4	3
#define WEAR_NECK_1	4
#define WEAR_NECK_2	5
#define WEAR_BODY	6
#define WEAR_HEAD	7
#define WEAR_FACE	8
#define WEAR_EYES	9
#define WEAR_EAR_R	10
#define WEAR_EAR_L	11
#define WEAR_LEGS	12
#define WEAR_FEET	13
#define WEAR_HANDS	14
#define WEAR_HANDS_2	15
#define WEAR_ARMS	16
#define WEAR_ARMS_2	17
#define WEAR_ABOUT	18
#define WEAR_BACK	19
#define WEAR_WAIST	20
#define WEAR_ONBELT	21
#define WEAR_WRIST_R	22
#define WEAR_WRIST_L	23
#define WEAR_WRIST_3	24
#define WEAR_WRIST_4	25
#define WEAR_HAND_R	26
#define WEAR_HAND_L	27
#define WEAR_HAND_3	28
#define WEAR_HAND_4	29

#define NUM_WEARS	30	/* This must be the # of eq positions!! */

/* position of messages in the list for WEAR_HAND_x, depending on object */
#define POS_WIELD_TWO	30
#define POS_HOLD_TWO	31
#define POS_WIELD	32
#define POS_WIELD_OFF	33
#define POS_LIGHT	34
#define POS_SHIELD	35
#define POS_HOLD	36




/* object-related defines ********************************************/


/* Item types: used by obj_data.obj_flags.type_flag */
#define ITEM_LIGHT      1		/* Item is a light source	*/
#define ITEM_SCROLL     2		/* Item is a scroll		*/
#define ITEM_WAND       3		/* Item is a wand		*/
#define ITEM_STAFF      4		/* Item is a staff		*/
#define ITEM_WEAPON     5		/* Item is a weapon		*/
#define ITEM_FIREWEAPON 6		/* Item is a fire weapon	*/
#define ITEM_MISSILE    7		/* Item is a missile		*/
#define ITEM_TREASURE   8		/* Item is a treasure, not gold	*/
#define ITEM_ARMOR      9		/* Item is armor		*/
#define ITEM_POTION    10 		/* Item is a potion		*/
#define ITEM_WORN      11		/* Unimplemented		*/
#define ITEM_OTHER     12		/* Misc object			*/
#define ITEM_TRASH     13		/* Trash - shopkeeps won't buy	*/
#define ITEM_TRAP      14		/* Unimplemented		*/
#define ITEM_CONTAINER 15		/* Item is a container		*/
#define ITEM_NOTE      16		/* Item is note 		*/
#define ITEM_DRINKCON  17		/* Item is a drink container	*/
#define ITEM_KEY       18		/* Item is a key		*/
#define ITEM_FOOD      19		/* Item is food			*/
#define ITEM_MONEY     20		/* Item is money (gold)		*/
#define ITEM_PEN       21		/* Item is a pen		*/
#define ITEM_BOAT      22		/* Item is a boat		*/
#define ITEM_FOUNTAIN  23		/* Item is a fountain		*/
#define ITEM_GRENADE   24		/* Item is a grenade		*/
#define ITEM_WEAPONCON 25		/* Item is a scabbard or quiver	*/
#define NUM_ITEM_TYPES 25

/* Take/Wear flags: used by obj_data.obj_flags.wear_flags */
#define ITEM_WEAR_TAKE		(1 << 0)  /* Item can be taken		*/
#define ITEM_WEAR_FINGER	(1 << 1)  /* Can be worn on finger	*/
#define ITEM_WEAR_NECK		(1 << 2)  /* Can be worn around neck 	*/
#define ITEM_WEAR_BODY		(1 << 3)  /* Can be worn on body 	*/
#define ITEM_WEAR_HEAD		(1 << 4)  /* Can be worn on head 	*/
#define ITEM_WEAR_LEGS		(1 << 5)  /* Can be worn on legs	*/
#define ITEM_WEAR_FEET		(1 << 6)  /* Can be worn on feet	*/
#define ITEM_WEAR_HANDS		(1 << 7)  /* Can be worn on hands	*/
#define ITEM_WEAR_ARMS		(1 << 8)  /* Can be worn on arms	*/
#define ITEM_WEAR_SHIELD	(1 << 9)  /* Can be used as a shield	*/
#define ITEM_WEAR_ABOUT		(1 << 10) /* Can be worn about body 	*/
#define ITEM_WEAR_WAIST 	(1 << 11) /* Can be worn around waist 	*/
#define ITEM_WEAR_WRIST		(1 << 12) /* Can be worn on wrist 	*/
#define ITEM_WEAR_WIELD		(1 << 13) /* Can be wielded		*/
#define ITEM_WEAR_HOLD		(1 << 14) /* Can be held		*/
#define ITEM_WEAR_BACK 		(1 << 15) /* Can be worn on back    	*/
#define ITEM_WEAR_ONBELT	(1 << 16) /* Can be worn from waist   	*/
#define ITEM_WEAR_FACE		(1 << 17) /* Can be worn on face 	*/
#define ITEM_WEAR_EYES		(1 << 18) /* Can be worn on eyes 	*/
#define ITEM_WEAR_EAR		(1 << 19) /* Can be worn on ears 	*/
#define ITEM_WEAR_SYMBIOT	(1 << 20) 
#define ITEM_WEAR_TERRAN_HARDPOINT	(1 << 21) 
#define ITEM_WEAR_TRULOR_HARDPOINT	(1 << 22) 


/* Extra object flags: used by obj_data.obj_flags.extra_flags */
#define ITEM_OLC	    (1 << 0)	/* Item made by olc & !approved */
#define ITEM_HUM           (1 << 1)	/* Item is humming		*/
#define ITEM_NORENT        (1 << 2)	/* Item cannot be rented	*/
#define ITEM_NODONATE      (1 << 3)	/* Item cannot be donated	*/
#define ITEM_NOINVIS	   (1 << 4)	/* Item cannot be made invis	*/
#define ITEM_INVISIBLE     (1 << 5)	/* Item is invisible		*/
#define ITEM_MAGIC         (1 << 6)	/* Item is magical		*/
#define ITEM_NODROP        (1 << 7)	/* Item is cursed: can't drop	*/
#define ITEM_BLESS         (1 << 8)	/* Item is blessed		*/
#define ITEM_ANTI_GOOD     (1 << 9)	/* Was ITEM_ANTI_GOOD		*/
#define ITEM_ANTI_EVIL     (1 << 10)	/* Was ITEM_ANTI_EVIL 		*/
#define ITEM_ANTI_NEUTRAL  (1 << 11)	/* Was ITEM_ANTI_NEUTRAL	*/
#define ITEM_ANTI_MAGICIAN (1 << 12)	/* Not usable by magicians      */
#define ITEM_ANTI_CLERIC   (1 << 13)	/* Not usable by enchanters	*/
#define ITEM_ANTI_THIEF	   (1 << 14)	/* Not usable by thieves	*/
#define ITEM_ANTI_WARRIOR  (1 << 15)	/* Not usable by warriors	*/
#define ITEM_NOSELL	   (1 << 16)	/* Shopkeepers won't touch it	*/
#define ITEM_NO_PURGE	    (1 << 17)   /* Must specifically purge      */
#define ITEM_TWO_HAND	    (1 << 18)   /* Requires two hands to hold   */
#define ITEM_HIDDEN         (1 << 19)   /* Item is hidden in the room   */
#define ITEM_PROXIMITY	    (1 << 20)	/* ^^^^^^^^^^^^^^^^^^^^^^^^^^^^ */
#define ITEM_GLOW	    (1 << 21)	/* ^^^^^^^^^^^^^^^^^^^^^^^^^^^^ */
#define ITEM_ANTI_RANGER    (1 << 22)
#define ITEM_ANTI_ASSASSIN  (1 << 23)
#define ITEM_ANTI_ALCHEMIST (1 << 24)
#define ITEM_ANTI_VAMPIRE   (1 << 25)
#define ITEM_ANTI_PSIONICIST (1 << 26)

/* Modifier constants used with obj affects ('A' fields) */
#define APPLY_NONE              0	/* No effect			*/
#define APPLY_STR               1	/* Apply to strength		*/
#define APPLY_DEX               2	/* Apply to dexterity		*/
#define APPLY_INT               3	/* Apply to constitution	*/
#define APPLY_WIS               4	/* Apply to wisdom		*/
#define APPLY_CON               5	/* Apply to constitution	*/
#define APPLY_CHA		6	/* Apply to charisma		*/
#define APPLY_CLASS             7	/* Reserved			*/
#define APPLY_LEVEL             8	/* Reserved			*/
#define APPLY_AGE               9	/* Apply to age			*/
#define APPLY_CHAR_WEIGHT      10	/* Apply to weight		*/
#define APPLY_CHAR_HEIGHT      11	/* Apply to height		*/
#define APPLY_MANA             12	/* Apply to max mana		*/
#define APPLY_HIT              13	/* Apply to max hit points	*/
#define APPLY_MOVE             14	/* Apply to max move points	*/
#define APPLY_GOLD             15	/* Reserved			*/
#define APPLY_EXP              16	/* Reserved			*/
#define APPLY_DR               17	/* Apply to Armor Class		*/
#define APPLY_HITROLL          18	/* Apply to hitroll		*/
#define APPLY_DAMROLL          19	/* Apply to damage roll		*/
#define APPLY_SAVING_PARA      20	/* Apply to save throw: paralz	*/
#define APPLY_SAVING_ROD       21	/* Apply to save throw: rods	*/
#define APPLY_SAVING_PETRI     22	/* Apply to save throw: petrif	*/
#define APPLY_SAVING_BREATH    23	/* Apply to save throw: breath	*/
#define APPLY_SAVING_SPELL     24	/* Apply to save throw: spells	*/
#define APPLY_PD               25	/* Apply to Armor Class		*/


/* Container flags - value[1] */
#define CONT_CLOSEABLE      (1 << 0)	/* Container can be closed	*/
#define CONT_PICKPROOF      (1 << 1)	/* Container is pickproof	*/
#define CONT_CLOSED         (1 << 2)	/* Container is closed		*/
#define CONT_LOCKED         (1 << 3)	/* Container is locked		*/


/* Some different kind of liquids for use in values of drink containers */
#define LIQ_WATER      0
#define LIQ_BEER       1
#define LIQ_WINE       2
#define LIQ_ALE        3
#define LIQ_DARKALE    4
#define LIQ_WHISKY     5
#define LIQ_LEMONADE   6
#define LIQ_FIREBRT    7
#define LIQ_LOCALSPC   8
#define LIQ_SLIME      9
#define LIQ_MILK       10
#define LIQ_TEA        11
#define LIQ_COFFE      12
#define LIQ_BLOOD      13
#define LIQ_SALTWATER  14
#define LIQ_CLEARWATER 15


/* other miscellaneous defines *******************************************/


/* Player conditions */
#define DRUNK        0
#define FULL         1
#define THIRST       2

/* Player point types for events */
#define EVT_HIT      0
#define EVT_MANA     1
#define EVT_MOVE     2

/* Sun state for sunlight */
#define SUN_DARK	0
#define SUN_RISE	1
#define SUN_LIGHT	2
#define SUN_SET		3


/* Sky conditions for weather */
#define SKY_CLOUDLESS	0
#define SKY_CLOUDY	1
#define SKY_RAINING	2
#define SKY_LIGHTNING	3


/* Rent codes */
#define RENT_UNDEF      0
#define RENT_CRASH      1
#define RENT_RENTED     2
#define RENT_CRYO       3
#define RENT_FORCED     4
#define RENT_TIMEDOUT   5

/* loading codes */
#define LOAD_CHARDATA 0

/* other #defined constants **********************************************/

#define MAX_LEVEL	1000 /* 201 is reserved for no level   */

#define MAX_TRUST	8
#define TRUST_IMMORT	1

#define TRUST_ADMIN	MAX_TRUST
#define TRUST_IMPL	7
#define TRUST_SUBIMPL	6
#define TRUST_GRGOD	5
#define TRUST_GOD	4
#define TRUST_CREATOR	3
#define TRUST_LORD	2
#define TRUST_AMBAS	1

#define LVL_FREEZE    TRUST_GRGOD

#define VIEW_OTHERS_CMDS  5	/* miminum trust to view other's commands   */

#define GOD_CMD 	  -1	/* command is usable only by gods (godcmd1) */
#define GOD_CMD2	  -2	/* command is usable only by gods (godcmd2) */
#define MOB_CMD 	  -3	/* command is usable only by mobs */

#define NUM_OF_DIRS	6	/* number of directions in a room (nsewud) */

#define NUM_OBJ_VALUES  8	/* Number of object values */

#define OPT_USEC	100000	/* 10 passes per second */
#define PASSES_PER_SEC	(1000000 / OPT_USEC)
#define RL_SEC		* PASSES_PER_SEC

#define PULSE_ZONE      (pulsezon RL_SEC)
#define PULSE_MOBILE    (pulsemob RL_SEC)
#define PULSE_VIOLENCE  (pulsevio RL_SEC)

/* Variables for the output buffering system */
#define MAX_SOCK_BUF            (12 * 1024) /* Size of kernel's sock buf   */
#define MAX_PROMPT_LENGTH       180          /* Max length of prompt       */
#define GARBAGE_SPACE		32          /* Space for **OVERFLOW** etc  */
#define SMALL_BUFSIZE		1024        /* Static output buffer size   */
/* Max amount of output that can be buffered */
#define LARGE_BUFSIZE	   (MAX_SOCK_BUF - GARBAGE_SPACE - MAX_PROMPT_LENGTH)

#define HISTORY_SIZE		5	/* Keep last 5 commands		*/
#define MAX_STRING_LENGTH	8192
#define MAX_INPUT_LENGTH	256	/* Max length per *line* of input */
#define MAX_RAW_INPUT_LENGTH	512	/* Max size of *raw* input */
#define MAX_MESSAGES		120
#define MAX_POSE_MESSAGES       60
#define MAX_NAME_LENGTH		20  /* Used in char_file_u *DO*NOT*CHANGE* */
#define MAX_PWD_LENGTH		10  /* Used in char_file_u *DO*NOT*CHANGE* */
#define MAX_TITLE_LENGTH	80  /* Used in char_file_u *DO*NOT*CHANGE* */
#define HOST_LENGTH		30  /* Used in char_file_u *DO*NOT*CHANGE* */
#define IDENT_LENGTH            15  /* Used in char_file_u *DO*NOT*CHANGE* */
#define EXDSCR_LENGTH		1600 /* Used in char_file_u *DO*NOT*CHANGE* */
#define MAX_TONGUE		3   /* Used in char_file_u *DO*NOT*CHANGE* */
#define MAX_SKILLS		300 /* Used in char_file_u *DO*NOT*CHANGE* */
#define MAX_AFFECT		32  /* Used in char_file_u *DO*NOT*CHANGE* */
#define MAX_OBJ_AFFECT		6 /* Used in obj_file_elem *DO*NOT*CHANGE* */

// #define NUM_COMBAT_MOVES 	7 // Number of elements in combat_moves struct
#define COMBAT_PROFILE_SIZE 	7 // Number of elements in combat profile array
#define COMBAT_DEFENSIVE	0
#define COMBAT_OFFENSIVE	1
#define COMBAT_BOTH		2

#define SKILL_CRIT_SUCCESS      100
#define SKILL_CRIT_FAILURE      -100
#define SKILL_NOCHANCE          -120

/**********************************************************************
* Structures                                                          *
**********************************************************************/


typedef signed char		sbyte;
typedef unsigned char		ubyte;
typedef signed short int	sh_int;
typedef unsigned short int	ush_int;
typedef unsigned long int	ulong;
#ifndef __cplusplus
typedef char			bool;
#endif

#ifndef CIRCLE_WINDOWS
typedef char			byte;
#endif

typedef sh_int	room_num;	/* A room's vnum type */
typedef sh_int	obj_num;	/* An object's vnum type */
typedef sh_int	mob_num;	/* A mob's vnum type */


/* generic class to track memory use */
class mem_track {
public:
    mem_track(void) {};
    virtual ~mem_track(void) {};

    void *operator new(size_t size);
    void operator delete(void *p, size_t size);
};


/* shared string struct */
typedef struct {
  char *str;
  int count;
} sstring;

#define ss_data(ss)   ((ss) ? (ss)->str : (char *) NULL)


/* all game data */
class all_data : public mem_track {
public:
   all_data(void) : mem_track() {};
   virtual ~all_data(void) {};

   virtual datatypes ClassType(void) { return DATATYPE_ALL_DATA; };
   virtual int Id(void) { return -1; };
   virtual int Vnum(void) { return -1; };

   ubyte purged;			/* this struct is set for purging*/
};

/* generic *_data class (room, char, obj) */
class game_data : public all_data {
public:
   game_data(void) {};
   virtual ~game_data(void) {};

   virtual datatypes ClassType(void) { return DATATYPE_GAME_DATA; };

   virtual char_data * AcquireCharacter(char * arg) { return NULL; };
   virtual char_data * AcquireCharacterRoom(char * arg) { return NULL; };

   virtual obj_data * AcquireObject(char * arg) { return NULL; };
   virtual obj_data * AcquireObjectList(char * arg, obj_data * list) { return NULL; };

   virtual room_data * AcquireRoom(char * arg) { return get_room(arg); };

   virtual void Command(char * argument, int cmd = 0) {};
   virtual void AtCommand(int target, char * argument) {};
   
   virtual char * Keywords(void) { return NULL; };
   virtual char * Name(void) { return NULL; };
   virtual char * SDesc(void) { return NULL; };

   inline virtual void Echo(char * arg, char_data * list) {};
   inline virtual void EchoAt(char * arg, game_data * target) {};
   inline virtual void EchoAround(char * arg, game_data * target, char_data * list) {};
   inline virtual void EchoDistance(char * arg, game_data * target) {};
   virtual void EchoZone(char * arg, int tozone = -1);

   inline virtual int InRoom(void) { return NOWHERE; };

   struct script_data *script;	        /* script info of game_data */
};


/* generic *_data class (room, char, obj) */
class Thing : public game_data {
public:
   Thing(void) {};
   virtual ~Thing(void) {};

   virtual datatypes ClassType(void) { return DATATYPE_THING; };
};


class char_data;
class obj_data;

class BinaryFile;

extern void sub_write(char *arg, char_data *ch, byte find_invis, int targets);

/* Extra description: used in objects, mobiles, and rooms */
struct extra_descr_data {
   sstring *keyword;              /* Keyword in look/examine          */
   sstring *description;          /* What to see                      */
   struct extra_descr_data *next; /* Next in list                     */
};


#define TARGET_CHAR	0
#define TARGET_OBJ	1
#define TARGET_ROOM	2

/* An affect structure */
class Affect {
  public:
   long	bitvector;       /* Tells which bits to set (AFF_XXX)       */
   char_data *caster;	 /* The caster involved, used for concentrations */

   sh_int type;          /* The type of spell that caused this      */
   sbyte modifier;       /* This is added to apropriate ability     */
   byte location;        /* Tells which ability to change(APPLY_XXX)*/
   sbyte level;          /* Level of affect */

   struct event *affect_event; /* The event for this affect         */

   Affect *next;

   void BreakConcentration();
   Affect();
   ~Affect();

};


// Used for concentration on spells or affects
class Concentration {
  public:
   Affect *affect;  // This must match MAX_SPELL_AFFECTS in spells.h!
   void *target;
   byte target_type;

   struct event *concentrate_event; /* The event for this affect         */

   Concentration *next;

   Concentration();
   ~Concentration();
   void BreakConcentration();
   void Update(char_data *ch, bool &broken);
};


/* object-related structures ******************************************/


/* object flags; used in obj_data */
struct obj_flag_data {
   int	value[NUM_OBJ_VALUES];	/* Values of the item (see list)    */
   sh_int hit;
   sh_int max_hit;      // Max hit for object
   sh_int mana;
   sh_int max_mana;     // Max move for object
   byte type_flag;	/* Type of item			    */
   byte material;	// New var: what material object is made of
   byte passive_defense; // New var: passive defense
   byte damage_resistance;	// New var: damage resistance
   int	wear_flags;	/* Where you can wear it	    */
   int	extra_flags;	/* If it hums, glows, etc.	    */
   int	extra_flags2;	/* If it hums, glows, etc.	    */
   int	weight;		/* Weight, what else?               */
   int	cost;		/* Value when sold (gp.)            */
   int	cost_per_day;	/* Cost to keep pr. real day        */
   int  timer;		/* Timer (for object proto)         */
   struct event *timer_event; /* Timer event for object     */
   long	bitvector;	/* To set chars bits                */
};


/* Used in obj_file_elem *DO*NOT*CHANGE* */
struct obj_affected_type {
   byte location;      /* Which ability to change (APPLY_XXX) */
   sbyte modifier;     /* How much it changes by              */
};


/* ================== Memory Structure for Objects ================== */
class obj_data : public Thing {
public:
   obj_data(void);
   obj_data(const obj_data *obj);
   ~obj_data(void);

   virtual datatypes ClassType(void) { return DATATYPE_OBJ_DATA; };
   virtual int Id(void) { return id; };
   virtual int Vnum(void) { return item_number; };
   virtual void Command(char * argument, int cmd = 0)
   {
     obj_command_interpreter(this, argument, cmd);
   };
   virtual void AtCommand(int target, char * argument);

   virtual char * Keywords(void) { return ss_data(this->name); };
   virtual char * Name(void) { return ss_data(this->short_description); };
   virtual char * SDesc(void) { return ss_data(this->description); };

   virtual char_data * AcquireCharacter(char * arg) 
   {
     if (*arg == UID_CHAR)
       return get_char(arg);
     else
       return get_char_by_obj(this, arg); 
   };
   virtual char_data * AcquireCharacterRoom(char * arg) 
   {
     if (*arg == UID_CHAR)
       return get_char(arg);
     else
       return get_char_by_obj(this, arg); 
   };

   virtual obj_data * AcquireObject(char * arg) 
   {
     if (*arg == UID_CHAR)
       return get_obj(arg); 
     else
       return get_obj_by_obj(this, arg); 
   };
   virtual obj_data * AcquireObjectList(char * arg, obj_data * list) 
   { 
     if (*arg == UID_CHAR)
       return get_obj(arg); 
     else
       return get_obj_by_obj(this, arg); 
   };

   virtual void Echo(char * arg, char_data * list)
   {
     if (list)
       sub_write(arg, list, TRUE, TO_ROOM | TO_CHAR);
   };
   virtual void EchoAt(char * arg, game_data * target)
   {
     if (target->ClassType() == DATATYPE_CHAR_DATA)
       sub_write(arg, (char_data *) target, TRUE, TO_CHAR);
   };
   virtual void EchoAround(char * arg, game_data * target, char_data * list)
   {
     target->Echo(arg, list);
   };
   virtual void EchoDistance(char * arg, game_data * target)
   {
     if (target != this)
       target->EchoDistance(arg, target);
   };
   

   virtual int InRoom(void);

   obj_num item_number;	/* Where in data-base			*/
   long id;			/* Unique id                            */
   room_num in_room;		/* In what room -1 when conta/carr	*/

   struct obj_flag_data obj_flags;/* Object information               */
   struct obj_affected_type affected[MAX_OBJ_AFFECT];  /* affects */

   sstring *name;                    /* Title of object :get etc.      */
   sstring *description;	      /* When in room                   */
   sstring *short_description;       /* when worn/carry/in cont.       */
   sstring *action_description;      /* What to write when used        */
   struct extra_descr_data *ex_description; /* extra descriptions     */
   char_data *carried_by;	      /* Carried by :NULL in room/conta */
   char_data *worn_by;		      /* Worn by?		        */
   char_data *dragged_by;	      /* Dragged by :NULL in room/conta */
   char_data *char_on_obj;	      /* Sitting_on :NULL in room/conta */
   sh_int worn_on;		  /* Worn where?		      */

   struct event *obj_event;	// Used for light sources, etc.

   obj_data *in_obj;	   /* In what object NULL when none    */
   obj_data *contains;     /* Contains objects  	       */

   obj_data *next_content; /* For 'contains' lists	       */
   obj_data *next;	   /* For the object list	       */

   Affect *affects;

   void obj_data::copyover(const obj_data *obj);
protected:
   void init(void);
};
/* ======================================================================= */


/* ====================== File Element for Objects ======================= */
/*                 BEWARE: Changing it will ruin rent files		   */
struct obj_file_elem {
   obj_num item_number;
   sh_int  position;

   int	value[4];
   int	extra_flags;
   int	weight;
   int	timer;
   long	bitvector;
   struct obj_affected_type affected[MAX_OBJ_AFFECT];
};


/* header block for rent files.  BEWARE: Changing it will ruin rent files  */
struct rent_info {
   int	time;
   int	rentcode;
   int	net_cost_per_diem;
   int	gold;
   int	account;
   int	nitems;
   int	spare0;
   int	spare1;
   int	spare2;
   int	spare3;
   int	spare4;
   int	spare5;
   int	spare6;
   int	spare7;
};
/* ======================================================================= */


/* room-related structures ************************************************/

#ifdef GOT_RID_OF_IT
struct raff_node {
	room_num room;          /* What room is it in?                 */
	int	 timer;         /* How long will it last?              */
	long	 affection;     /* Which Affection does the Room have? */
	int	 spell;         /* the Spell Number                    */
	
	struct raff_node *next; /* Link to the Next Room Affection     */
};
#endif


struct room_affect_event_data {
  struct Affect *affects;
  struct room_data *room;
};


struct room_direction_data {
   char	*general_description;       /* When look DIR.			*/

   char	*keyword;		/* for open/close			*/

   long exit_info;		/* Exit info				*/
   obj_num key;			/* Key's number (-1 for no key)		*/
   room_num to_room;		/* Where direction leads (NOWHERE)	*/
   ush_int hitpoints;		/* Door's hit points			*/
   ubyte material;		/* Material type of door		*/
   ubyte dam_resist;		/* Door's damage resistance		*/
   sbyte pick_modifier;		/* Difficulty of door to lockpick	*/
};


/* ================== Memory Structure for room ======================= */
class room_data : public game_data {
public:
    room_data(void);
    ~room_data(void);

   virtual datatypes ClassType(void) { return DATATYPE_ROOM_DATA; };
   virtual int Id(void) { return id; };
   virtual int Vnum(void) { return number; };
   virtual void Command(char * argument, int cmd = 0)
   {
     wld_command_interpreter(this, argument, cmd);
   };
   virtual void AtCommand(int target, char * argument);

   virtual char * Keywords(void) { return name; };
   virtual char * Name(void) { return name; };
   virtual char * SDesc(void) { return NULL; };

   virtual char_data * AcquireCharacter(char * arg) 
   {
     if (*arg == UID_CHAR)
       return get_char(arg);
     else
       return get_char_by_room(this, arg);
   };
   virtual char_data * AcquireCharacterRoom(char * arg)
   {
     if (*arg == UID_CHAR)
       return get_char(arg);
     else
       return get_char_by_room(this, arg);
   };

   virtual obj_data * AcquireObject(char * arg) 
   {
     if (*arg == UID_CHAR)
       return get_obj(arg);
     else
       return get_obj_by_room(this, arg);
   };
   virtual obj_data * AcquireObjectList(char * arg, obj_data * list)
   {
     if (*arg == UID_CHAR)
       return get_obj(arg);
     else
       return get_obj_in_list(arg, list);
   };

   virtual void Echo(char * arg, char_data * list)
   {
     if (list) {
       act(arg, FALSE, list, 0, 0, TO_ROOM);
       act(arg, FALSE, list, 0, 0, TO_CHAR);
     }
   };
   virtual void EchoAt(char * arg, game_data * target)
   {
     if (target->ClassType() == DATATYPE_CHAR_DATA)
       sub_write(arg, (char_data *) target, TRUE, TO_CHAR);
   };
   virtual void EchoAround(char * arg, game_data * target, char_data * list)
   {
     target->Echo(arg, list);
   };
   virtual void EchoDistance(char * arg, game_data * target);

   inline virtual int InRoom(void) { return number; };

   room_num number;		/* Rooms number	(vnum)		      */
   long id;			/* Unique id                          */
   sh_int zone;                 /* Room zone (for resetting)          */
   sh_int sector_type;            /* sector type (move/hide)            */
   char	*name;                  /* Rooms name 'You are ...'           */
   char	*description;           /* Shown when entered                 */
   struct extra_descr_data *ex_description; /* for examine/look       */
   struct room_direction_data *dir_option[NUM_OF_DIRS]; /* Directions */
   int room_flags;		/* DEATH,DARK ... etc                 */
   int room_flags2;	       /* DEATH,DARK ... etc		     */
   sh_int room_length;	       /* length of room (n-s) in meters     */
   sh_int room_width;	       /* width of room (e-w) in meters      */
   sh_int room_height;	       /* height of room in meters	     */

   sh_int coordx;		// Coordinates on the surface map;
   sh_int coordy;
   

   byte light;                  /* Number of lightsources in room     */
   SPECIAL(*func);
   char *farg;

   obj_data *contents;  	     /* List of items in room		   */
   char_data *people;		     /* List of NPC / PC in room	   */

   // struct obj_data *contents;   /* List of items in room              */
   // struct char_data *people;    /* List of NPC / PC in room           */
   long  room_affections;       /* Bitvector for room affections      */

   Affect *affects;
};
/* ====================================================================== */


/* char-related structures ************************************************/

/* list of events */
struct event_list {
  struct event *event;
  int type;
  struct event_list *next;
};

/* memory structure for characters */
struct memory_rec_struct {
   long	id;
   time_t last_death;
   struct memory_rec_struct *next;
};

typedef struct memory_rec_struct memory_rec;


// Daniel Houghton's skill structure
// A skill structure for PC's/NPC's
struct skill_store {
   sh_int skillnum;	      /* Number of skill to refer to	      */
   ubyte proficient;	      /* The modifier to the skill roll       */
   ubyte theory;	      /* The number of points invested        */

   struct skill_store *next;  /* Pointer to next skill  	      */
};


class SkillStore {
  private:
   struct skill_store *skills;

  public:
   SkillStore();
   ~SkillStore();


   // Functions to acquire
   int GetTable(struct skill_store **i) const;
   int FindSkill(sh_int num, struct skill_store **i = NULL) const;
   int GetSkillMod(sh_int num = 0, struct skill_store *i = NULL) const;
   int GetSkillPts(sh_int num = 0, struct skill_store *i = NULL) const;
   int GetSkillTheory(sh_int skillnum, struct skill_store *i) const;
   int CheckSkillPrereq(char_data *ch, sh_int &skillnum);
   int GetSkillChance(char_data *ch, sh_int skillnum, 
                      struct skill_store *i, byte chkdefault = 0);
   void FindBestRoll(char_data *ch, sh_int &skillnum, sh_int &defaultnum,
                     byte &chance, byte &mod, sh_int bottom = FIRST_SKILLO, 
		     sh_int top = TOP_SKILLO_DEFINE);
   byte DiceRoll(char_data *ch, sh_int &skillnum, byte &chance, 
                 struct skill_store *i = NULL);
   int RollSkill(char_data *ch, sh_int skillnum);
   int RollSkill(char_data *ch, sh_int skillnum, byte chance);
   int RollSkill(char_data *ch, sh_int skillnum, sh_int defaultnum, byte chance);
   int RollSkill(char_data *ch, struct skill_store *i);
   int SkillContest(char_data *ch, sh_int cskillnum, byte cmod,
                    char_data *victim, sh_int vskillnum, byte vmod,
                    byte &offense, byte &defense);
   int AddSkill(sh_int skillnum, 
		struct skill_store **i = NULL, 
		ubyte pts = 1, ubyte theory = 0);
   int RemoveSkill(sh_int num = 0);
   int LearnSkill(sh_int num, struct skill_store *i = NULL, 
		  sh_int pts = 1);
   int LearnTheory(sh_int num, struct skill_store *i = NULL, 
		   sh_int pts = 1);
   int SetSkill(sh_int num = 0, struct skill_store *i = NULL, 
		ubyte trainpts = 0, ubyte theorypts = 0);
   int LearnSkill(char_data *ch, sh_int num, struct skill_store *i = NULL, 
		  sh_int pts = 1);
   int SetSkill(char_data *ch, sh_int num = 0, struct skill_store *i = NULL, 
		ubyte trainpts = 0, ubyte theorypts = 0);
   int LoadSkills(BinaryFile &pf);
   int SaveSkills(BinaryFile &pf);
   int LoadSkills(FILE *infile, int nr);
   int SaveSkills(FILE *outfile);
   int TotalPoints(char_data *ch);
   void ClearSkills();
   sh_int FindBestDefault(char_data *ch, sh_int &skillnum, int chance, 
                          byte &mod, byte chkdefault);

};


// Daniel Houghton's combat profile
class CombatProfile {
  private:
    byte attack_offset, defense_offset;
    byte posture;			// Normal, defense, or offense?
    ubyte moves[COMBAT_PROFILE_SIZE + 1];

  public:
    // This is a minor violation of object-oriented methodology, but rather
    // than get religious about it, let's do what works.  A lot of functions
    // will be wanting to access these variables, and we don't want to muck
    // with friend classes just for this, do we? -DH
    byte attack_pts, defense_pts;

    CombatProfile();
    ~CombatProfile();
    
    int GetMove(ubyte offset);
    int SetMove(sh_int &move, byte &offset);
    int SetPosture(char *string);
    void NextMove(byte &move_offset);
    void NextOffense();
    void NextDefense();
    int CombatMove(byte type);
    void CombatCost(byte type, byte cost);
    void PointUpdate(char_data *ch);
    void PointUpdate(int increase);
    void ZeroPoints(byte type = COMBAT_BOTH);
    void ProfileDisplay(char_data *ch);
    int SaveProfile(BinaryFile &pf);
    int LoadProfile(BinaryFile &pf);

    friend void do_circle(struct char_data *ch, char *command, char *argument, 
                     int cmd, int subcmd);
    friend void combat_round(char_data *ch, char_data *victim);

};


/* This structure is purely intended to be an easy way to transfer */
/* and return information about time (real or mudwise).            */
struct time_info_data {
   int hours, day, month;
   sh_int year;
};


/* These data contain information about a players time data */
struct time_data {
   time_t birth;    /* This represents the characters age                */
   time_t logon;    /* Time of the last logon (used to calculate played) */
   int	played;     /* This is the total accumulated time played in secs */
};

/* This is for matching up runes with spells */
struct rune_player_type {
    ubyte trigger;
    ubyte location;
    ubyte sphere;
    ubyte power;
    ubyte technique;
    ubyte manifest;
    ubyte focus;
    char name[MAX_INPUT_LENGTH];
};

/* general player-related info, usually PC's and NPC's */
struct char_player_data {
    char passwd[MAX_PWD_LENGTH+1]; /* character's password      */
    sstring *name;                 /* PC / NPC s name (kill ...  )         */
    sstring *short_descr;          /* for NPC 'actions'                    */
    sstring *long_descr;           /* for 'look'			   */
    sstring *description;          /* Extra descriptions                   */
    sstring *title;                /* PC / NPC's title                     */
    struct time_data time;         /* PC's AGE in days                     */
    long level;         /* PC / NPC's level                     */
    ubyte hometown;		   /* PC s Hometown (zone)                 */
    byte sex;           /* PC / NPC's sex                       */
    byte clss;                     /* PC / NPC's class		           */
    byte race;                     /* PC / NPC's race                      */
    ubyte bodytype;		   /* PC / NPC's patron bodytype           */
    ubyte weight;       /* PC / NPC's weight                    */
    ubyte height;       /* PC / NPC's height                    */
    ubyte material;	// Material type - generally referring to outer hide
};


/* Char's abilities.  Used in char_file_u *DO*NOT*CHANGE* */
struct char_ability_data {
   sbyte str;
   sbyte intel;
   sbyte wis;
   sbyte dex;
   sbyte con;
   sbyte cha;
   sbyte vision;
   sbyte hearing;
   sbyte movement;
   sbyte magery;
};


/* Char's points.  Used in char_file_u *DO*NOT*CHANGE* */
struct char_point_data {
   sh_int mana;
   sh_int max_mana;     /* Max move for PC/NPC			   */
   sh_int hit;
   sh_int max_hit;      /* Max hit for PC/NPC                      */
   sh_int move;
   sh_int max_move;     /* Max move for PC/NPC                     */

   int	gold;           /* Money carried                           */
   int	bank_gold;	/* Gold the char has in a bank account	   */
   int	exp;            /* The experience of the player            */
   int	totalpts;       /* The experience of the player            */

   sbyte hitroll;       /* Any bonus or penalty to the hit roll    */
   sbyte damroll;       /* Any bonus or penalty to the damage roll */
   sbyte passive_defense; // New var: what material object is made of
   ubyte damage_resistance;	// New var: what material object is made of
};


/* 
 * char_special_data_saved: specials which both a PC and an NPC have in
 * common, but which must be saved to the playerfile for PC's.
 *
 * WARNING:  Do not change this structure.  Doing so will ruin the
 * playerfile.  If you want to add to the playerfile, use the spares
 * in player_special_data.
 */
struct char_special_data_saved {
   int	alignment;		/* +-1000 for alignments                */
   int	lawfulness;		/* +-1000 for alignments                */
   long	idnum;			/* player's idnum; -1 for mobiles	*/
   long	act;			/* act flag for NPC's; player flag for PC's */
   long	act2;			/* act2 flag for NPC's; player flag for PC's */

   long	affected_by;		/* Bitvector for spells/skills affected by */
   long	affected_by2;		/* Bitvector for spells/skills affected by */

   long advantages;
   long disadvantages;

   sh_int exp_modifier;

   sh_int apply_saving_throw[5]; /* Saving throw (Bonuses)		*/
   sh_int apply_resistance[MAX_DAMTYPE + 1]; /* Resistance to damage (Bonuses) */
};


/* Special playing constants shared by PCs and NPCs which aren't in pfile */
struct char_special_data {
   char_data *fighting;		/* Opponent				*/
   char_data *riding;		/* Char ridden by this char             */
   char_data *rider;		/* Person riding this mount             */
   char_data *grappling;	// Person being grappled (future use)
   char_data *grappler;		// Person grappling this character (future use)

   long hunting;		/* Char hunted by this char		*/
   long seeking;		/* Char sought by this char		*/
   long stalking;		/* Char stalked by this char		*/
   long aiming;			/* Char hunted by this char		*/

   obj_data *obj_dragged;	// Object being dragged by character
   obj_data *obj_on;		// Object character is on

   room_num roomseeking;	/* Room sought by this char		*/

   byte position;		/* Standing, fighting, sleeping, etc.	*/
   byte carry_items;		/* Number of items carried		*/

   int	carry_weight;		/* Carried weight			*/
   int	timer;			/* Timer for update			*/

// This shouldn't be in EVERY mob in the game.  Rather, we'll put together
// a separate index for this data.
//   unsigned long int deaths;		/* Mobs death count!			*/
//   unsigned long int kills;		/* Mobs kill count!			*/

   struct event *action;	/* event for action which takes time    */

   struct char_special_data_saved saved; /* constants saved in plrfile	*/
};


/*
 *  If you want to add new values to the playerfile, do it here.  DO NOT
 * ADD, DELETE OR MOVE ANY OF THE VARIABLES - doing so will change the
 * size of the structure and ruin the playerfile.  However, you can change
 * the names of the spares to something more meaningful, and then use them
 * in your new code.  They will automatically be transferred from the
 * playerfile into memory when players log in.
 */
struct player_special_data_saved {
   // byte skills[MAX_SKILLS+1];	/* array of skills plus skill 0		*/
   // byte PADDING0;	        /* used to be spells_to_learn           */
   // byte talks[MAX_TONGUE];	/* PC s Tongues 0 for NPC		*/
   int	wimp_level;		/* Below this # of hit points, flee!	*/
   sh_int invis_level;		/* level of invisibility		*/
   sh_int freeze_level;		/* level of invisibility		*/
   room_num load_room;		/* Which room to place char in		*/
   long	pref;			/* preference flags for PC's.		*/
   long	pref2;			/* preference flags for PC's.		*/
   ubyte bad_pws;		/* number of bad password attemps	*/
   sbyte conditions[3];         /* Drunk, full, thirsty			*/

   long max_level;		/* Highest level PC has been            */
   long level_died;		/* level last died at                   */

   long godcmds1;
   long	godcmds2;

   long motdStamp;
   int spells_to_learn;		/* How many can you learn yet this level*/
   time_t last_death;		/* when the person last died 	        */
   int clan;
   ubyte clan_rank;
   ubyte saycolor;
   byte page_length;		/* number of lines in a page of output  */
   ubyte trust;

   char *wwwadd;

   char *prompt;
   
   char *ice_listen;

   long imc_deaf;             /* IMC deaf-to_channel flags            */
   long imc_allow;            /* IMC channel allow overrides          */
   long imc_deny;             /* IMC channel deny overrides           */

};

/*
 * Specials needed only by PCs, not NPCs.  Space for this structure is
 * not allocated in memory for NPCs, but it is for PCs and the portion
 * of it labelled 'saved' is saved in the playerfile.  This structure can
 * be changed freely; beware, though, that changing the contents of
 * player_special_data_saved will corrupt the playerfile.
 */
struct player_special_data {
   struct player_special_data_saved saved;

   char	*poofin;		/* Description on arrival of a god.     */
   char	*poofout;		/* Description upon a god's exit.       */
   struct alias *aliases;	/* Character's aliases			*/
   long last_tell;		/* idnum of last tell from		*/
#ifdef OASIS_OLC
   struct olc_data *olc;	     /*. OLC info - defined in olc.h   .*/
#endif /* OASIS_OLC */
   int last_olc_targ;           /* olc control                          */
   int last_olc_mode;		/* olc control				*/
   char	*host;			/* old hostname				*/
   char *ident;			/* old ident                            */
   struct rune_player_type runes; /* runes up in head */
   unsigned long int deaths;	// Player death count
   unsigned long int kills;     // Player kill count
};


/* Specials used by NPCs, not PCs */
struct mob_special_data {
   int	attack_type;	/* The Attack Type Bitvector for NPC's*/
   int  attack_num;	/* Type number of attacks of that type */
   memory_rec *memory;	    /* List of attackers to remember	       */
   int wait_state;	    /* Wait state for bashed mobs	       */
   sh_int disposition;		       /* How easy mount is to control  */
   // sh_int exp_modifier;
   byte max_riders;		       /* Number of riders possible     */
   // byte last_direction;     /* The last direction the monster went     */
   byte default_pos;        /* Default position for NPC                */
   byte damnodice;    /* The number of damage dice's   */
   byte damsizedice;  /* The size of the damage dice's */
};


/* Structure used for chars following other chars */
struct follow_type {
   char_data *follower;
   struct follow_type *next;
};


/* ================== Structure for player/non-player ===================== */
class char_data : public Thing {
public:
   char_data(void);
   char_data(const char_data *ch);
   ~char_data(void);

   virtual datatypes ClassType(void) { return DATATYPE_CHAR_DATA; };
   virtual int Id(void) { return id; };
   virtual int Vnum(void) { return nr; };
   virtual void Command(char * argument, int cmd = 0)
   {
     command_interpreter(this, argument, cmd);
   };
   virtual void AtCommand(int target, char * argument);

   virtual char * Keywords(void) { return ss_data(player.name); };
   virtual char * Name(void) { return ss_data(player.short_descr); };
   virtual char * SDesc(void) { return ss_data(player.long_descr); };
   virtual char * Title(void) { return ss_data(player.title); };

   virtual char_data * AcquireCharacter(char * arg)
   { 
     if (*arg == UID_CHAR)
       return get_char(arg);
     else
       return get_char_vis(this, arg);
   };

   virtual char_data * AcquireCharacterRoom(char * arg)
   { 
     if (*arg == UID_CHAR)
       return get_char(arg);
     else
       return get_char_room_vis(this, arg);
   };

   virtual obj_data * AcquireObject(char * arg)
   { 
     if (*arg == UID_CHAR)
       return get_obj(arg);
     else 
       return get_obj_vis(this, arg); 
   };
   virtual obj_data * AcquireObjectList(char * arg, obj_data * list)
   {
     if (*arg == UID_CHAR)
       return get_obj(arg);
     else 
       return get_obj_in_list_vis(this, arg, list); 
   };

   virtual void Echo(char * arg, char_data * list)
   {
     sub_write(arg, this, TRUE, TO_ROOM);
   };
   virtual void EchoAt(char * arg, game_data * target)
   {
     if (target->ClassType() == DATATYPE_CHAR_DATA)
       sub_write(arg, (char_data *) target, TRUE, TO_CHAR);
   };
   virtual void EchoAround(char * arg, game_data * target, char_data * list)
   {
     target->Echo(arg, list);
   };
   virtual void EchoDistance(char * arg, game_data * target);

   inline virtual int InRoom(void) { return in_room; };

   int save(int load_room = NOWHERE);
   int load(char *name, int mode = LOAD_CHARDATA);

   char *name(void)	       { return ss_data(player.name); };
   char *short_descr(void)     { return ss_data(player.short_descr); };
   char *title(void)	       { return ss_data(player.title); };

   void set_name(char *name);
   void set_short(char *short_descr = NULL);
   void set_title(char *title = NULL);

   void to_world(void);
   void from_world(void);

   /* these things should eventually be made private */

   sh_int nr;			       /* Mob's rnum			*/
   long id;			       /* Unique id of character	*/
   room_num in_room;                    /* Location (real room number)	  */
   room_num was_in_room;		 /* location for linkdead people  */

   struct char_player_data player;       /* Normal data                   */
   struct char_ability_data real_abils;	 /* Abilities without modifiers   */
   struct char_ability_data aff_abils;	 /* Abils with spells/stones/etc  */
   struct char_point_data points;        /* Points                        */
   struct char_special_data char_specials;	/* PC/NPC specials	  */
   struct player_special_data *player_specials; /* PC specials		  */
   struct player_special_data_saved *player_specials_saved; /* PC saved specs */
   struct mob_special_data mob_specials;	/* NPC specials		  */

   SkillStore skills;		/* The character's skills	 */
   CombatProfile combat_profile;

   Affect *affects;       /* affected by what spells       */
   Concentration *concentration; /* concentration on what spells       */
   
    obj_data *equipment[NUM_WEARS];	/* Equipment array               */

    obj_data *carrying;			/* Head of list                  */

   struct descriptor_data *desc;         /* NULL for mobiles              */

    char_data *next_in_room;		/* For room->people - list       */
    char_data *next;			/* For either monster or ppl-list*/
    char_data *next_fighting;		/* For fighting list             */
    char_data *next_riding;		/* For other people sharing mount*/
    char_data *next_on_obj;		/* For other people on object    */

   struct follow_type *followers;        /* List of chars followers       */
    char_data *master;			/* Who is char following?        */

    struct event *points_event[3];	/* events for regening H/M/V     */
    struct event_list *damage_events;	/* list of events damaging char  */
    struct script_memory *memory;       /* for mob memory triggers         */
    ubyte group_pos;			// Position in a group (fore/rear)
    sbyte type;			        /* Is this a simple or extended mob? */


   void char_data::copyover(const char_data *ch);

protected:
    void init(void);

    // saving routines
    int save_player_data(BinaryFile &pf);
    int save_player_data_disk(BinaryFile &pf);
    int save_aliases(BinaryFile &pf);
    int save_skills(BinaryFile &pf);
    int save_runes(BinaryFile &pf);
    int save_objects(BinaryFile &pf);
    int save_cookies(BinaryFile &pf);

    // load routines
    int load_player_data(BinaryFile &pf);
    int load_aliases(BinaryFile &pf);
    int load_skills(BinaryFile &pf);
    int load_runes(BinaryFile &pf);
    int load_objects(BinaryFile &pf);
    int load_cookies(BinaryFile &pf);
};
/* ====================================================================== */


/* An affect structure.  Used in char_file_u *DO*NOT*CHANGE* */
struct affected_file_type {
  sh_int type;          /* The type of spell that caused this      */
  sbyte modifier;       /* This is added to apropriate ability     */
  byte location;        /* Tells which ability to change(APPLY_XXX)*/
  long bitvector;       /* Tells which bits to set (AFF_XXX)       */
  long duration;	/* How long the spell lasts                */
};


/* ==================== File Structure for Player ======================= */
/*             BEWARE: Changing it will ruin the playerfile		  */
struct char_file_u {
   /* char_player_data */
   byte sex;
   byte clss;
   byte race;
   byte level;
   sh_int hometown;
   time_t birth;   /* Time of birth of character     */
   int	played;    /* Number of secs played in total */

   struct char_special_data_saved char_specials_saved;
   struct player_special_data_saved player_specials_saved;
   struct char_ability_data abilities;
   struct char_point_data points;
   struct affected_file_type affected[MAX_AFFECT];

   time_t last_logon;		/* Time (in secs) of last logon */
   char host[HOST_LENGTH+1];	/* host of last logon */
   char ident[IDENT_LENGTH+1];  /* account of last logon        */
};
/* ====================================================================== */


/* descriptor-related structures ******************************************/


#define EDITFUNC(name)	void (name)(struct descriptor_data *d, \
				    char *str, void *obj, int terminator)
#define EDIT_APPEND	    (1 << 0)	/* Append byte flag for editor */
#define EDIT_ONELINE	    (1 << 1)	/* One-line byte flag for editor */

struct editor_data {
  char *str;			/* string being edited             */
  size_t max_str;		/* max length of string            */
  EDITFUNC(*func);		/* finishing func to handle string */
  void   *obj;			/* info needed by finishing func   */
  byte flags;			/* Configuration flags for editor  */
};


struct txt_block {
   char	*text;
   int aliased;
   struct txt_block *next;
};


struct txt_q {
   struct txt_block *head;
   struct txt_block *tail;
};


#define TERM_PLAIN	0
#define TERM_ANSI	1
#define TERM_VT100	2


struct descriptor_data {
   socket_t	descriptor;	/* file descriptor for socket		*/
   int  ident_sock;		/* socket used for ident process	*/
   int  peer_port;		/* port of peer				*/
   char	host[HOST_LENGTH+1];	/* hostname				*/
   char ident[IDENT_LENGTH+1];  /* account name                         */
   byte close_me;               /* flag: this desc. should be closed    */
   byte	bad_pws;		/* number of bad pw attemps this login	*/
   byte idle_tics;		/* tics idle at password prompt		*/
   ubyte termtype;		// The type of terminal settings
   int	connected;		/* mode of 'connectedness'		*/
   int	wait;			/* wait for how many loops		*/
   int	desc_num;		/* unique num assigned to desc		*/
   time_t login_time;		/* when the person connected		*/
   char *showstr_head;		/* for keeping track of an internal str	*/
   char **showstr_vector;	/* for paging through texts		*/
   int  showstr_count;		/* number of pages to page through	*/
   int  showstr_page;		/* which page are we currently showing?	*/
   struct editor_data *ed;	/* used by the string editing system    */
   int	has_prompt;		/* is the user at a prompt?             */
   char	inbuf[MAX_RAW_INPUT_LENGTH];  /* buffer for raw input		*/
   char	last_input[MAX_INPUT_LENGTH]; /* the last input			*/
   char small_outbuf[SMALL_BUFSIZE];  /* standard output buffer		*/
   char *output;		/* ptr to the current output buffer	*/
   char **history;             /* History of commands, for ! mostly.   */
   int history_pos;            /* Circular array position.             */
   int  bufptr;			/* ptr to end of current output		*/
   int	bufspace;		/* space left in the output buffer	*/
   struct txt_block *large_outbuf; /* ptr to large buffer, if we need it */
   struct txt_q input;		/* q of unprocessed input		*/
   struct char_data *character;	/* linked to char			*/
   struct char_data *original;	/* original char if switched		*/
   struct descriptor_data *snooping; /* Who is this char snooping	*/
   struct descriptor_data *snoop_by; /* And who is snooping this char	*/
   struct descriptor_data *next; /* link to next descriptor		*/
};


/* other miscellaneous structures ***************************************/

typedef SPECIAL(*SpecProc);

struct msg_type {
   char	*attacker_msg;  /* message to attacker */
   char	*victim_msg;    /* message to victim   */
   char	*room_msg;      /* message to room     */
};


struct message_type {
   struct msg_type die_msg;	/* messages when death			*/
   struct msg_type miss_msg;	/* messages when miss			*/
   struct msg_type hit_msg;	/* messages when hit			*/
   struct msg_type god_msg;	/* messages when hit on god		*/
   struct message_type *next;	/* to next messages of this kind.	*/
};


struct message_list {
   int	a_type;			/* Attack type				*/
   int	number_of_attacks;	/* How many attack messages to chose from. */
   struct message_type *msg;	/* List of messages.			*/
};


// Body wear slots added in by Daniel Houghton
struct body_wear_slots {
  byte hitlocation;	// The location on the 2d6 hit table, -1 means that it doesn't exist.
  ubyte damage_mult;	// Damage multiplier for hitting sensitive locations (default is 10)
  ubyte armor_value;	// Added to passive defense for a given location
  ubyte damage_max;	// Maximum damage for this slot
  byte msg_wearon;	// The number element to use in the wear_on_message array.
  byte msg_already_wear;	// The number element to use in the already_wearing array.
  int wear_bitvector;	// The bitvector of what items can be held in a given spot.
  char *partname;	// The long name of the body part
  char *keywords;	// The short name used to designate a wear location.
};


struct wear_on_message {
  char *worn_msg;
  char *to_room;
  char *to_char;
};


struct combat_profile_moves {
  char *move;
  sh_int skillnum;
  sh_int message;
  byte speed;
  byte locationmod;
  byte type;  // If true, offense.
  void (*command_pointer)
  (struct char_data *ch, char *command, char *argument, int cmd, int subcmd);
  int subcmd;
};


// Used for table of object values in constants.c
struct oedit_objval_tab {
  char *prompt;
  long min;
  long max;
  const char **values;		/* list of values, terminated with '\n'   */
  byte type;			/* type of variable (int/long...) EDT_x   */
};


// A "package" to be handed down to hit() containing all necessary information
// for an attack.
class Attack {
  public:
    obj_data *weapon;   // Weapon used

    sh_int move,	// What combat move is being performed
           skillnum,	// Skill number to be rolled against for success
	   message,	// Message number to use
	   dam;		// Damage plus

    byte chance,	// Chance on 3d6 of the attack succeeding
         mod,		// Modifier to skillroll
         speed,		// Cost in Combat Points
	 damtype,	// Damage type (basic, slashing, piercing)
	 location,	// Body location to be struck
	 hand,		// Which hand is being used.  Will be useful for
	 		// dual wielding.
	 response,	// Indicates success of defense move.
	 succeed_by;    // Modified by hit(), indicating success.
	 
    Attack();
    ~Attack();
    int WeaponMove(char_data *ch, sh_int do_move);

};


struct syllable {
  char *org;
  char *news;
};


struct advantage_dat {
  long advantage_filter;
  long disadvantage_filter;
  byte cost;
};
