/************************************************************************
 *  OasisOLC - zedit.c						v1.5	*
 *									*
 *  Copyright 1996 Harvey Gilpin.					*
 ************************************************************************/

#include "conf.h"
#include "sysdep.h"

#include "structs.h"
#include "scripts.h"
#include "index.h"
#include "comm.h"
#include "spells.h"
#include "utils.h"
#include "db.h"
#include "boards.h"
#include "olc.h"
#include "constants.h"
#include "events.h"

/*
 * Turn on zedit debugging.  Please mail log results to greerga@van.ml.org
 * This will attempt to find the problem with replacing other zedit commands
 * when you add unrelated ones.
 */
#if 0
#define DEBUG	1
#endif

/*-------------------------------------------------------------------*/

/*
 * External data structures.
 */
// extern struct zone_data *zone_table;
// extern struct room_data *world;
// extern int top_of_zone_table;
// extern struct char_data *mob_index;
// extern struct index_data *mob_index;
// extern struct obj_data *obj_proto;
// extern struct index_data *obj_index;
// extern char *equipment_types[];
// extern char *dirs[];

/*-------------------------------------------------------------------*/

/*
 * Function prototypes.
 */
void zedit_disp_menu(char_data *ch);
void zedit_setup(char_data *ch, int room_num);
void add_cmd_to_list(struct reset_com **list, struct reset_com *newcmd, int pos);
void remove_cmd_from_list(struct reset_com **list, int pos);
void delete_command(char_data *ch, int pos);
int new_command(char_data *ch, int pos);
int start_change_command(char_data *ch, int pos);
void zedit_disp_comtype(char_data *ch);
void zedit_disp_arg1(char_data *ch);
void zedit_disp_arg2(char_data *ch);
void zedit_disp_arg3(char_data *ch);
void zedit_save_internally(char_data *ch);
void zedit_save_to_disk(int zone_num);
void zedit_create_index(int znum, char *type);
void zedit_new_zone(struct char_data *ch, int vzone_num);

/*-------------------------------------------------------------------*/

/*
 * Nasty internal macros to clean up the code.
 */
#define ZCMD		(zone_table[OLC_ZNUM(ch)].cmd[subcmd])
#define MYCMD		(OLC_ZONE(ch)->cmd[subcmd])
#define OLC_CMD(ch)	(OLC_ZONE(ch)->cmd[OLC_VAL(ch)])
#define PROTO_MOB(num)  ((char_data *) mob_index[(num)]->proto)
#define PROTO_OBJ(num)  ((obj_data *) obj_index[(num)]->proto)
#define PROTO_TRIG(num) ((trig_data *) trig_index[(num)]->proto)
/*-------------------------------------------------------------------*/

/*
 * Utility functions.
 */

/*-------------------------------------------------------------------*/

void zedit_setup(char_data *ch, int room_num)
{
  struct zone_data *zone;
  int subcmd = 0, count = 0, cmd_room = -1;

  /*
   * Allocate one scratch zone structure.  
   */
  CREATE(zone, struct zone_data, 1);

  /*
   * Copy all the zone header information over.
   */
  zone->name = str_dup(zone_table[OLC_ZNUM(ch)].name);
  zone->lifespan = zone_table[OLC_ZNUM(ch)].lifespan;
  zone->top = zone_table[OLC_ZNUM(ch)].top;
  zone->reset_mode = zone_table[OLC_ZNUM(ch)].reset_mode;
  /*
   * The remaining fields are used as a 'has been modified' flag  
   */
  zone->number = 0;	/* Header information has changed.	*/
  zone->age = 0;	/* The commands have changed.		*/

  /*
   * Start the reset command list with a terminator.
   */
  CREATE(zone->cmd, struct reset_com, 1);
  zone->cmd[0].command = 'S';

  /*
   * Add all entries in zone_table that relate to this room.
   */
  while (ZCMD.command != 'S') {
    switch (ZCMD.command) {
    case 'H':
    case 'M':
    case 'O':
      cmd_room = ZCMD.arg3;
      break;
    case 'D':
    case 'R':
      cmd_room = ZCMD.arg1;
      break;
    default:
      break;
    }
    if (cmd_room == room_num) {
#if defined(DEBUG)
      log("zedit_setup called add_cmd_to_list.");
#endif
      add_cmd_to_list(&(zone->cmd), &ZCMD, count);
      count++;
    }
    subcmd++;
  }

  OLC_ZONE(ch) = zone;
  /*
   * Display main menu.
   */
  zedit_disp_menu(ch);
}

/*-------------------------------------------------------------------*/

/*
 * Save all the information in the player's temporary buffer back into
 * the current zone table.
 */
void zedit_save_internally(char_data *ch)
{
  int subcmd = 0, cmd_room = -2, room_num = real_room(OLC_NUM(ch));

  /*
   * Delete all entries in zone_table that relate to this room so we
   * can add all the ones we have in their place.
   */
  while (ZCMD.command != 'S') {
    switch (ZCMD.command) {
    case 'H':
    case 'M':
    case 'O':
      cmd_room = ZCMD.arg3;
      break;
    case 'D':
    case 'R':
      cmd_room = ZCMD.arg1;
      break;
    default:
      break;
    }
    if (cmd_room == room_num) {
#if defined(DEBUG)
      log("zedit_save_internally called remove_cmd_from_list.");
#endif
      remove_cmd_from_list(&(zone_table[OLC_ZNUM(ch)].cmd), subcmd);
    } else
      subcmd++;
  }

  /*
   * Now add all the entries in the players descriptor list  
   */
  subcmd = 0;
  while (MYCMD.command != 'S') {
#if defined(DEBUG)
    log("zedit_save_internally called add_cmd_to_list.");
#endif
    add_cmd_to_list(&(zone_table[OLC_ZNUM(ch)].cmd), &MYCMD, subcmd);
    subcmd++;
  }

  /*
   * Finally, if zone headers have been changed, copy over  
   */
  if (OLC_ZONE(ch)->number) {
    free(zone_table[OLC_ZNUM(ch)].name);
    zone_table[OLC_ZNUM(ch)].name = str_dup(OLC_ZONE(ch)->name);
    zone_table[OLC_ZNUM(ch)].top = OLC_ZONE(ch)->top;
    zone_table[OLC_ZNUM(ch)].reset_mode = OLC_ZONE(ch)->reset_mode;
    zone_table[OLC_ZNUM(ch)].lifespan = OLC_ZONE(ch)->lifespan;
  }
  olc_add_to_save_list(zone_table[OLC_ZNUM(ch)].number, OLC_ZEDIT);
}

/*-------------------------------------------------------------------*/

/*
 * Some common code to count the number of comands in the list.
 */
int count_commands(struct reset_com *list)
{
  int count = 0;

  while (list[count].command != 'S')
    count++;

  return count;
}

/*-------------------------------------------------------------------*/

/*
 * Adds a new reset command into a list.  Takes a pointer to the list
 * so that it may play with the memory locations.
 */
void add_cmd_to_list(struct reset_com **list, struct reset_com *newcmd, int pos)
{
  int count, i, l;
  struct reset_com *newlist;

  /*
   * Count number of commands (not including terminator).
   */
  count = count_commands(*list);

  /*
   * Value is +2 for the terminator and new field to add.
   */
  CREATE(newlist, struct reset_com, count + 2);

  /*
   * Even tighter loop to copy the old list and insert a new command.
   */
  for (i = 0, l = 0; i <= count; i++) {
    newlist[i] = ((i == pos) ? *newcmd : (*list)[l++]);
#if defined(DEBUG)
    sprintf(buf, "add_cmd_to_list: added %c %d %d %d %d",
		newlist[i].command, newlist[i].arg1, newlist[i].arg2,
		newlist[i].arg3, newlist[i].line);
    log(buf);
#endif
  }

  /*
   * Add terminator, then insert new list.
   */
  newlist[count + 1].command = 'S';
  free(*list);
  *list = newlist;
}

/*-------------------------------------------------------------------*/

/*
 * Remove a reset command from a list.  Takes a pointer to the list
 * so that it may play with the memory locations.
 */
void remove_cmd_from_list(struct reset_com **list, int pos)
{
  int count, i, l;
  struct reset_com *newlist;

  /*
   * Count number of commands (not including terminator)  
   */
  count = count_commands(*list);

  /*
   * Value is 'count' because we didn't include the terminator above
   * but since we're deleting one thing anyway we want one less.
   */
  CREATE(newlist, struct reset_com, count);

  /*
   * Even tighter loop to copy old list and skip unwanted command.
   */
  for (i = 0, l = 0; i < count; i++) {
    if (i != pos) {
#if defined(DEBUG)
      sprintf(buf, "remove_cmd_from_list: kept %c %d %d %d %d",
		(*list)[i].command, (*list)[i].arg1, (*list)[i].arg2,
		(*list)[i].arg3, (*list)[i].line);
#endif
      newlist[l++] = (*list)[i];
    }
#if defined(DEBUG)
    else
      sprintf(buf, "remove_cmd_from_list: deleted %c %d %d %d %d",
		(*list)[i].command, (*list)[i].arg1, (*list)[i].arg2,
		(*list)[i].arg3, (*list)[i].line);
    log(buf);
#endif
  }
  /*
   * Add the terminator, then insert the new list.
   */
  newlist[count - 1].command = 'S';
  free(*list);
  *list = newlist;
}

/*-------------------------------------------------------------------*/

/*
 * Error check user input and then add new (blank) command  
 */
int new_command(char_data *ch, int pos)
{
  int subcmd = 0;
  struct reset_com *new_com;

  /*
   * Error check to ensure users hasn't given too large an index  
   */
  while (MYCMD.command != 'S')
    subcmd++;

  if ((pos > subcmd) || (pos < 0))
    return 0;

  /*
   * Ok, let's add a new (blank) command 
   */
  CREATE(new_com, struct reset_com, 1);
  new_com->command = 'N';
#if defined(DEBUG)
  log("new_command called add_cmd_to_list.");
#endif
  add_cmd_to_list(&OLC_ZONE(ch)->cmd, new_com, pos);
  return 1;
}

/*-------------------------------------------------------------------*/
/*
 * Error check user input and then remove command  
 */

void delete_command(char_data *ch, int pos)
{
  int subcmd = 0;

  /*
   * Error check to ensure users hasn't given too large an index  
   */
  while (MYCMD.command != 'S')
    subcmd++;

  if ((pos >= subcmd) || (pos < 0))
    return;

  /*
   * Ok, let's zap it  
   */
#if defined(DEBUG)
  log("delete_command called remove_cmd_from_list.");
#endif
  remove_cmd_from_list(&OLC_ZONE(ch)->cmd, pos);
}

/*-------------------------------------------------------------------*/
/*
 * Error check user input and then setup change  
 */

int start_change_command(char_data *ch, int pos)
{
  int subcmd = 0;

  /*
   * Error check to ensure users hasn't given too large an index  
   */
  while (MYCMD.command != 'S')
    subcmd++;

  if ((pos >= subcmd) || (pos < 0))
    return 0;

  /*
   * Ok, let's get editing  
   */
  OLC_VAL(ch) = pos;
  return 1;
}

/**************************************************************************
 Menu functions 
 **************************************************************************/

/*
 * the main menu 
 */
void zedit_disp_menu(char_data *ch)
{
  int subcmd = 0, room, counter = 0;

  room = real_room(OLC_NUM(ch));

  /*
   * Menu header  
   */
  sprintf(buf,
#if defined(CLEAR_SCREEN)
	  "[H[J"
#endif
	  "Room number: &cC%d&c0		Room zone: &cC%d&c0\r\n"
	  "&cGZ&c0) Zone name   : &cY%s\r\n"
	  "&cGL&c0) Lifespan    : &cY%d minutes\r\n"
	  "&cGT&c0) Top of zone : &cY%d\r\n"
	  "&cGR&c0) Reset Mode  : &cY%s&c0\r\n"
	  "[Command list]\r\n",

	  OLC_NUM(ch), 
	  zone_table[OLC_ZNUM(ch)].number,
	  OLC_ZONE(ch)->name ? OLC_ZONE(ch)->name : "<NONE!>",
	  OLC_ZONE(ch)->lifespan,
	  OLC_ZONE(ch)->top,
	  OLC_ZONE(ch)->reset_mode ?
	   ((OLC_ZONE(ch)->reset_mode == 1) ?
	   "Reset when no players are in zone." :
	   "Normal reset.") :
	   "Never reset");

  /*
   * Print the commands for this room into display buffer.
   */
  while (MYCMD.command != 'S') {
    /*
     * Translate what the command means.
     */
    switch (MYCMD.command) {
    case 'M':
      sprintf(buf2, "%sLoad %s [&cC%d&cY], Max : %d",
	      MYCMD.if_flag ? " then " : "",
	      ss_data(PROTO_MOB(MYCMD.arg1)->player.short_descr),
	      GET_MOB_NUM(PROTO_MOB(MYCMD.arg1)),
//	      mob_index[MYCMD.arg1].virtual, 
	      MYCMD.arg2);
      break;
    case 'G':
      sprintf(buf2, "%sGive it %s [&cC%d&cY], Max : %d",
	      MYCMD.if_flag ? " then " : "",
	      ss_data(PROTO_OBJ(MYCMD.arg1)->short_description),
	      GET_OBJ_NUM(PROTO_OBJ(MYCMD.arg1)),
//	      obj_proto[MYCMD.arg1].short_description,
//	      obj_index[MYCMD.arg1].virtual, 
	      MYCMD.arg2);
      break;
    case 'O':
      sprintf(buf2, "%sLoad %s [&cC%d&cY], Max : %d",
	      MYCMD.if_flag ? " then " : "",
	      ss_data(PROTO_OBJ(MYCMD.arg1)->short_description),
	      GET_OBJ_NUM(PROTO_OBJ(MYCMD.arg1)), MYCMD.arg2
	      );
      break;
    case 'H':
      sprintf(buf2, "%sHide %s [&cC%d&cY], Max : %d",
	      MYCMD.if_flag ? " then " : "",
	      ss_data(PROTO_OBJ(MYCMD.arg1)->short_description),
	      GET_OBJ_NUM(PROTO_OBJ(MYCMD.arg1)), MYCMD.arg2
	      );
      break;
    case 'E':
      sprintf(buf2, "%sEquip with %s [&cC%d&cY], %s, Max : %d",
	      MYCMD.if_flag ? " then " : "",
	      ss_data(PROTO_OBJ(MYCMD.arg1)->short_description),
	      GET_OBJ_NUM(PROTO_OBJ(MYCMD.arg1)), 
	      equipment_types[MYCMD.arg3], MYCMD.arg2
	      );
      break;
    case 'P':
      sprintf(buf2, "%sPut %s [&cC%d&cY] in %s [&cC%d&cY], Max : %d",
	      MYCMD.if_flag ? " then " : "",
	      ss_data(PROTO_OBJ(MYCMD.arg1)->short_description),
	      GET_OBJ_NUM(PROTO_OBJ(MYCMD.arg1)),
	      ss_data(PROTO_OBJ(MYCMD.arg3)->short_description),
	      GET_OBJ_NUM(PROTO_OBJ(MYCMD.arg3)),
	      MYCMD.arg2
	      );
      break;
    case 'R':
      sprintf(buf2, "%sRemove %s [&cC%d&cY] from room.",
	      MYCMD.if_flag ? " then " : "",
	      ss_data(PROTO_OBJ(MYCMD.arg2)->short_description),
	      GET_OBJ_NUM(PROTO_OBJ(MYCMD.arg2))
	      );
      break;
    case 'D':
      sprintf(buf2, "%sSet door %s as %s.",
	      MYCMD.if_flag ? " then " : "",
	      dirs[MYCMD.arg2],
	      MYCMD.arg3 ? ((MYCMD.arg3 == 1) ? "closed" : "locked") : "open"
	      );
      break;
    case '%':
      sprintf(buf2, "%sIf chance of %d in 100...",
	      MYCMD.if_flag ? " then " : "", MYCMD.arg1
	      );
      break;
    case 'T':
      sprintf(buf2, "%sAttach %s trigger [&cC%d&cY].",
	      MYCMD.if_flag ? " then " : "", 
	      trig_attach_types[MYCMD.arg1], 
	      MYCMD.arg2
	      );
      break;
    default:
      strcpy(buf2, "<Unknown Command>");
      break;
    }
    /*
     * Build the display buffer for this command  
     */
    sprintf(buf1, "&c0%d - &cY%s\r\n", counter++, buf2);
    strcat(buf, buf1);
    subcmd++;
  }
  /*
   * Finish off menu  
   */
  sprintf(buf1,
	  "&c0%d - <END OF LIST>\r\n"
	  "&cGN&c0) New command.\r\n"
	  "&cGE&c0) Edit a command.\r\n"
	  "&cGD&c0) Delete a command.\r\n"
	  "&cGQ&c0) Quit\r\nEnter your choice : ",
	  counter);

  strcat(buf, buf1);
  send_to_char(buf, ch);

  OLC_MENU(ch) = ZEDIT_MAIN_MENU;
}

/*-------------------------------------------------------------------*/

/*
 * Print the command type menu and setup response catch. 
 */
void zedit_disp_comtype(char_data *ch)
{
  sprintf(buf,
#if defined(CLEAR_SCREEN)
	"[H[J"
#endif
	"&cGM&c0) Load Mobile to room             &cGO&c0) Load Object to room\r\n"
	"&cGE&c0) Equip mobile with object        &cGG&c0) Give an object to a mobile\r\n"
	"&cGP&c0) Put object in another object    &cGD&c0) Open/Close/Lock a Door\r\n"
	"&cGR&c0) Remove an object from the room  &cGH&c0) Hide Object in room\r\n"
	"&cGT&c0) Attach a trigger                &cG%%&c0) Percentage chance\r\n"
	"What sort of command will this be? : ");
  send_to_char(buf, ch);
  OLC_MENU(ch) = ZEDIT_COMMAND_TYPE;
}

/*-------------------------------------------------------------------*/

/*
 * Print the appropriate message for the command type for arg1 and set
 * up the input catch clause  
 */
void zedit_disp_arg1(char_data *ch)
{
  switch (OLC_CMD(ch).command) {
  case 'M':
    send_to_char("Input mob's vnum : ", ch);
    OLC_MENU(ch) = ZEDIT_ARG1;
    break;
  case 'H':
  case 'O':
  case 'E':
  case 'P':
  case 'G':
    send_to_char("Input object vnum : ", ch);
    OLC_MENU(ch) = ZEDIT_ARG1;
    break;
  case 'D':
  case 'R':
    /*
     * Arg1 for these is the room number, skip to arg2  
     */
    OLC_CMD(ch).arg1 = real_room(OLC_NUM(ch));
    zedit_disp_arg2(ch);
    break;
  case '%':
    send_to_char("Input a chance from 1 to 100 : ", ch);
    OLC_MENU(ch) = ZEDIT_ARG1;
    break;
  case 'T':
    send_to_char("&cG1&c0) Mobiles  &cG2&c0) Objects\r\n"
                 "Input trigger attach type : ", ch);
    OLC_MENU(ch) = ZEDIT_ARG1;
    break;
  default:
    /*
     * We should never get here  .
     */
    cleanup_olc(ch, CLEANUP_ALL);
    mudlog("SYSERR: OLC: zedit_disp_arg1(): Help!", BRF, TRUST_CREATOR, TRUE);
    send_to_char("Oops...\r\n", ch);
    return;
  }
}

/*-------------------------------------------------------------------*/

/*
 * Print the appropriate message for the command type for arg2 and set
 * up the input catch clause.
 */
void zedit_disp_arg2(char_data *ch)
{
  int i = 0;

  switch (OLC_CMD(ch).command) {
  case 'H':
  case 'M':
  case 'O':
  case 'E':
  case 'P':
  case 'G':
    send_to_char("Input the maximum number that can exist on the mud : ", ch);
    break;
  case 'D':
    while (*dirs[i] != '\n') {
      sprintf(buf, "%d) Exit %s.\r\n", i, dirs[i]);
      send_to_char(buf, ch);
      i++;
    }
    send_to_char("Enter exit number for door : ", ch);
    break;
  case 'R':
    send_to_char("Input object's vnum : ", ch);
    break;
  case 'T':
    send_to_char("Input trigger vnum : ", ch);
    break;
  case '%':
  default:
    /*
     * We should never get here, but just in case...
     */
    cleanup_olc(ch, CLEANUP_ALL);
    mudlog("SYSERR: OLC: zedit_disp_arg2(): Help!", BRF, TRUST_CREATOR, TRUE);
    send_to_char("Oops...\r\n", ch);
    return;
  }
  OLC_MENU(ch) = ZEDIT_ARG2;
}

/*-------------------------------------------------------------------*/

/*
 * Print the appropriate message for the command type for arg3 and set
 * up the input catch clause.
 */
void zedit_disp_arg3(char_data *ch)
{
  int i = 0;

  switch (OLC_CMD(ch).command) {
  case 'E':
    for (i = 0; i < NUM_WEARS; ++i) {
      sprintf(buf, "%2d) %26.26s%s", i,
	   equipment_types[i], (i && i % 2) ? "\r\n" : "\t" );
      send_to_char(buf, ch);
    }
#ifdef GOT_RID_OF_IT
    while (*equipment_types[i] != '\n') {
      sprintf(buf, "%2d) %26.26s %2d) %26.26s\r\n", i,
	   equipment_types[i], i + 1, (*equipment_types[i + 1] != '\n') ?
	      equipment_types[i + 1] : "");
      send_to_char(buf, ch);
      if (*equipment_types[i + 1] != '\n')
	i += 2;
      else
	break;
    }
#endif
    send_to_char("Location to equip : ", ch);
    break;
  case 'P':
    send_to_char("Vnum of the container : ", ch);
    break;
  case 'D':
    send_to_char("0)  Door open\r\n"
		 "1)  Door closed\r\n"
		 "2)  Door locked\r\n"
		 "Enter state of the door : ", ch);
    break;
  case 'T':
  case 'H':
  case 'M':
  case 'O':
  case 'R':
  case 'G':
  case '%':
  default:
    /*
     * We should never get here, just in case.
     */
    cleanup_olc(ch, CLEANUP_ALL);
    mudlog("SYSERR: OLC: zedit_disp_arg3(): Help!", BRF, TRUST_CREATOR, TRUE);
    send_to_char("Oops...\r\n", ch);
    return;
  }
  OLC_MENU(ch) = ZEDIT_ARG3;
}

/**************************************************************************
  The GARGANTAUN event handler
 **************************************************************************/

void zedit_parse(char_data *ch, char *arg)
{
  int pos, i = 0;

  switch (OLC_MENU(ch)) {
/*-------------------------------------------------------------------*/
  case ZEDIT_CONFIRM_SAVESTRING:
    switch (*arg) {
    case 'y':
    case 'Y':
      /*
       * Save the zone in memory, hiding invisible people.
       */
      send_to_char("Saving zone info in memory.\r\n", ch);
      zedit_save_internally(ch);
      sprintf(buf, "OLC: %s edits zone info for room %d.",
		GET_NAME(ch), OLC_NUM(ch));
      mudlog(buf, CMP, MAX(TRUST_CREATOR, GET_INVIS_LEV(ch)), TRUE);
      /* FALL THROUGH */
    case 'n':
    case 'N':
      cleanup_olc(ch, CLEANUP_ALL);
      break;
    default:
      send_to_char("Invalid choice!\r\n", ch);
      send_to_char("Do you wish to save the zone info? : ", ch);
      break;
    }
    break;
   /* End of ZEDIT_CONFIRM_SAVESTRING */

/*-------------------------------------------------------------------*/
  case ZEDIT_MAIN_MENU:
    switch (*arg) {
    case 'q':
    case 'Q':
      if (OLC_ZONE(ch)->age || OLC_ZONE(ch)->number) {
	send_to_char("Do you wish to save the changes to the zone info? (y/n) : ", ch);
	OLC_MENU(ch) = ZEDIT_CONFIRM_SAVESTRING;
      } else {
	send_to_char("No changes made.\r\n", ch);
	cleanup_olc(ch, CLEANUP_ALL);
      }
      break;
    case 'n':
    case 'N':
      /*
       * New entry.
       */
      send_to_char("What number in the list should the new command be? : ", ch);
      OLC_MENU(ch) = ZEDIT_NEW_ENTRY;
      break;
    case 'e':
    case 'E':
      /*
       * Change an entry.
       */
      send_to_char("Which command do you wish to change? : ", ch);
      OLC_MENU(ch) = ZEDIT_CHANGE_ENTRY;
      break;
    case 'd':
    case 'D':
      /*
       * Delete an entry.
       */
      send_to_char("Which command do you wish to delete? : ", ch);
      OLC_MENU(ch) = ZEDIT_DELETE_ENTRY;
      break;
    case 'z':
    case 'Z':
      /*
       * Edit zone name.
       */
      send_to_char("Enter new zone name : ", ch);
      OLC_MENU(ch) = ZEDIT_ZONE_NAME;
      break;
    case 't':
    case 'T':
      /*
       * Edit top of zone.
       */
      if (GET_LEVEL(ch) < TRUST_IMPL)
	zedit_disp_menu(ch);
      else {
	send_to_char("Enter new top of zone : ", ch);
	OLC_MENU(ch) = ZEDIT_ZONE_TOP;
      }
      break;
    case 'l':
    case 'L':
      /*
       * Edit zone lifespan.
       */
      send_to_char("Enter new zone lifespan : ", ch);
      OLC_MENU(ch) = ZEDIT_ZONE_LIFE;
      break;
    case 'r':
    case 'R':
      /*
       * Edit zone reset mode.
       */
      send_to_char("\r\n"
		   "0) Never reset\r\n"
		   "1) Reset only when no players in zone\r\n"
		   "2) Normal reset\r\n"
		   "Enter new zone reset type : ", ch);
      OLC_MENU(ch) = ZEDIT_ZONE_RESET;
      break;
    default:
      zedit_disp_menu(ch);
      break;
    }
    break;
    /* End of ZEDIT_MAIN_MENU */

/*-------------------------------------------------------------------*/
  case ZEDIT_NEW_ENTRY:
    /*
     * Get the line number and insert the new line.
     */
    pos = atoi(arg);
    if (isdigit(*arg) && new_command(ch, pos)) {
      if (start_change_command(ch, pos)) {
	zedit_disp_comtype(ch);
	OLC_ZONE(ch)->age = 1;
      }
    } else
      zedit_disp_menu(ch);
    break;

/*-------------------------------------------------------------------*/
  case ZEDIT_DELETE_ENTRY:
    /*
     * Get the line number and delete the line.
     */
    pos = atoi(arg);
    if (isdigit(*arg)) {
      delete_command(ch, pos);
      OLC_ZONE(ch)->age = 1;
    }
    zedit_disp_menu(ch);
    break;

/*-------------------------------------------------------------------*/
  case ZEDIT_CHANGE_ENTRY:
    /*
     * Parse the input for which line to edit, and goto next quiz.
     */
    pos = atoi(arg);
    if (isdigit(*arg) && start_change_command(ch, pos)) {
      zedit_disp_comtype(ch);
      OLC_ZONE(ch)->age = 1;
    } else
      zedit_disp_menu(ch);
    break;

/*-------------------------------------------------------------------*/
  case ZEDIT_COMMAND_TYPE:
    /*
     * Parse the input for which type of command this is, and goto next
     * quiz.
     */
    OLC_CMD(ch).command = toupper(*arg);
    if (!OLC_CMD(ch).command || (strchr("MOPEDGRHT%", OLC_CMD(ch).command) == NULL)) {
      send_to_char("Invalid choice, try again : ", ch);
    } else {
      if (OLC_VAL(ch)) {	/* If there was a previous command. */
	send_to_char("Is this command dependent on the success of the previous one? (y/n)\r\n", ch);
	OLC_MENU(ch) = ZEDIT_IF_FLAG;
      } else {	/* 'if-flag' not appropriate. */
	OLC_CMD(ch).if_flag = 0;
	zedit_disp_arg1(ch);
      }
    }
    break;

/*-------------------------------------------------------------------*/
  case ZEDIT_IF_FLAG:
    /*
     * Parse the input for the if flag, and goto next quiz.
     */
    switch (*arg) {
    case 'y':
    case 'Y':
      OLC_CMD(ch).if_flag = 1;
      break;
    case 'n':
    case 'N':
      OLC_CMD(ch).if_flag = 0;
      break;
    default:
      send_to_char("Try again : ", ch);
      return;
    }
    zedit_disp_arg1(ch);
    break;

/*-------------------------------------------------------------------*/
  case ZEDIT_ARG1:
    /*
     * Parse the input for arg1, and goto next quiz.
     */
    if (!isdigit(*arg)) {
      send_to_char("Must be a numeric value, try again : ", ch);
      return;
    }
    switch (OLC_CMD(ch).command) {
    case 'M':
      if (mob_index[(pos = atoi(arg))]) {
	OLC_CMD(ch).arg1 = pos;
	zedit_disp_arg2(ch);
      } else
	send_to_char("That mobile does not exist, try again : ", ch);
      break;
    case 'H':
    case 'O':
    case 'P':
    case 'E':
    case 'G':
      if (obj_index[(pos = atoi(arg))]) {
	OLC_CMD(ch).arg1 = pos;
	zedit_disp_arg2(ch);
      } else
	send_to_char("That object does not exist, try again : ", ch);
      break;
    case '%':
      pos = atoi(arg);
      if ((pos > 0) && (pos <= 100)) {
	OLC_CMD(ch).arg1 = pos;
	zedit_disp_menu(ch);
      } else
	send_to_char("Input a chance from 1 to 100 : ", ch);
      break;
    case 'T':
      pos = atoi(arg);
      if ((pos >= 1) && (pos <= 2)) {
	OLC_CMD(ch).arg1 = pos - 1;
	zedit_disp_arg2(ch);
      } else
	send_to_char("You must select an attachment type : ", ch);
      break;
    case 'D':
    case 'R':
    default:
      /*
       * We should never get here.
       */
      cleanup_olc(ch, CLEANUP_ALL);
      mudlog("SYSERR: OLC: zedit_parse(): case ARG1: Ack!", BRF, TRUST_CREATOR, TRUE);
      send_to_char("Oops...\r\n", ch);
      break;
    }
    break;

/*-------------------------------------------------------------------*/
  case ZEDIT_ARG2:
    /*
     * Parse the input for arg2, and goto next quiz.
     */
    if (!isdigit(*arg)) {
      send_to_char("Must be a numeric value, try again : ", ch);
      return;
    }
    switch (OLC_CMD(ch).command) {
    case 'H':
    case 'M':
    case 'O':
      OLC_CMD(ch).arg2 = atoi(arg);
      OLC_CMD(ch).arg3 = real_room(OLC_NUM(ch));
      zedit_disp_menu(ch);
      break;
    case 'G':
      OLC_CMD(ch).arg2 = atoi(arg);
      zedit_disp_menu(ch);
      break;
    case 'P':
    case 'E':
      OLC_CMD(ch).arg2 = atoi(arg);
      zedit_disp_arg3(ch);
      break;
    case 'D':
      pos = atoi(arg);
      /*
       * Count directions.
       */
      while (*dirs[i] != '\n')
	i++;
      if ((pos < 0) || (pos > i))
	send_to_char("Try again : ", ch);
      else {
	OLC_CMD(ch).arg2 = pos;
	zedit_disp_arg3(ch);
      }
      break;
    case 'R':
      if (obj_index[(pos = atoi(arg))]) {
	OLC_CMD(ch).arg2 = pos;
	zedit_disp_menu(ch);
      } else
	send_to_char("That object does not exist, try again : ", ch);
      break;
    case 'T':
      if (trig_index[(pos = atoi(arg))]) {
#ifdef GOT_RID_OF_IT
        if (GET_TRIG_ATTACH(PROTO_TRIG(pos)) == OLC_CMD(ch).arg1) {
	  OLC_CMD(ch).arg2 = pos;
	  if (OLC_CMD(ch).arg1 == WLD_TRIGGER)
	    OLC_CMD(ch).arg3 = OLC_NUM(ch);
	  zedit_disp_menu(ch);
	} else {
	  send_to_char("Mismatched trigger type, try again : ", ch);
	}
#else
	OLC_CMD(ch).arg2 = pos;
	if (OLC_CMD(ch).arg1 == WLD_TRIGGER)
	  OLC_CMD(ch).arg3 = OLC_NUM(ch);
#endif
	zedit_disp_menu(ch);
      } else
	send_to_char("That trigger does not exist, try again : ", ch);
      break;
    case '%':
    default:
      /*
       * We should never get here, but just in case...
       */
      cleanup_olc(ch, CLEANUP_ALL);
      mudlog("SYSERR: OLC: zedit_parse(): case ARG2: Ack!", BRF, TRUST_CREATOR, TRUE);
      send_to_char("Oops...\r\n", ch);
      break;
    }
    break;

/*-------------------------------------------------------------------*/
  case ZEDIT_ARG3:
    /*
     * Parse the input for arg3, and go back to main menu.
     */
    if (!isdigit(*arg)) {
      send_to_char("Must be a numeric value, try again : ", ch);
      return;
    }
    switch (OLC_CMD(ch).command) {
    case 'E':
      pos = atoi(arg);
      if ((pos < 0) || (pos >= NUM_WEARS))
	send_to_char("Try again : ", ch);
      else {
	OLC_CMD(ch).arg3 = pos;
	zedit_disp_menu(ch);
      }
      break;
    case 'P':
      if (obj_index[(pos = atoi(arg))]) {
	OLC_CMD(ch).arg3 = pos;
	zedit_disp_menu(ch);
      } else
	send_to_char("That object does not exist, try again : ", ch);
      break;
    case 'D':
      pos = atoi(arg);
      if ((pos < 0) || (pos > 2))
	send_to_char("Try again : ", ch);
      else {
	OLC_CMD(ch).arg3 = pos;
	zedit_disp_menu(ch);
      }
      break;
    case 'T':
    case '%':
    case 'H':
    case 'M':
    case 'O':
    case 'G':
    case 'R':
    default:
      /*
       * We should never get here, but just in case...
       */
      cleanup_olc(ch, CLEANUP_ALL);
      mudlog("SYSERR: OLC: zedit_parse(): case ARG3: Ack!", BRF, TRUST_CREATOR, TRUE);
      send_to_char("Oops...\r\n", ch);
      break;
    }
    break;

/*-------------------------------------------------------------------*/
  case ZEDIT_ZONE_NAME:
    /*
     * Add new name and return to main menu.
     */
    if (OLC_ZONE(ch)->name)
      free(OLC_ZONE(ch)->name);
    else
      log("SYSERR: OLC: ZEDIT_ZONE_NAME: no name to free!");
    OLC_ZONE(ch)->name = str_dup(arg);
    OLC_ZONE(ch)->number = 1;
    zedit_disp_menu(ch);
    break;

/*-------------------------------------------------------------------*/
  case ZEDIT_ZONE_RESET:
    /*
     * Parse and add new reset_mode and return to main menu.
     */
    pos = atoi(arg);
    if (!isdigit(*arg) || (pos < 0) || (pos > 2))
      send_to_char("Try again (0-2) : ", ch);
    else {
      OLC_ZONE(ch)->reset_mode = pos;
      OLC_ZONE(ch)->number = 1;
      zedit_disp_menu(ch);
    }
    break;

/*-------------------------------------------------------------------*/
  case ZEDIT_ZONE_LIFE:
    /*
     * Parse and add new lifespan and return to main menu.
     */
    pos = atoi(arg);
    if (!isdigit(*arg) || (pos < 0) || (pos > 240))
      send_to_char("Try again (0-240) : ", ch);
    else {
      OLC_ZONE(ch)->lifespan = pos;
      OLC_ZONE(ch)->number = 1;
      zedit_disp_menu(ch);
    }
    break;

/*-------------------------------------------------------------------*/
  case ZEDIT_ZONE_TOP:
    /*
     * Parse and add new top room in zone and return to main menu.
     */
    if (OLC_ZNUM(ch) == top_of_zone_table)
      OLC_ZONE(ch)->top = MAX(OLC_ZNUM(ch) * 100, MIN(32000, atoi(arg)));
    else
      OLC_ZONE(ch)->top = MAX(OLC_ZNUM(ch) * 100, MIN(zone_table[OLC_ZNUM(ch) + 1].number * 100, atoi(arg)));
    zedit_disp_menu(ch);
    break;

/*-------------------------------------------------------------------*/
  default:
    /*
     * We should never get here, but just in case...
     */
    cleanup_olc(ch, CLEANUP_ALL);
    mudlog("SYSERR: OLC: zedit_parse(): Reached default case!", BRF, TRUST_CREATOR, TRUE);
    send_to_char("Oops...\r\n", ch);
    break;
  }
}

/*
 * End of parse_zedit()  
 */
